/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/************************************************************
 * CfgItem.java
 ************************************************************/

package utilities.cfg;

/* Global imports */
import java.beans.*;
import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

public class CfgManager
{
    CfgItem mainblock;
    String filename;
    Hashtable configured_objects = new Hashtable();
    Hashtable configured_objects_reverse = new Hashtable(); //reverse hash
    static Hashtable manager_mappings = new Hashtable();
    boolean unspace_keys = false;
    
    public static void setManager(ConfiguredObject o, CfgManager m)
    {
        if (o == null || m == null)
            return;
        if (manager_mappings.containsKey(o))
            manager_mappings.remove(o);
        manager_mappings.put(o, m);
    }
    
    public static void clearManager(ConfiguredObject o)
    {
        if (manager_mappings.containsKey(o))
            manager_mappings.remove(o);
    }
    
    public static CfgManager getManager(ConfiguredObject o)
    {
        if (manager_mappings.containsKey(o))
            return (CfgManager) manager_mappings.get(o);
        else
            return null;
    }
    
    public CfgManager(CfgItem main, String fname)
    {
        if (main == null || !(main.getValue() instanceof CfgSection))
            mainblock = new CfgItem("main", new CfgSection());
        else
            mainblock = main;
        if (fname == null)
            filename = "main.cfg";
        else
            filename = fname;
    }
    
    public CfgManager()
    {
        this(null, null);
    }
    
    public CfgManager(String fname)
    {
        FileInputStream cfgstream = null;
        CfgFileParser parser;
        if (fname == null)
            filename = "main.cfg";
        else
            filename = fname;
        try {
            cfgstream = new FileInputStream(filename);
        } catch (FileNotFoundException fnf) {
            try {
                FileOutputStream newout = new FileOutputStream(filename);
                OutputStreamWriter writer = new OutputStreamWriter(newout);
                writer.write("main {\n}\n");
                writer.close();
                newout.close();
                cfgstream = new FileInputStream(filename);
            } catch (Exception e) {
                e.printStackTrace();
                return;
            }
        }
        if (cfgstream == null)
            return;
        parser = new CfgFileParser(cfgstream);
        try {
            mainblock = parser.Input();
            cfgstream.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (mainblock == null)
            mainblock = new CfgItem("main", new CfgSection());
    }
    
    public String getFileName()
    {
        return filename;
    }
    
    public void setFileName(String s)
    {
        filename = s;
    }
    
    public Object get(String key)
    {
        if (unspace_keys)
            key = unspace(key);
        StringTokenizer st = new StringTokenizer(key, "/.,");
        synchronized (mainblock)
        {
            Object cur_level = mainblock;
            while (st.hasMoreTokens())
            {
                if (cur_level == null)
                    return null;
                String s = st.nextToken();
                if (s.length() == 0)
                    continue;
                if (cur_level instanceof CfgItem)
                {
                    Object cur_val = ((CfgItem)cur_level).getValue();
                    if (cur_val instanceof CfgSection)
                        cur_level = ((CfgSection)cur_val).getItem(s);
                    else
                        return null;
                } else if (cur_level instanceof CfgSection) {
                    cur_level = ((CfgSection)cur_level).getItem(s);
                } else {
                    return null;
                }
            }
            return cur_level;
        }
    }

    public void remove(String key)
    {
        if (unspace_keys)
            key = unspace(key);
        StringTokenizer st = new StringTokenizer(key, "/.,");
        CfgSection parent = null;
        String s = null;
        synchronized (mainblock)
        {
            Object cur_level = mainblock;
            while (st.hasMoreTokens())
            {
                if (cur_level == null)
                    return;
                s = st.nextToken();
                if (s.length() == 0)
                    continue;
                if (cur_level instanceof CfgItem)
                {
                    Object cur_val = ((CfgItem)cur_level).getValue();
                    if (cur_val instanceof CfgSection)
                    {
                        parent = (CfgSection) cur_val;
                        cur_level = ((CfgSection)cur_val).getItem(s);
                    } else {
                        return;
                    }
                } else if (cur_level instanceof CfgSection) {
                    parent = (CfgSection) cur_level;
                    cur_level = ((CfgSection)cur_level).getItem(s);
                } else {
                    return;
                }
            }
            if (parent != null)
            {
                Object o = parent.getItem(s);
                parent.removeItem(s);
            }
        }
    }

    public void put(String key, boolean b)
    {
        put(key, new Boolean(b));
    }

    public void put(String key, int i)
    {
        put(key, new Integer(i));
    }

    public void put(String key, Object o)
    {
        if (o instanceof Boolean)
            o = o.toString();
        if (unspace_keys)
            key = unspace(key);
        StringTokenizer st = new StringTokenizer(key, "/.,");
        CfgSection cur_val;
        String s = null;
        synchronized (mainblock)
        {
            CfgItem cur_level = mainblock;
            while (st.hasMoreTokens())
            {
                s = st.nextToken();
                if (s.length() == 0)
                    continue;
                if (st.hasMoreTokens())
                {
                    if (!(cur_level.getValue() instanceof CfgSection))
                    {
                        cur_level.setValue(new CfgSection());
                    }    
                    cur_val = (CfgSection) cur_level.getValue();
                    if (!cur_val.contains(s))
                    {
                        cur_val.addItem(new CfgItem(s, new CfgSection()));
                    }
                    cur_level = cur_val.getItem(s);
                } else {
                    cur_val = (CfgSection) cur_level.getValue();
                    if (cur_val == null)
                        return;
                    CfgItem newitem = new CfgItem(s, o);
                    cur_val.addItem(newitem);
                }
            }
        }
    }

    public void updateFromConfiguredObjects()
    {
        Enumeration e = configured_objects.keys();
        while (e.hasMoreElements())
        {
            ConfiguredObject o = (ConfiguredObject) e.nextElement();
            o.saveCfg();
        }
    }

    public void updateToConfiguredObjects()
    {
        Enumeration e = configured_objects.keys();
        while (e.hasMoreElements())
        {
            ConfiguredObject o = (ConfiguredObject) e.nextElement();
            o.updateCfg();
        }
    }

    public void addConfiguredObject(ConfiguredObject o, String newkey)
    {
        if (o == null || newkey == null)
            return;
        removeConfiguredObject(o);
        if (unspace_keys)
            newkey = unspace(newkey);
        removeConfiguredPath(newkey);
        setManager(o, this);
        configured_objects.put(o, newkey);
        o.updateCfg();
        o.saveCfg();
    }

    public void wipeConfiguredObject(ConfiguredObject o)
    {
        if (configured_objects.containsKey(o))
        {
            remove((String) configured_objects.get(o));
            clearManager(o);
            configured_objects_reverse.remove(configured_objects.get(o));
            configured_objects.remove(o);
        }
    }

    public void removeConfiguredObject(ConfiguredObject o)
    {
        if (configured_objects.containsKey(o))
        {
            clearManager(o);
            configured_objects_reverse.remove(configured_objects.get(o));
            configured_objects.remove(o);
        }
    }

    public void removeConfiguredPath(String key)
    {
        if (configured_objects_reverse.containsKey(key))
        {
            remove(key);
            configured_objects.remove(configured_objects_reverse.get(key));
            configured_objects_reverse.remove(key);
        }
    }

    public ConfiguredObject getKeyObject(String path)
    {
        if (configured_objects_reverse.containsKey(path))
            return (ConfiguredObject) configured_objects_reverse.get(path);
        else
            return null;
    }

    public String getObjectKey(ConfiguredObject o)
    {
        if (configured_objects.containsKey(o))
            return (String) configured_objects.get(o);
        else
            return null;
    }
    
    public void setObjectKey(ConfiguredObject o, String key)
    {
        addConfiguredObject(o, key);
    }
    
    public Object getValue(String key)
    {
        Object o = get(key);
        if (o == null || o instanceof CfgSection)
            return null;
        else if (o instanceof CfgItem)
            return ((CfgItem) o).getValue();
        else
            return o;
    }
   
    public Boolean getBooleanValue(String key)
    {
        Object o = getValue(key);
        if (o == null)
            return null;
        if (o instanceof String)
        {
            if (((String)o).equals("true") || ((String)o).equals("yes"))
                return new Boolean(true);
            else if (((String)o).equals("false") || ((String)o).equals("no"))
                return new Boolean(false);
            else
                return null;
        } else if (o instanceof Integer) {
            if (((Integer)o).equals(new Integer(0)))
                return new Boolean(false);
            else
                return new Boolean(true);
        } else {
            return null;
        }
    }
    
    public String getStringValue(String key)
    {
        Object o = getValue(key);
        if (o instanceof String)
            return (String) o;
        else
            return null;
    }
    
    public Integer getIntegerValue(String key)
    {
        Object o = getValue(key);
        if (o instanceof Integer)
            return (Integer) o;
        else
            return null;
    }
    
    public Point getPointValue(String key)
    {
        Object o = getValue(key);
        if (o instanceof Point)
            return (Point) o;
        else
            return null;
    }
   
    public void save()
    {
        updateFromConfiguredObjects();
        FileWriter cfgwriter;
        try {
            System.err.println("trying to open " + filename);
            cfgwriter = new FileWriter(filename);
            System.err.println("trying to write file");
            synchronized (mainblock)
            {
                cfgwriter.write(mainblock.toString());
            }
            cfgwriter.close();
        } catch (Exception e) {
            e.printStackTrace(); 
            return;
        }
    }
    
    protected String unspace(String s)
    {
        String result = new String("");
        StringTokenizer st = new StringTokenizer(s);
        while (st.hasMoreTokens())
        {
            result = result + st.nextToken();
        }
        return result;
    }
}

