/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

package utilities;

import java.util.*;

public class UTimer
{
    protected long started = 0;
    protected long stopped = 0;
    protected boolean running = false;
    protected boolean completed = false;
    protected Boolean access = new Boolean(true);
    protected String name = "";
    protected int window = -1;
    protected int windowcounter = 0;
    protected boolean windowed = false;
    protected double [] data = null; 
    
    protected static Hashtable timers = new Hashtable();
   
   
    /**
     * returns a UTimer with the specified name.  If the timer doesn't exist,
     * it'll create one.
     * @param s the name of the timer to get.
     */ 
    public static UTimer getTimer(String s)
    {
        if (timers.contains(s))
            return (UTimer) timers.get(s);
        else {
            UTimer t = new UTimer(s);
            timers.put(s, t);
            return t;
        }
    }
    
    /**
     * removes the specified timer from the global timer list.
     * @param s the name of the timer to dispense with.
     */
    public static void clearTimer(String s)
    {
        timers.remove(s);
    }
    
    /**
     * will time the specified timer's action, for count times, with the given
     * windowed setting.  This is the simplest way I could come up with to do
     * this, without having functions at first class objects.  Here is some
     * sample code that I use to time stuff in BetterArea:
     * <code>
     *   UTimer.time(100, new UTimer("findArea with partials") {
     *               public void action()
     *               {
     *                   b.setDepth(temp);
     *                   temparea = b.findArea();
     *                   b.flushCache();
     *                   b.printcachestats = false;
     *               }
     *           }, true);
     * </code>
     * This code creates an anonymous UTimer class with an action function that
     * calls the BetterArea's find area function.  Some workarounds are
     * necessary to use variables from the local scope outside of the class
     * definition; the need to be final and there are some other wierdnesses.
     * For more detail see BetterArea.main()
     * @param count the number of times to time the action.
     * @param t the UTimer to use - it's recommended that you fill in that
     * timer's <b>action</b> function, or not much will happen.
     * @param whether to use windowing or not.
     * @see BetterArea#main
     * @see #setWindowed
     * @see #action
     */
    public static long time(int count, UTimer t, boolean windowed)
    {
        if (windowed)
        {
            t.setWindowed(true, count);
        } else {
            t.setWindowed(false, 0);
        }
        return time(count, t);
    }
   
    /**
     * times something using whatever windowing setting the UTimer already has.
     * @param count the number of times to run the action.
     * @param t the timer to use.
     * @see #time(int,UTimer,boolean)
     */
    public static long time(int count, UTimer t)
    {
        while (count > 0)
        {
            t.startTimer();
            t.action();
            t.stopTimer();
            count--;
        }
        System.out.println(t);
        return t.getTime();
    }
    
    /**
     * times the specified timer's action once.
     * @param t the UTimer to use.
     * @see #time(int,Timer,boolean)
     */
    public static long time(UTimer t)
    {
        return time(1, t);
    }
    
    /**
     * constructor for a named timer.  This will not create a windowed timer.
     * @param n the name of the timer.
     */
    public UTimer(String n)
    {
        name = n;
    }
    
    /**
     * constructor for a windowed timer.  This creates a UTimer with a specified
     * name, and a specified window.
     * @param n the name of the timer.
     * @param window the size of the window.
     * @see #setWindowed
     */
    public UTimer(String n, int window)
    {
        this(n);
        setWindowed(true, window);
    }
   
    /**
     * returns whether the timer is windowed.
     */
    public boolean isWindowed() { return windowed; }
    
    /**
     * Sets whether the timer is windowed.  This idea is stolen from Bryan's
     * Observer class - the window is the size of the array in which to store
     * data.  Up to w datapoints will be stored, and then the array counter
     * will loop around and start filling the beginning of the array again.  If
     * the timer is not windowed, only one time will be stored.  Successive
     * calls to startTimer and stopTimer without resetting the timer will just
     * add on to the time.  If it is windowed, they will increment the array
     * counter.
     * @param b whether the UTimer should use windowing.
     * @param w the size of the window to use.
     */
    public void setWindowed(boolean b, int w) 
    { 
        windowed = b;
        if (windowed)
        {
            window = w;
            data = new double[w];
            resetWindow();
        } else {
            data = null;
        }
    }
    
    /**
     * returns the size of the window used.
     */
    public int getWindow() { return window; }
    
    /**
     * clears the window of all data.
     */
    public void resetWindow()
    {
        for (int i = 0; i < data.length; i++)
            data[i] = Double.MIN_VALUE;
        windowcounter = 0;
    }
   
    /**
     * Starts the timer.
     */ 
    public void startTimer()
    {
        synchronized (access)
        {
            if (running)
            {
                System.err.println("cannot start timer " + this);
                return;
            }
            if (windowed)
            {
                started = System.currentTimeMillis();
                running = true;
                completed = false;
            } else if (!windowed) {
                started = System.currentTimeMillis() - getTime();
                running = true;
            }
        }
    }
    
    /**
     * Stops the timer.
     */
    public long stopTimer()
    {
        synchronized (access)
        {
            if (running)
            {
                stopped = System.currentTimeMillis();
                if (windowed)
                {
                    data[windowcounter++] = stopped - started;
                    if (windowcounter > window)
                        windowcounter = 0;
                }

                running = false;
                completed = true;
                return getTime();
            } else {
                System.err.println("cannot stop timer " + this);
            }
            return -1;
        }
    }
    
    /**
     * Resets the timer.  If windowing is enabled, all data is cleared.
     */
    public void resetTimer()
    {
        synchronized (access)
        {
            if (windowed)
                resetWindow();
            windowcounter = 0;
            running = false;
            completed = false;
            stopped = 0;
            started = 0;
        }
    }
       
    /**
     * returns the result of timing.  If windowing is enabled, this returns the
     * results of the most recent timing, and if not, returns the total of all
     * timing since the last reset.  If no timing has occured, returns 0.
     */
    public long getTime()
    {
        synchronized (access)
        {
            if (!running && completed)
            {
                return stopped - started;
            }
            return 0;
        }
    }
    
    /**
     * returns the average of all timing.  If windowing is enabled, this will
     * compute an average, and if not, it will just return the results of
     * getTime().
     */
    public double getAverage()
    {
        synchronized (access)
        {
            if (!windowed)
                return (double) getTime();
            double avg = 0.0;
            int count = 0;
            for (int i = 0; i < data.length; i++)
            {
                if (data[i] != Double.MIN_VALUE)
                    avg = (avg * count + data[i]) / ++count;
            }
            return avg;
        }
    }
    
    /**
     * returns the number of times collected.
     */
    public int getCount()
    {
        synchronized (access)
        {
            if (!windowed)
                return 1;
            int count = 0;
            for (int i = 0; i < data.length; i++)
            {
                if (data[i] != Double.MIN_VALUE)
                    count++;
            }
            return count;
        }
    }
       
    /**
     * returns a human readable display of the timer.  You should be able to
     * just print this and it will show you what you want.
     */ 
    public String toString()
    {
        String s =  new String("[ " + name + ": ");
        if (running)
        {
            long l = System.currentTimeMillis();
            s = new String(s + "been running for [" + started + "ms] ");
        } else if (completed) {
            if (isWindowed())
            {
                s = new String(s + "average time [" + getAverage() + 
                        "ms] count [" + getCount() + "] ");
            } else {
                s = new String(s + "time [" + getTime() + "ms] ");
            }
        } else {
            s = new String(s + "not running");
        }
        s = new String(s + "]");
        return s;
    }
   
    /**
     * the static time function will time whatever this function does.  You
     * should override it if you want to use the timer this way.  The suggested
     * way of doing this is using an anonymous class.
     * @see #time
     */ 
    public void action()
    {
        
    }
}

