/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/**
***************************************************************
* $Source: /nfs/shelob/cvs/masl/Mass/modules/utilities/utilities/TaemsRandom.java,v $
* $Revision: 1.1 $
******************************************************************
*
*                Random Generator for respecting TAEMS proba
*
*     @author : Regis Vincent [Regis.Vincent@sophia.inria.fr]
*              
*
*     Creation date: 04 Sep 97 21:22
*  Last file update: $Date: 2005/01/20 18:20:12 $
****************************************************************
*/
package utilities;

import java.util.*;
import java.math.BigDecimal;

/* Local Import */
import utilities.Log;
import utilities.Distribution;
 

/**
 * This class defines a new TaemsRandom generator based on the 
 * Random for java.util.Random
 * The only difference is, this classe, you can get the seed !
 **/
public class TaemsRandom {
  private static TaemsRandom defaultRandom;
  private float []distribution;
  private long seed;
  private Random Rg;
  private int base;
  private int sum;
  private static Log logger;
  /**
   * @param long seed is the seed for the random number
   **/
  public TaemsRandom(long seed) {
    logger = Log.getDefault();
    this.seed = seed;
    Rg = new Random(this.seed);
    logger.log("(Random Generator) Random seed sets to " + this.getSeed() + " (given)");
  }
  
  public TaemsRandom() {
    logger = Log.getDefault();
    this.seed = System.currentTimeMillis();
    Rg = new Random(this.seed);
    logger.log("(Random Generator) Random seed sets to " + this.getSeed() + " (time depending)");
  }


  /**
   * setDistribution set the distribution for the TaemsRandom
   * number (using the Distribution Structure)
   **/
  public void setDistribution(Distribution d)
  {
      this.setDistribution(d.getArray());
  }


  /**
   * setDistribution set the distribution for the TaemsRandom
   * number the form is a pair of [ percentage value ]
   **/
  public synchronized void setDistribution(float args[])
  {
    while (this.distribution != null) {
      try { wait() ;} catch (InterruptedException ex) {}
    }
    int i,j, pointeur;
    float tmp[] = new float[(args.length/2)];
    this.base = 1 ;
    // Maybe it will be another value (100 or 1000 or 10000).
    // Checking for the good base.
    for ( i = 0 ; i < args.length ; i ++) {
      while (!this.isAnInteger(args[i]*this.base))
	{
	  this.base = this.base*10;
	  logger.log("(Random Generator) New base : " + this.base,3);
	}
      tmp[i/2]=args[i];
      i++;
    }
    
    this.sum = 0;
    // Computing the total sum ....
    for ( i = 0 ; i < tmp.length ; i ++)
      this.sum = (int)Math.rint((double)tmp[i]*this.base) + this.sum;
    
    logger.log("(Random Generator) base = " + this.base + " and total sum on " + this.sum, 3);
    
    this.distribution = new float[this.sum];
    if(args.length  != 0) {
      pointeur = 0;
      for ( i = 0 ; i < args.length ; i ++) {
	int max = pointeur + (int)Math.rint((double)this.base*args[i]);
	logger.log("(Random Generator) from " + (pointeur+1) + " to " + max + " Value = " + args[i+1],3);
	for (j = pointeur; j < max; j++, pointeur++)
	  this.distribution[j] = args[i+1];
	i++;
      }
    }
  } 
       

  /**
   * unsetDistribution to avoid concurrent access to the same Random structure 
   **/
  public synchronized void unsetDistribution()
  {
    this.distribution = null;
    this.base = 0;
    this.sum = 0;
    notifyAll();
  }
  
  /**
   * nextInt returns a number between 0 to the sum
   * @return int 
   **/
  public int nextInt()
  {
    if (this.base != 0) {
      Float f = new Float (this.sum * this.Rg.nextFloat()) ;
      logger.log("(Random Generator) Index choosen : " + f.intValue(), 3);
      return(f.intValue());
    }
    else
      logger.log("(Random Generator) Error try to access to random distribution without distribution set !");
      
    return(0);
  }


  /**
   * nextFloat returns a uniformly distributed number between 0 to 1.0
   * @return float
   **/
  public float nextFloat() {
    if (this.Rg != null)
      return(this.Rg.nextFloat());
    else
      logger.log("(Random Generator) Error try to access to random number without initialization !");
    return(0);
  }
   
  /**
   * nextGaussian returns a number between 0 to 1.0 from a gaussian distribution
   * @return double
   **/
  public double nextGaussian() {
    if (this.Rg != null)
      return(this.Rg.nextGaussian());
    else
      logger.log("(Random Generator) Error try to access to random number without initialization !");
    return(0.0);
  }
 
  /**
   * nextValue returns a value corresponding at the distribution
   **/
  public float nextValue() {
    if (this.base != 0) 
      {
	float f = this.distribution[this.nextInt()];
	logger.log("(Random Generator) Value at this index : " + f,3);
	return(f);
      }
    else
     logger.log("(Random Generator) Error try to access to random distribution without distribution set !");
    return(0); 
  }

  /**
   * isAnInteger(float value) 
   **/
  public boolean isAnInteger(float value) {
    if ((Math.ceil((double)value) - value) <= 0.0001)
      return(true);
    else return(false);
  }	
  
  public long getSeed() {
    return (this.seed);
  }

    /**
     * returns a Long Value used for Action/Commitment or any
     * kind of uniq ID (completly indepedant of Taems).
     */
    public long getUniqID() {
	return(this.Rg.nextLong());
    }

  /**
   * Sets the default Taems Random Number Generator
   */
  public static synchronized void setDefault(TaemsRandom r) {
    defaultRandom = r;
    logger.log("(Random Generator) Random Generator available");
  }
  
   /**
   * Gets the default log object
   * @return The default Log object
   */
  public static TaemsRandom getDefault() {
    return defaultRandom;
  }


}
