/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/************************************************************
 * NodePanel.java
 ************************************************************/

package utilities;

/* Global imports */
import javax.swing.*;
import java.beans.*;
import java.awt.*;
import java.io.*;
import java.util.*;
import java.net.*;
import utilities.cfg.*;


public class NodePanel extends JPanel implements ConfiguredObject
{
    protected String label = "";
    protected boolean selected = false;
    protected boolean nodefilled = true;
    protected boolean moveable = true;
    protected Shape shape = null;
    
    public static int H_MARGIN = 5;
    public static int V_MARGIN = 2;
    
    public static Font normalFont = new Font("SansSerif", Font.PLAIN, 12);
    public static Font smallFont = new Font("SansSerif", Font.PLAIN, 8);
    
    GraphNode parent;
    
    public NodePanel(GraphNode n, String l)
    {
        super();
        setNode(n);
        setLabel(l);
        setLayout(null);
    }
   
    protected NodePanel()
    {
        this(null, null);
    }
    
    public GraphNode getNode()
    {
        return parent;
    }
    
    protected void setNode(GraphNode n)
    {
        parent = n;
    }
    
    public boolean isSelected()
    {
        return selected;
    }
    
    public void setSelected(boolean b)
    {
        selected = b;
    }
    
    public boolean isNodeFilled()
    {
        return nodefilled;
    }
    
    public void setNodeFilled(boolean b)
    {
        nodefilled = b;
    }
    
    public boolean isMoveable()
    {
        return moveable;
    }
    
    public void setMoveable(boolean b)
    {
        moveable = b;
    }
    
    public String getLabel()
    {
        return label;
    }
    
    public void setLabel(String s)
    {
        label = s;
    }

    public void paintComponent(Graphics g)
    {
        super.paintComponent(g);
        paintBackImage(g);
        g.setFont(normalFont);
        FontMetrics fm = g.getFontMetrics();
        Rectangle rect = getBounds();
        rect = new Rectangle(0, 0, rect.width - 1, rect.height - 1);

        if (!isVisible())
            return;

        if (isSelected())
            g.setColor(Color.yellow);
        else
            g.setColor(getBackground());
        if (isNodeFilled())
            g.fillRoundRect(0, 0, rect.width, rect.height, getRectRoundness(), getRectRoundness());
        g.setColor(Color.black);
        g.drawRoundRect(0, 0, rect.width, rect.height, getRectRoundness(), getRectRoundness());

        // draw label
        try {
            g.setColor(getForeground());
            // lower lefthand corner
            g.drawString(getLabel(), rect.x + H_MARGIN, rect.y + rect.height - 
                    (V_MARGIN + fm.getDescent()));
        } catch (java.lang.NullPointerException e) {
            System.err.println("Error drawing label");
            e.printStackTrace();
        }
        //paintLines(g);
    }

    public Dimension getMinimumSize()
    {
        return new Dimension(25, 25);
    }

    public Dimension getPreferredSize()
    {
        if (bg_image != null)
            return bg_image.getPreferredSize();
        else if (getShape() != null)
            return getShape().getBounds().getSize();
        else
            return calculateSize(getFontMetrics(getFont()));
    }
    
    ImageComponent bg_image = null;
    URL bg_image_url = null;

    public void setBackImage(ImageComponent i)
    {
        bg_image = i;
        bg_image_url = null;
        if (bg_image == null)
        {
            setNodeFilled(true);
        } else {
            bg_image.setLocation(getLocation());
            setPreferredSize(bg_image.getPreferredSize());
            setNodeFilled(false);
        }
    }

    public void setBackImage(URL u)
    {
        setBackImage(new ImageComponent(u));
        bg_image_url = u;
    }
    
    public URL getBackImageURL()
    {
        return bg_image_url;
    }
    
    public ImageComponent getBackImage()
    {
        if (GraphLayers.getGlobal().isVisible("geographic"))
            return bg_image;
        else
            return null;
    }

    
    public void paintBackImage(Graphics g)
    {
        if (bg_image == null)
            return;
        bg_image.setLocation(getLocation());
        bg_image.paint(g);
    }
    
    public Shape getShape()
    {
        return shape;
    }
     
    public void setShape(Shape s)
    {
        if (s instanceof Polygon)
        {
            throw new IllegalArgumentException();
        } else if (s instanceof Rectangle) {
            shape = ((Rectangle)s).getBounds();
            setLocation(((Rectangle)shape).getLocation());
            setSize(new Dimension(shape.getBounds().x, shape.getBounds().y));
            setMoveable(false);
        }
    }
    
    public int calculateHeight(FontMetrics fm)
    {
        if (getLabel() == null)
        {
            return 0;
        } else {
            return fm.getHeight() + V_MARGIN * 2;
        }
    }

    public int calculateWidth(FontMetrics fm)
    {
        if (getLabel() == null)
        {
            return 0;
        } else {
            return fm.stringWidth(getLabel()) + H_MARGIN * 2;
        }
    }

    public Dimension calculateSize(FontMetrics fm)
    {
        return new Dimension(calculateWidth(fm), calculateHeight(fm));
    }


    /**
     * Get the background colour.
     */
    public Color getBackground()
    {
        return Color.white;
    }

    public Color getForeground()
    {
        return Color.black;
    }

    public int getRectRoundness()
    {
        return 0;
    }
    
    public Point getLocationForLine()
    {
        Point p = getLocation();
        Dimension d = getSize();
        return new Point(p.x + d.width / 2, p.y + d.height / 2);
    }
    
    public Point getLocationForLine(GraphNode src)
    {
        return getLocationForLine();
    }
    
    public synchronized void updateCfg()
    {
        String s;
        Boolean b;
        Integer i;
        Point p;
        CfgManager manager = CfgManager.getManager(this);
        if (manager == null)
        {
            System.err.println("NodePanel " + getLabel() + 
                    "has no cfg manager...");
            return;
        }
        String key = manager.getObjectKey(this);
        if (key == null || key.length() == 0)
            return;
        s = manager.getStringValue(key + "/name");
        if (s != null)
            setLabel(s);
        b = manager.getBooleanValue(key + "/nodefilled");
        if (b != null)
            setNodeFilled(b.booleanValue());
        b = manager.getBooleanValue(key + "/moveable");
        if (b !=  null)
            setMoveable(b.booleanValue());
        
        int width = getWidth();
        int height = getHeight();
        i = manager.getIntegerValue(key + "/height");
        if (i != null)
            height = i.intValue();
        i = manager.getIntegerValue(key + "/width");
        if (i != null)
            width = i.intValue();
        setSize(width, height);
        p = manager.getPointValue(key + "/upleft");
        if (p != null)
            setLocation(p);
        s = manager.getStringValue(key + "/background");
        if (s != null)
            try 
            {
                setBackImage(new URL(s));
            } catch (MalformedURLException e) { }
    }
        
    public synchronized void saveCfg()
    {
        CfgManager manager = CfgManager.getManager(this);
        if (manager == null)
        {
            System.err.println("NodePanel " + getLabel() + 
                    "has no cfg manager...");
            return;
        }
        String key = manager.getObjectKey(this);
        if (key == null || key.length() == 0)
            return;
        manager.put(key + "/name", getLabel());
        manager.put(key + "/nodefilled", isNodeFilled());
        manager.put(key + "/moveable", isMoveable());
        manager.put(key + "/upleft", getLocation());
        Dimension d = getSize();
        manager.put(key + "/width", getWidth());
        manager.put(key + "/height", getHeight());
        if (getBackImageURL() != null)
            manager.put(key + "/background", getBackImageURL().toString());
    }
}
