/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 * Message.java
 ***************************************************************************************/
package utilities;


import java.io.*;
import java.net.*;
import java.util.StringTokenizer;

/**
 * Base inter-agent (or whatever..) message class.
 */
public class Message implements Cloneable {
  /**
   * Message Delimiter.  Reference this variable if at all
   * possible if you need to deal with low level message
   * parsing, but if you are in a non-java environment, the
   * actual delimiter is currently "\r\nEOM\r\n" - although
   * it is not guarunteed to remain that string.
   */
  protected static final String MSG_DELIM = "\r\nEOM\r\n";
  protected static final int MSG_WINDOW = 100;

  private Object content;
  private String sourceaddr = "", destaddr = "";
  private int sourceport = -1, destport = -1;
  private int sendtime = -1, receivetime = -1;
  private String data = "";
  private Connection connection;

  /**
   * Constructor, dosen't do anything.
   */
  public Message() { }

  /**
   * Constructor
   * @param c The message to send
   */
  public Message(Object c) {
    this (c, null, -1, null, -1);
  }

  /**
   * Constructor
   * @param c The message to send
   * @param da The destination address
   * @param dp The destination port
   */
  public Message(Object c, String da, int dp) {
    this (c, null, -1, da, dp);
  }

  /**
   * Constructor
   * @param c The message to send
   * @param sa The source address
   * @param sp The source port
   * @param da The destination address
   * @param dp The destination port
   */
  public Message(Object c, String sa, int sp, String da, int dp) {
    content = c;
    sourceaddr = sa;
    sourceport = sp;
    destaddr = da;
    destport = dp;
    data = c.toString();
  }

  /**
   * Accessor functions
   */
  public Object getContent() { return content; }
  public void setContent(Object c) { content = c; }

  public String getData() { return data; }
  public void setData(String d) { data = d; }

  public Connection getConnection() { return connection; }
  public void setConnection(Connection c) { connection = c; }

  public String getSourceAddr() { return sourceaddr; }
  public void setSourceAddr(String sa) { sourceaddr = sa; }
  public int getSourcePort() { return sourceport; }
  public void setSourcePort(int sp) { sourceport = sp; }

  public String getDestAddr() { return destaddr; }
  public void setDestAddr(String da) { destaddr = da; }
  public int getDestPort() { return destport; }
  public void setDestPort(int dp) { destport = dp; }

  public int getSendTime() { return sendtime; }
  public void setSendTime(int t) { sendtime = t; }
  public int getReceiveTime() { return receivetime; }
  public void setReceiveTime(int t) { receivetime = t; }

  /**
   * Returns the first word in the content (used to determine the
   * type of content).
   * @return The word, or an empty string if none found
   */
  public String contentWord() {
      return contentWord(content.toString());
  }

  /**
   * Returns the first word in the string.
   * Uses whitespace as the delimeter.
   * @param str The string to examine
   * @return The word, or an empty string if none found
   */
  public String contentWord(String str) {
    StringTokenizer st = new StringTokenizer(str);

    if (st.hasMoreElements())
      return st.nextToken();
    else
      return new String("");
  }

  /**
   * Returns the string after the first word in the content (used
   * to get the supposed data out of the content).
   * @return The data, or an empty string if none found
   */
  public String contentData() {
    return contentData(content.toString());
  }

  /**
   * Returns the string after the first word in the string.
   * Uses whitespace as the delimeter.
   * @param str The string to examine
   * @return The data, or an empty string if none found
   */
  public String contentData(String str) {
    StringTokenizer st = new StringTokenizer(str);
    String out;

    if (st.countTokens() > 1) {
      return str.substring(st.nextToken().length()+1, str.length());
    } else
      return new String("");
  }

  /**
   * Same as above, but uses MSG_DELIM as the delimiter.  This method retained
   * for compatibility.
   * Sends the message across a stream
   * @param b The writer to use to send the message
   */
  public boolean send(BufferedWriter b) {
    return(send(b, MSG_DELIM));
  }

  /**
   * Sends the message across a stream
   * @param b The writer to use to send the message
   * @param d The message delimiter
   */
  public boolean send(BufferedWriter b, String d) {
    try {      
      b.write(getData());
      if (getData().length() <= 0)
          System.err.println("Warning: About to send empty message");
      if (!getData().endsWith(d)) b.write(d);
      b.flush();
      
    } catch (IOException e) {
      System.err.println("Error sending message: " + e);
      return false;
    }
    
    return true;
  }

  /**
   * Recieves a message (string) from a stream.  Typically this
   * should be called to fill in the data parameter of the 
   * Message constructor.  This is a blocking receive, use
   * the ready() function of BufferedReader to determine beforehand
   * if a message is available.
   * @param b The reader used to get the message
   * @param d The message delimiter
   * @return The message, or null if error occurs
   */
    public static String receive(BufferedReader b, String d) {
        char bufr[] = new char[MSG_WINDOW];
        StringBuffer strb = new StringBuffer();
        String str;
        int goodlen, len, total = 0;

        try {
            goodlen = MSG_WINDOW - d.length();
            while (true) {
                synchronized(b) {
                    int time = 0;
                    if (!b.ready()) {
                        time += 10;
                        try { Thread.sleep(10); } catch (InterruptedException e) { }
                        if (time > 3000)
                            System.err.println("Dangit, I was all set to read in some data and they psyched me! (I'm probably deadlocked now)");
                    }
                    b.mark(MSG_WINDOW);
                    len = b.read(bufr, 0, MSG_WINDOW);
                    if (len == -1) break;
                    b.reset();
                    if (String.valueOf(bufr, 0, len).indexOf(d) >= 0) {
                        goodlen = String.valueOf(bufr).indexOf(d) + d.length();
                        len = b.read(bufr, 0, goodlen);
                        strb.append(bufr, 0, len);
                        break;
                    }
                    total += len = b.read(bufr, 0, goodlen);
                    strb.append(bufr, 0, len);
                }
            }
            /*
              while (strb.toString().indexOf(d) < 0) {
              strb.append((char)b.read());
              }
            */
            str = new String(strb);

        } catch (IOException e) {
            System.err.println("Error receiving message: " + e);
            return null;
        }

        if (str.indexOf(d) >= 0) {
            str = str.substring(0, str.indexOf(d));
            if (str.length() <= 0)
                System.err.println("Warning: Received empty message");
            return str;

        } else {
            System.err.println("Error receiving message: Delimiter (" + MSG_DELIM + ") not found in message -\n" + str);
            return null;
        }
    }

  /**
   * Same as above, but uses MSG_DELIM as the delimiter.  This method retained
   * for compatibility.
   * @param b The reader used to get the message
   * @return The message, or null if error occurs
   */
  public static String receive(BufferedReader b) {
    return receive(b, MSG_DELIM);
  }

  /**
   * Recieves a message (string) from a stream.  Typically this
   * should be called to fill in the data parameter of the 
   * Message constructor.  This is a blocking receive, use
   * the ready() function of BufferedReader to determine beforehand
   * if a message is available.
   * @param b The reader used to get the message
   * @param length The length of the data to read
   * @return The message, or null if error occurs
   */
  public static String receive(BufferedReader b, int length) {
    char bufr[] = new char[MSG_WINDOW];
    StringBuffer strb = new StringBuffer();
    String str;
    int goodlen, total = 0, len;

    try {
      goodlen = MSG_WINDOW;
      while (total < length) {
	if (!b.ready()) {
	  try { Thread.sleep(10); } catch (InterruptedException e) { }
	}
	if (total + goodlen > length)
	  goodlen = length - total;
	total += len = b.read(bufr, 0, goodlen);
	if (len == -1) break;
	strb.append(bufr, 0, len);
      }
      str = new String(strb);

    } catch (IOException e) {
      System.err.println("Error receiving message: " + e);
      return null;
    }

    if (str.length() <= 0) {
      System.err.println("Warning: Received empty message");

    } else if (str.length() < length) {
      System.err.println("Warning: Did not meet the requested length");
    }

    return str;
  }

    /**
     * Clone
     */
    public Object clone() {
        Message n = null;

        try {
            n = (Message)super.clone();
        } catch (Exception e) {
            System.out.println("Clone Error: " + e);
        }

        n.setContent(getContent());
        n.setSourceAddr(getSourceAddr());
        n.setDestAddr(getDestAddr());
        n.setSourcePort(getSourcePort());
        n.setDestPort(getDestPort());
        n.setSendTime(getSendTime());
        n.setReceiveTime(getReceiveTime());
        n.setConnection(getConnection());
        n.data = data;

        return n;
    }

  /**
   * Stringify's the message.  For logging, not transfer.
   */
  public String toString() {
      return ("Message (from " + sourceaddr + ":" + sourceport + " at " + getSendTime() +
      ") (to " + destaddr + ":" + destport + " at " + getReceiveTime() + ") Data: \"" +
      getData() + "\"" + "  Content: \"" + getContent() + "\"");
  }
}
