/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Log.java
 ***************************************************************************************/
package utilities;

/* Global Inclues */
import java.io.*;
import java.util.*;

/* Local Includes */
import utilities.Clock;

/**
 * Logs stuff, or formats log strings
 */
public class Log extends OutputStreamWriter {

    /**
     * Logging constants, use them if you want
     */
  public static final int LOG_SILENT  = -1;
  public static final int LOG_EMERG   = 0;
  public static final int LOG_ALERT   = 0;
  public static final int LOG_CRIT    = 0;
  public static final int LOG_ERR     = 0;
  public static final int LOG_WARNING = 1;
  public static final int LOG_INFO    = 2;
  public static final int LOG_DEBUG   = 3;
  public static final int LOG_DEBUG2  = 4;
  public static final int LOG_SPAM    = 5;

  public static final String ANSI_S = "\033[";
  public static final String ANSI_F = "\033[m";

  public static final String ANSI_C_RED = "31m";
  public static final String ANSI_C_GRN = "32m";
  public static final String ANSI_C_YEL = "33m";
  public static final String ANSI_C_BLU = "34m";
  public static final String ANSI_C_PUR = "35m";
  public static final String ANSI_C_CYN = "36m";
  public static final String ANSI_C_PNK = "37m";

  private static final int DEFAULT_LEVEL = 3;
  private static Log defLog;
  protected static long time = Calendar.getInstance().getTime().getTime();
  protected static String name = "???";
  private boolean flush = true;
  private static boolean absolute = false;
  protected int level;
  private static boolean colors = true;

  /**
   * Constructor
   */
  public Log(OutputStream s, int l) {
    super(s);

    setLevel(l);
    log("Logfile started.");
  }

  /**
   * Constructor
   * @param s The stream to write to
   */
  public Log(OutputStream s) {
    this(s, DEFAULT_LEVEL);
  }
  
  /**
   * Constructor
   * @param file The file to log to
   */
  public Log(String file) throws IOException { 
    this(new FileOutputStream(file));
  }
  
  /**
   * Constructor, with append boolean
   * @param file The file to log to
   * @param append Append to file or not
   */
  public Log(String file, boolean append) throws IOException { 
    this(new FileOutputStream(file, append));
  }
  
  /**
   * Destructor
   */
  protected void finalize() throws IOException {
    log("Logfile completed.");
    flush();
    close();
  }
  
  /**
   * Sets the name
   */
  public void setName(String n) { name = n; }
  public String getName() { return name; }

  /**
   * Sets the colored output flag
   */
  public static void setColors(boolean c) {
  	colors = c;
  }

  /**
   * Sets the default log stream
   */
  public static synchronized void setDefault(Log l) {
    defLog = l;
  }
  
  /**
   * Gets the default log object, a generic one is generated if none available
   * @return The default Log object
   */
  public static synchronized Log getDefault() {
    if (defLog == null) setDefault(new Log(System.err));

    return defLog;
  }

    /**
     * Gets the logging level
     */
    public int getLogLevel() { return level; }

  /**
   * Sets the logging level
   * @param l The new log level
   * @deprecated Use setLogLevel
   */
  public void setLevel(int l) {
    setLogLevel(l);
  }

  /**
   * Sets the logging level
   * @param l The new log level
   */
  public void setLogLevel(int l) {
    level = l;
  }

  /**
   * Formats a string so it can be logged
   * @param s The string to add formatting to
   * @param l The level to format at
   * @return The formatted string
   */
  public static String formatStr(String s, int l) {
    long t = Calendar.getInstance().getTime().getTime();
    if (!absolute) t -= time;

    //s = "(" + t + "-" + Thread.currentThread().getName() + ") [" + Clock.getTime() + ":" + name + "] " + s + "\n";

    if (colors) {
	    if (l == 0) {
		s = ANSI_S + ANSI_C_RED + s + ANSI_F;
	    } else if (l == 1) {
		s = ANSI_S + ANSI_C_PUR + s + ANSI_F;
	    }

	    s = ANSI_S + ANSI_C_CYN + "(" + Thread.currentThread().getName() + ")" + ANSI_F
		+ ANSI_S + ANSI_C_GRN + " [" + Clock.getTime() + ":" + name + "] " + ANSI_F
		+ s + "\n";
    } else {
	    s = "(" + Thread.currentThread().getName() + ")" +
		" [" + Clock.getTime() + ":" + name + "] " +
		s + "\n";
    }



    /*
    switch(l) {
    case 0:
      s = "(" + Clock.getTime() + ")" + s + "\n";
      break;
    case 1:
      s = "(" + Clock.getTime() + ") #" + Thread.currentThread().getName() + ": " + s + "\n";
      break;
    default:
      s = "(" + Calendar.getInstance().getTime().getTime() + "-" + Clock.getTime() + ") #" + Thread.currentThread().getName() + ": " + s + "\n";
    }
    */

    return s;
  }
  
  /**
   * Logs a string to the logfile or stream
   * @param s The string to log
   * @param l The logging level of the string
   */
  public static final Integer lock = new Integer(0);
  public void log(String s, int l) {
    
    if (l > level) return;
    
    try {
      synchronized(lock) {
        write(formatStr(s, l));
      }
      if (flush) flush();
      
    } catch (IOException e) {
      System.err.println("Error writing to logfile.");
    }
  }
  
  /**
   * Logs with the default log level
   */
  public void log(String s) {
    log(s, DEFAULT_LEVEL);
  }
  
  /**
   * Sets flushing status
   */
  public void setFlush(boolean f) {
    flush = f;
    if (flush) flush();
  }
  
  /**
   * Flushes the write buffer
   */
  public void flush() {
    try {
      super.flush();
    } catch (IOException e) {
      System.err.println("Error flushing logfile.");
    }
  }
}
