/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 * LengthPrefixMessage.java
 ***************************************************************************************/

package utilities;

import java.io.*;
import java.text.*;

/**
 * A message encoding where the content length is prepended to the
 * message in a fixed width field (e.g. a valid message would be
 * "0005hello").  Currently, this field with is fixed in the PREFIXLEN
 * constant.
 */
public class LengthPrefixMessage extends Message {
  protected static final int PREFIXLEN = 6;

  /**
   * Send in your encoded message to decode it
   * @param s The length-prefixed message to decode
   */
  public LengthPrefixMessage(String s) {

    setData(s);
    setContent(parseData(s));
  }

  /**
   * A "tell" message
   * @param c The content to send
   * @param da The destination address
   */
  public LengthPrefixMessage(Object c, String da) {
    this("tell", c, da);
    updateData();
  }

  /**
   * Normal constructor
   * @param p The message performative (e.g. "ask", "tell", etc.)
   * @param c The content to send
   * @param da The destination address
   */
  public LengthPrefixMessage(String p, Object c, String da) {
    super(c, "", 0, da, 0);
    updateData();
  }

  /**
   * Updates the string data, assumes content length will fit in 
   * PREFIXLEN characters.
   */
  private void updateData() {
    String prefix = "" + getContent().toString().length();

    while (prefix.length() < PREFIXLEN)
      prefix = "0" + prefix;

    setData(prefix + getContent().toString());
  }

  /**
   * Parses the encoded string.
   * @param s The string to parse
   */
  private String parseData(String s) {
    String str = "";

    if (s.length() >= PREFIXLEN) {
      String prefix = s.substring(0, PREFIXLEN);
      int len = Integer.parseInt(prefix);

      if (s.length() != PREFIXLEN + len)
	System.err.println("Warning: Message prefix (" + len + ") and content length (" + s.length() + ") don't match");

      str = s.substring(PREFIXLEN, PREFIXLEN + len);
    }

    return str;
  }

  /**
   * Reads it in, first looking for the length prefix, then
   * using that to read the content.
   * @param b The reader used to get the message
   * @return The message data, or null if error occurs
   */
  public static String receive(BufferedReader b) {
    String prefix = receive(b, PREFIXLEN);
    int len = 0;

    // Read the length in
    try {
      len = Integer.parseInt(prefix);
    } catch (NumberFormatException e) {
      System.err.println("Error: " + e);
      return "";
    }

    // Then read the rest
    String str = receive(b, len);

    return prefix + str;
  }

  /**
   * Sets the message's content string
   * @param c The content
   */
  public void setContent(Object c) {
    super.setContent(c);
    updateData();
  }
}
