/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 * KQMLMessage.java
 ***************************************************************************************/
package utilities;

import java.io.*;
import java.text.*;

//import KQMLLayer.*;
import utilities.KQML;

/**
 * This class is essentially a wrapper for the KQMLLayer message class
 */
public class KQMLMessage extends Message implements Cloneable {
  private String perf = "";
  private KQML kqmlmess = new KQML();

  /**
   * Send in your encoded message to decode it
   * @param s The KQML formatted message to decode
   */
  public KQMLMessage(String s) {

    try {
      kqmlmess.parseMessage(s);
      super.setData(s);
      super.setSourceAddr(kqmlmess.getValue("sender"));
      if (getField("sender-port") != null)
          super.setSourcePort(Integer.parseInt(kqmlmess.getValue("sender-port")));
      super.setDestAddr(kqmlmess.getValue("receiver"));
      if (getField("receiver-port") != null)
          super.setDestPort(Integer.parseInt(kqmlmess.getValue("receiver-port")));
      super.setContent(kqmlmess.getValue("content"));
      perf = kqmlmess.getValue("performative");
    } catch (ParseException e) {
      System.err.println(e);
    }
  }

  /**
   * A "tell" message
   * @param c The content to send
   * @param da The destination address
   */
  public KQMLMessage(Object c, String da) {
    this("tell", c, da);
  }

  /**
   * Normal constructor
   * @param p The message performative (e.g. "ask", "tell", etc.)
   * @param c The content to send
   * @param da The destination address
   */
  public KQMLMessage(String p, Object c, String da) {
    this(p, c, da, -1);
  }

  /**
   * Normal constructor
   * @param p The message performative (e.g. "ask", "tell", etc.)
   * @param c The content to send
   * @param da The destination address
   * @param po The port number
   */
  public KQMLMessage(String p, Object c, String da, int po) {
    super(c, "", -1, da, po);
    perf = p;
    addField("language", "KQML");
    makeMessage();
  }

  /**
   * Creates the correct KQML data to send across the network.
   * This function is called automatically any time the message
   * parameters are changed.
   */
  private void makeMessage() {
    addField("performative", perf);
    addField("sender", getSourceAddr());
    if (getSourcePort() >= 0)
        addField("sender-port", String.valueOf(getSourcePort()));
    addField("receiver", getDestAddr());
    if (getDestPort() >= 0)
        addField("receiver-port", String.valueOf(getDestPort()));

    if (getContent() != null) {
        String c = getContent().toString();
        if (c.indexOf(" ") > 0) {
            if ((!c.startsWith("(")) && (!c.endsWith(")"))) {
                c = "(" + c + ")";
            }
        }
        addField("content", c);
    } else {
        removeField("content");
    }
  }

  /**
   * Adds a field/ value pair to the message
   * @param fn The name of the field to add (e.g. "in-reply-to")
   * @param fc The content of that field (e.g. "pulse")
   */
  public void addField(String fn, String fc) {
    if (fc == null) { removeField(fn); return; }

    kqmlmess.addField(fn, fc);
    updateData();
  }

  /**
   * Removes a field/ value pair from the message
   * @param fn The name of the field to remove (e.g. "in-reply-to")
   */
  public void removeField(String fn) {

    kqmlmess.removeField(fn);
    updateData();
  }

  /**
   * Updates the string data
   */
  private void updateData() {
    setData(kqmlmess.getSendString());
  }

  /**
   * Get a field/ value pair to the message
   * @param fn The name of the field to get (e.g. "in-reply-to")
   * @return The content of that field
   */
  public String getField(String fn) {
    return kqmlmess.getValue(fn);
  }

  /**
   * Performative accessors
   */
  public String getPerformative() { return perf; }
  public void setPerformative(String p) {
    perf = p;
    makeMessage();
  }

  /**
   * Sets the message's content string
   * @param c The content
   */
  public void setContent(Object c) {
    super.setContent(c);
    makeMessage();
  }

  /**
   * Sets the message's destination address
   * @param da The destination, typically an agent's name
   */
  public void setDestAddr(String da) {
    super.setDestAddr(da);
    makeMessage();
  }

  /**
   * Sets the message's destination port
   * @param sp The destination port number
   */
  public void setDestPort(int dp) {
    super.setDestPort(dp);
    makeMessage();
  }

  /**
   * Sets the message's source address
   * @param sa The source, the name of the agent sending the message
   */
  public void setSourceAddr(String sa) {
    super.setSourceAddr(sa);
    makeMessage();
  }

  /**
   * Sets the message's source port
   * @param sp The source port number
   */
  public void setSourcePort(int sp) {
    super.setSourcePort(sp);
    makeMessage();
  }

  /**
   * Returns the first word in the content (used to determine the
   * type of content), cleaning the KQML stuff off first
   */
  public String contentWord() {
    if (getContent() == null) return null;
    return contentWord(cleanString(getContent().toString()));
  }

  /**
   * Returns the data in the content (the stuff after the first
   * word in the content), cleaning the KQML stuff off first
   */
  public String contentData() {
    if (getContent() == null) return null;
    return contentData(cleanString(getContent().toString()));
  }

  /**
   * Strips off the KQML parens, useful on content string
   * @param str The string to clean.
   */
  private String cleanString(String str) {
    if (str == null) return null;

    // Strip off the KQML parens
    if (str.startsWith("("))
      str = str.substring(1);
    if (str.endsWith(")"))
      str = str.substring(0, str.length()-1);
    str = str.trim();

    return str;
  }


    /**
     * Clone
     */
    public Object clone() {
        KQMLMessage n = (KQMLMessage)super.clone();

        n.setPerformative(getPerformative());
        n.kqmlmess = (KQML)kqmlmess.clone();

        makeMessage();

        return n;
    }

  /**
   * Stringify, not the same thing as the network-ready data string
   */
  public String toString() {
    return ("Message (from " + getSourceAddr() + ") " +
      "(to " + getDestAddr() + ") \"" +
      getData() + "\"\n");
  }
}





