/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	KQML.java
 ***************************************************************************************/
package utilities;

/* Global Inclues */
import java.io.*;
import java.util.*;
import java.text.*;

/**
 */
public class KQML implements Cloneable {
  Hashtable fields = new Hashtable();
  String performative;

  /**
   * Boring constructor
   */
  public KQML() {
    performative = "";
  }

  /**
   * Parse constructor
   * @param str The KQML message parse
   * @exception ParseException Thrown when there is an error parsing the string
   * @see #parseMessage
   */
  public KQML(String str) throws ParseException {
    parseMessage(str);
  }

  /**
   * Parses the message into this object
   * @param str The kqml formatted message
   * @exception ParseException Thrown when there is an error parsing the string
   */
  public void parseMessage(String str) throws ParseException {
    StringCharacterIterator it = new StringCharacterIterator(str);
    int start, end, count;
    String item, value;

    // Empty current table
    fields.clear();

    // Avoid space
    while (Character.isWhitespace(it.current())) it.next();
    if (it.current() == CharacterIterator.DONE)
      throw new ParseException("KQML message must begin with (", it.getIndex());

    // Check start
    if (it.current() != '(')
      throw new ParseException("KQML message must begin with (", it.getIndex());
    it.next();

    // Avoid space
    while (Character.isWhitespace(it.current())) it.next();
    if (it.current() == CharacterIterator.DONE)
      throw new ParseException("Error finding performative", it.getIndex());

    // Get performative
    start = end = it.getIndex();
    while (true) {
      if ((Character.isLetterOrDigit(it.current())) ||
	   Character.getType(it.current()) == Character.DASH_PUNCTUATION)
	it.next();
      else if (Character.isWhitespace(it.current())) {
	end = it.getIndex();
	break;
      } else
	throw new ParseException("Error finding performative", it.getIndex());
    }
    performative = str.substring(start, end);

    // Start getting fields
    while(true) {
      // Skip whitespace
      while (Character.isWhitespace(it.current())) it.next();
      if (it.current() == CharacterIterator.DONE)
	throw new ParseException("Unexpected end of message", it.getIndex());

      // Look for end of message
      if (it.current() == ')')
	break;

      // Look for start of parameter
      else if (it.current() == ':') {
	// Get parameter name
	it.next();
	start = end = it.getIndex();
	while (it.current() != CharacterIterator.DONE) {
	  if (it.current() == '(') {
	    throw new ParseException("Error getting parameter name", it.getIndex());

	  } else if (it.current() == ')') {
	    end = it.getIndex();
	    break;

	  } else if (Character.isWhitespace(it.current())) {
	    end = it.getIndex();
	    break;

	  } else
	    it.next();
	}
	item = str.substring(start, end);
	
	// Skip whitespace
	while (Character.isWhitespace(it.current())) it.next();
	if (it.current() == CharacterIterator.DONE)
	  throw new ParseException("Unexpected end of message", it.getIndex());

	// New parameter?
	if (it.current() == ':') {
	  fields.put(item, "");
	  continue;
	} else if (it.current() == ')') {
	  continue;
	}

	// Get parameter value
	count = 0;
	start = end = it.getIndex();
	while (it.current() != CharacterIterator.DONE) {
	  if (it.current() == '(')  {
	    it.next();
	    count++;

	  } else if (it.current() == ')') {
	    if (count == 0) {
	      end = it.getIndex();
	      break;
	    } else {
	      it.next();
	      count--;
	    }

	  } else if (Character.isWhitespace(it.current())) {
	    if (count == 0) {
	      end = it.getIndex();
	      break;
	    } else
	      it.next();

	  } else
	    it.next();
	}
	value = str.substring(start, end);

	// Add it
	fields.put(item, value);

      } else
	throw new ParseException("Unexpected end of message", it.getIndex());
    }
  }

  /**
   * Returns the value of a parameter
   * @param i The name of the value to query
   * @return The value, or null if not found
   */
  public String getValue(String i) {
    
    if (i.equalsIgnoreCase("performative"))
      return getPerformative();
   
    return (String)fields.get(i);
  }

  /**
   * Sets the value of a parameter
   * @param i The name of the value to set
   * @param v The value to set 
   * @deprecated Use addField
   */
  public void addFieldValuePair(String i, String v) {
      addField(i, v);
  }

  /**
   * Sets the value of a parameter
   * @param i The name of the value to set
   * @param v The value to set 
   */
  public void addField(String i, String v) {

    if (i.equalsIgnoreCase("performative")) {
      setPerformative(v);
      return;
    }
   
    fields.put(i, v);
  }

  /**
   * Removes a parameter
   * @param i The name of the value to remove
   */
  public void removeField(String i) {

    if (i.equalsIgnoreCase("performative")) {
      return;
    }
   
    fields.remove(i);
  }

  /**
   * Gets the performative value
   * @return The performative
   */
  public String getPerformative() { return performative; }

  /**
   * Sets the performative value
   * @param p The new performative
   */
  public void setPerformative(String p) { performative = p; }

  /**
   * Encodes the actual string data that is to be sent
   * @return The message string
   */
  public String getSendString() {
    StringBuffer buf = new StringBuffer();
    
    buf.append("(");
    buf.append(getPerformative());

    Enumeration e = fields.keys();
    while(e.hasMoreElements()) {
      String key = (String)e.nextElement();
      buf.append(" :" + key + " " + getValue(key));
    }

    buf.append(")");

    return buf.toString();
  }

    /**
     * Clone
     */
    public Object clone() {
        KQML n = new KQML();

        n.setPerformative(getPerformative());

        Enumeration e = fields.keys();
        while(e.hasMoreElements()) {
            String key = (String)e.nextElement();
            n.addField(key, getValue(key));
        }

        return n;
    }

  /**
   * Writes us out to a string
   * @return A stringification of us
   */
  public String toString() {
    return getSendString();
  }
}
