/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/************************************************************
 * GraphNode.java
 ************************************************************/

package utilities;

/* Global imports */
import javax.swing.*;
import java.beans.*;
import java.awt.*;
import java.io.*;
import java.util.*;

import utilities.cfg.*;

/**
 * This is a superclass for the generic purpose of drawing
 *  graphs made up of nodes and edges.  Its primary purpose
 *  is for the resource hierarchy interface.
 * <P>
 * It should probably not be instantiated directly.  It is
 *  modeled after taems' Node class, minus the agent/tames specific
 *  stuff.
 */

public class GraphNode implements Serializable, Cloneable, ConfiguredObject,
                                  Comparable 
{ 
    protected Vector outedges = new Vector();
    protected Vector inedges = new Vector();
    protected Vector undiredges = new Vector();  // undirected edges
    protected Hashtable attributes = new Hashtable();
    protected boolean draggable = false;
    protected transient boolean selected;

    /**
     * Default constructor
     * @param l The node's label
     */
    public GraphNode(String l)
    {
        setLabel(l);
    }

    /**
     * Blank Constructor
     */
    public GraphNode()
    {
        this(null);
    }

    public synchronized void updateCfg()
    {
        Object tmp;
        CfgManager manager = CfgManager.getManager(this);
        if (manager == null)
        {
            System.err.println("GraphNode " + getLabel() + "has no cfg manager...");
            return;
        }
        String key = manager.getObjectKey(this);
        if (key == null || key.length() == 0)
            return;
        tmp = manager.getValue(key + "/name");
        if ((tmp != null) && (tmp instanceof String))
            setLabel((String)tmp);
    }
        
    public synchronized void saveCfg()
    {
        CfgManager manager = CfgManager.getManager(this);
        if (manager == null)
            return;
        String key = manager.getObjectKey(this);
        if (key == null || key.length() == 0)
            return;
        manager.put(key + "/name", getLabel());
    }
    
    /**
     * Accessors
     */
    public String getLabel() 
    { 
        return label;
    }

    /*
     * @deprecated Nothing that uses panel should be called any more
     */

    String label;
    public void setLabel(String l) 
    {
        label = l;
    }

    /**
     * Note that this uses the toString() method for the key
     * to actually store the object.
     * @param k The key identifying the desired attribute
     * @return The object, or null if not found
     */
    public Object getAttribute(Object k) {
        return attributes.get(k.toString());
    }

    /**
     * Sets an attribute's data.  Attribute keys should not
     * contain whitespace.
     * <P>
     * If a key is added with a null data object, the key will
     * be removed from the attribute set.
     * @param k The key identifying the desired attribute
     * @param d The attribute data
     */
    public void setAttribute(Object k, Object d) { 
        if (d == null)
            attributes.remove(k.toString());
        else
            attributes.put(k.toString(), d);
    }

    /**
     * Returns the attribute names
     * @return An enumeration of all the attribute keys
     */
    public Enumeration getAttributes() { return attributes.keys(); }

    /**
     * Determines if the node has a particular attribute
     * @return True if the attribute is present
     */
    public boolean hasAttribute(Object k) {
        return attributes.containsKey(k.toString());
    }

    public void addEdge(GraphNode t)
    {
        if (!(undiredges.contains(t)))
        {
            undiredges.addElement(t);
            t.addEdge(this);
        }
    }

    public void removeEdge(GraphNode t)
    {
        if (undiredges.contains(t))
        {
            undiredges.removeElement(t);
            t.removeEdge(this);
        }
    }

    public void addEdge(GraphEdge e, GraphNode t)
    {
        e.setEndpoints(this, t);
    }

    public void removeEdge(GraphEdge e)
    {
        if ((outedges.contains(e)) || (inedges.contains(e)))
            e.excise();
    }

    protected void removeInEdge(GraphEdge e)
    {
        inedges.removeElement(e);
    }

    public Enumeration getOutEdges()
    {
        return outedges.elements();
    }

    protected void removeOutEdge(GraphEdge e)
    {
        outedges.removeElement(e);
    }

    protected void addOutEdge(GraphEdge e)
    {
        if (this instanceof GraphEdge)
        {
            System.err.println("Error: Cannot attach an Edge to another Edge");
            return;
        }
        outedges.addElement(e);
    }

    public boolean hasOutEdges()
    {
        return (outedges.size() > 0);
    }

    public int numOutEdges()
    {
        return outedges.size();
    }

    public Enumeration getInEdges()
    {
        return inedges.elements();
    }

    public Enumeration getUndirEdges()
    {
        return undiredges.elements();
    }

    public boolean hasUndirEdges()
    {
        return undiredges.size() > 0;
    }

    public int numUndirEdges()
    {
        return undiredges.size();
    }

    protected void addInEdge(GraphEdge e)
    {
        if (this instanceof GraphEdge)
        {
            System.err.println("Error: Cannot attach an Edge to another Edge");
            return;
        }
        inedges.addElement(e);
    }

    public boolean hasInEdges()
    {
        return (inedges.size() > 0);
    }

    public int numInEdges()
    {
        return inedges.size();
    }

    public void excise()
    {
        while (hasInEdges())
        {
            GraphEdge e = (GraphEdge) inedges.firstElement();
            e.setTo(null);
        }
        while (hasOutEdges())
        {
            GraphEdge e = (GraphEdge) outedges.firstElement();
            e.setTo(null);
        }
    }

    public int compareTo(Object o)
    {
        if (!(o instanceof GraphNode))
            return 1;
        if (equals(o))
            return 0;
        int result;
        if (getLabel() == null)
        {
            if (((GraphNode) o).getLabel() == null)
                result = 0;
            else
                result = 1;
        } else {
            if (((GraphNode) o).getLabel() == null)
                result = -1;
            else
                result = getLabel().compareTo(((GraphNode) o).getLabel());
        }
        if (result == 0)
        {
            if (o == this)
                return 0;
            else
                return o.hashCode() - hashCode();
        } else
            return result;
    }
    
    public boolean equals(Object o)
    {
        return this == o;
    }
    
    public boolean matches(Object o) 
    {
        GraphNode n;
        if (o == null)
            return true;
        if (o instanceof GraphNode)
            n = (GraphNode) o;
        else
            return false;
        if (n.getLabel() != null) 
        {
            if (getLabel() == null) 
            {
                return false;
            } else {
                if (!getLabel().startsWith(n.getLabel())) 
                    return false;
            }
        }
        if (n.getClass().isInstance(this))
            return true;
        return false;
    }

    public Object clone()
    {
        GraphNode cloned = null;

        try {
            cloned = (GraphNode) super.clone();
        } catch (Exception e) {
            System.out.println("Clone Error: " + e);
        }

        cloned.outedges = new Vector();
        Enumeration e = getOutEdges();
        while (e.hasMoreElements())
        {
            GraphEdge g = (GraphEdge)((GraphEdge) e.nextElement()).clone();
            g.setFrom(cloned);
        }
        cloned.inedges = new Vector();
        return cloned;
    }

    public String toString()
    {
        return getLabel();
    }

    /***********************************************
     *              Drawing stuff                  *
     ***********************************************/

    public JPanel getDefaultPanel()
    {
        return new NodePanel(this, getLabel());
    }
} 
