/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************
 *     GraphLayout.java
 ***************************************/
 
package utilities;

import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import utilities.cfg.*;
      
public class GraphLayout implements LayoutManager2, ConfiguredObject
{
    Hashtable components = new Hashtable();
    TreeMap nodes = new TreeMap();
    Hashtable locations = new Hashtable();
    Dimension preferred = new Dimension(25, 25);
    Point origin = new Point(0, 0);
    boolean layedoutonce = false;
    protected int H_SPACE = 50;
    protected int V_SPACE = 50;
    protected int H_SPACE2 = 50;
    protected int V_SPACE2 = 50;
    protected int H_MARGIN = 50;
    protected int V_MARGIN = 50;    
    int rows, columns;
    int currow = 0;
    int curcol = 0;
    boolean willsave = true;
    boolean handlesrecursiveplacement = true;
    String savefile = null;
    
    public GraphLayout(int r, int c, String savename, JTextArea txt)
    {
        rows = r;
        columns = c;
        text.setEditable(false);
        text = txt;
        savefile = savename;
        if (savefile == null)
            willsave = false;
        text.setBackground(Color.lightGray);
        loadNodePlacements();
    }
   
    public GraphLayout(int r, int c)
    {
        this(r, c, null, new JTextArea());
    }
    
    public GraphLayout(String n, JTextArea txt)
    {
        this(0, 0, n, txt);
    }
    
    public GraphLayout()
    {
        this(0, 0, null, new JTextArea());
    }
    
    public void setSave(boolean b)
    {
        willsave = b;
    }
    
    public void setRecursivePlacement(boolean b)
    {
        handlesrecursiveplacement = b;
    }
    
    public boolean willHandleRecursivePlacement()
    {
        return handlesrecursiveplacement;
    }
    
    public void setSaveFile(String s)
    {
        savefile = new String(s);
    }
    
    public void setSpacing(int hspace, int vspace, int hspace2, int vspace2, int hmargin, int vmargin)
    {
        H_SPACE = Math.max(hspace, 0);
        V_SPACE = Math.max(vspace, 0);
        H_SPACE2 = Math.max(hspace2, 0);
        V_SPACE2 = Math.max(vspace2, 0);
        H_MARGIN = Math.max(hmargin, 0);
        V_MARGIN = Math.max(vmargin, 0);
    }
    
    public void addLayoutComponent(String name, Component component) 
    { 
        addLayoutComponent(component, (Object) name);
    }
    
    public void removeLayoutComponent(Component component)
    {
        if (components.contains(component))
            components.remove(component);
    }
     
    public Dimension preferredLayoutSize(Container parent)
    {
        if (!layedoutonce)
            placeNodes();
        return preferred;
    }
     
    public Dimension minimumLayoutSize(Container parent)
    {
        return preferredLayoutSize(parent);
    }
     
    public void layoutContainer(Container parent)
    {
        placeNodes();
        addMenuIfNecessary(parent);
    }
    
    GraphNode notanode = new GraphNode("not a node - if this is visible, there" + 
            " is a bug in the Graph rendering system");
    
    public void addLayoutComponent(Component comp, Object constraints)
    {
        if (constraints != null && constraints instanceof GraphNode)
        {
            components.put(comp, constraints);
            nodes.put(constraints, comp);
        } else {
            System.err.println("here!");
            Object o = nodes.get(notanode);
            if (o == null || !(o instanceof Set))
                nodes.put(notanode, o = new TreeSet(new ComponentComparator()));
            ((Set) o).add(comp);
            components.put(comp, notanode);
            return;
        }
            
        comp.addMouseListener(new MouseListener() {
                public void mouseClicked(MouseEvent e) { }
                public void mouseEntered(MouseEvent e) { }
                public void mouseExited(MouseEvent e) { }
                public void mousePressed(MouseEvent e) { processMouseEvent(e); }
                public void mouseReleased(MouseEvent e) { processMouseEvent(e); }

            });
        comp.addMouseMotionListener(new MouseMotionListener() {
                public void mouseDragged(MouseEvent e) { processMouseEvent(e); }
                public void mouseMoved(MouseEvent e) { }            
            });
    }
    
    public Dimension maximumLayoutSize(Container target)
    {
        return preferredLayoutSize(target);
    }
    
    public float getLayoutAlignmentX(Container target)
    {
        return (float) 0.5;
    }
    
    public float getLayoutAlignmentY(Container target)
    {
        return (float) 0.5;
    }
     
    public void invalidateLayout(Container target)
    {
    
    }
     
    public boolean findConflict(Component c, Rectangle location)
    {
        Enumeration e = components.keys();
        LayoutManager layout;
        while (e.hasMoreElements())
        {
            Component com = (Component) e.nextElement();
            if (c == com)
                continue;
            if (com.getBounds().intersects(location) && hasBeenPlaced(com))
            {

                // if the component uses a graphlayout we can recurse and make maximal use of space...

                if (com instanceof Container 
                        && (layout = ((Container)com).getLayout()) instanceof GraphLayout // does it use a graphlayout?
                        && layout != this                                                 // just in case
                        && ((GraphLayout)layout).willHandleRecursivePlacement()           // is it willing to do this?
                        && !(c instanceof Container && (((Container)c).getLayout() instanceof GraphLayout)) ) // no two layout using components should overlap
                {
                    Rectangle newlocation = new Rectangle(location.x - com.getLocation().x, location.y - com.getLocation().y, location.width, location.height);
                    if (((GraphLayout)layout).findConflict(c, newlocation))
                    {
                        return true;
                    }
                } else
                    return true;
            }
        }
        return false;
    }
    
    public Vector findNodes(Point p)
    {
        Vector v = new Vector();
        
        Enumeration e = components.keys();
        while (e.hasMoreElements())
        {
            Component c = (Component) e.nextElement();
            if (c.getBounds().contains(p) && c.isVisible())
                v.addElement(c);
        }
        return v;
    }
    
    public Component findNode(Point p)
    {
        Vector v = findNodes(p);
        Enumeration e = v.elements();
        Component i = null;
        Component n = null;
        while (e.hasMoreElements())
        {
            Component c = (Component) e.nextElement();
            GraphNode no = getNode(c);
            if (no instanceof GraphEdge)
                return c;
            else if (isMoveable(c))
                i = c;
            else
                n = c;
        }
        if (n != null)
            return n;
        else
            return i;
    }        
    
    protected boolean isMoveable(Component c)
    {
        if (c instanceof NodePanel)
            return ((NodePanel)c).isMoveable();
        else
            return false;
    }
    
    Point snap = new Point(0, 0);
    Point dragoffset = new Point(0, 0);
    Component selected = null;
    JTextArea text = new JTextArea("Nothing Selected");
    
    public JTextArea getText() { return text; }
    
    boolean thorough_placement = false;
    
    // speed up things...
    GraphNode ncached = null;
    Component sourcecached = null;
      
    protected void processMouseEvent(MouseEvent e)
    {
        Point p = e.getPoint();
        GraphNode n = null;
        
        // this first little bit strikes a delicate balance between relying on e.getSource() and
        //  findNode, neither of which produce correct behaviour by themselves

        Component source = (Component) e.getSource();

        p = new Point(p.x + source.getLocation().x, p.y + source.getLocation().y);
         
        // only call findNode (a rather thorough [read: slow] function) if absolutely necessary
        if (source == sourcecached)
        {
            n = ncached;
        } else if (e.getID() != MouseEvent.MOUSE_PRESSED) {
            n = ncached;
            source = sourcecached;
        } else {
            Component temp;
            temp = findNode(p);
            if (temp != null)    
                source = temp;
            if (source != null)
                n = getNode(source);
            ncached = n;
            sourcecached = source;
        } 

        if (source != null && source instanceof JComponent)
        {
            JPopupMenu menu = (JPopupMenu) ((JComponent)source).getClientProperty("GraphMenu");
            if (e.isPopupTrigger())
            {
                if (menu != null && !menu.isVisible())
                {
                    // FIXME - funky menuness - FIXME
                    // leave all this commented stuff in here until I understand how it works
                    
                    //menu.show(source, p.x - source.getLocation().x , p.y - source.getLocation().y);
                    menu.show((Component) e.getSource(), e.getPoint().x, e.getPoint().y);
                    //menu.setInvoker(source);
                    //menu.setLocation(new Point(e.getPoint().x + source.getLocationOnScreen().x, e.getPoint().y + source.getLocationOnScreen().y));
                    //menu.setVisible(true);
                    //System.err.println("p is " + p + " and source is at " + source.getLocation());
                    //System.err.println("p is " + e.getPoint() + " menu is " + menu + " at " + menu.getLocation());
                    return;
                }
            } else {
                if (e.getID() == MouseEvent.MOUSE_PRESSED && menu != null && menu.isVisible())
                    menu.setVisible(false);
            }
        }
        
        if (e.getID() == MouseEvent.MOUSE_PRESSED) {
            if (selected != null)
                if (selected instanceof NodePanel)
                    ((NodePanel)selected).setSelected(false);
            
            if (n != null)
            {
                if (n instanceof GraphNode)
                {
                    if (source instanceof NodePanel)
                        ((NodePanel)source).setSelected(true);
                    if (getNode(selected) != n)
                    {
                        System.err.println(n.toString());
                        text.setText(n.toString());
                        selected = source;
                        if (menu_action_container != null)
                            menu_action_container.repaint();
                        else if (source != null)
                            source.repaint();
                        source.getParent().repaint();
                    }
                }
                snap = new Point(source.getLocation());
                if (e.getSource() == source)
                    dragoffset = new Point(e.getX(), e.getY());
                else
                    dragoffset = new Point((e.getX() + ((Component) e.getSource()).getLocation().x) - source.getLocation().x, 
                                           (e.getY() + ((Component) e.getSource()).getLocation().y) - source.getLocation().y);
                
            } else
                text.setText("Nothing selected.");
        }
        if (e.getID() == MouseEvent.MOUSE_RELEASED) {
            if (selected != null && source == selected)
            {
                if (!isMoveable(selected))
                {    
                    selected.setLocation(snap);
                    if (menu_action_container != null)
                        menu_action_container.repaint();
                    else if (source != null)
                        source.repaint();
                    source.getParent().repaint();
                } else
                    saveNodePlacements();
            }
        }
        if (e.getID() == MouseEvent.MOUSE_DRAGGED) {

            if (selected != null && source == selected)
            {
                Point z;
                if (selected == e.getSource())
                    z = new Point(e.getX() + selected.getLocation().x, e.getY() + selected.getLocation().y);
                else
                    z = new Point(e.getX() + ((Component)e.getSource()).getLocation().x,
                                  e.getY() + ((Component)e.getSource()).getLocation().y);
                //Point z = new Point(selected.getLocation().x, selected.getLocation().y);
                //System.err.println("dragging! " + source + " and z is " + z);
                if (z.x >= 0 && z.y >= 0)
                {
                    selected.setLocation(new Point(z.x - dragoffset.x, z.y - dragoffset.y));
                }
            }
            if (menu_action_container != null)
                menu_action_container.repaint();
            else if (source != null)
                source.repaint();
                    source.getParent().repaint();
        }
        if (source != null)
        {
            storeNodePlacement(source);
        }
    }
    
    JComponent menu_action_container = null;
    JPopupMenu the_menu = null;
    Vector unaddedMenuItems = new Vector();
    
    public void addMenuItem(JMenuItem item)
    {
        if (the_menu == null)
            unaddedMenuItems.addElement(item);
        else
            the_menu.add(item);
    }
    
    // this also adds a mouse event listener.
    protected void addMenuIfNecessary(Container c)
    {
        c.addMouseListener(new MouseListener() {
                public void mouseClicked(MouseEvent e) { }
                public void mouseEntered(MouseEvent e) { }
                public void mouseExited(MouseEvent e) { }
                public void mousePressed(MouseEvent e) { processMouseEvent(e); }
                public void mouseReleased(MouseEvent e) { processMouseEvent(e); }
            });

        // sorry, AWT containers dont get a menu...
        if (c instanceof JComponent)
        {
            JComponent jc = (JComponent) c;   // hey jc, hey jc..
            menu_action_container = jc;
            Boolean B = (Boolean) jc.getClientProperty("GraphLayoutMenuInPlace");
            if (B == null || B.booleanValue() == false)
            {
                // use JMenus not menus
                JPopupMenu menu = (JPopupMenu) jc.getClientProperty("GraphMenu");
                if (menu == null)
                {
                    menu = new JPopupMenu();
                    jc.putClientProperty("GraphMenu", menu);
                }
                the_menu = menu;
                JMenuItem m = new JMenuItem("Refresh");
                m.addActionListener(new ActionListener()
                    {
                        public void actionPerformed(ActionEvent e) { menu_action_container.repaint(); }
                    });
                menu.add(m);
        
                m = new JMenuItem("Reset node placements");
                m.addActionListener(new ActionListener()
                    {
                        public void actionPerformed(ActionEvent e)
                        {
                            resetNodePlacing();
                            menu_action_container.repaint();
                        }
                    });
                menu.add(m);
                Enumeration e = unaddedMenuItems.elements();
                while (e.hasMoreElements())
                    menu.add((JMenuItem) e.nextElement());
                unaddedMenuItems = new Vector();
                menu.setInvoker(c);
                jc.putClientProperty("GraphLayoutMenuInPlace", new Boolean(true));
            }
        }
    }

    public Component getSelected()
    {
        return selected;
    }
    
    public void setSelected(Component c)
    {
        selected = c;
    }
    
    public GraphNode getNode(Component c)
    {
        if (c == null)
            return null;
        Object fnord = components.get(c);
        if (fnord == notanode)
            return null;
        else
            return (GraphNode) fnord;
    }
   
    public Component getComponent(GraphNode n)
    {
        return (Component) nodes.get(n);
    }
    
    protected String getSaveName(Component c)
    {
        String s = null;
        GraphNode n;
        if ((n = getNode(c)) != null)
        {
            if (n instanceof GraphNode)
            {
                s = n.getLabel();
                if ((s == null || s == "") && n instanceof GraphEdge)
                    s = "Edge: " + ((GraphEdge) n).getFrom().getLabel() +" to " + ((GraphEdge)n).getTo().getLabel();
            } else
                s = n.getLabel();
        } else
            s = String.valueOf(c.hashCode());
        return s;
    }
    
    protected void storeNodePlacement(Component c)
    {
        String s = getSaveName(c);
        
        if (s != null && s != "")
            locations.put(s, c.getLocation());
    }
     
    protected void restoreNodePlacement(Component c)
    {
        String s = getSaveName(c);
        Point loading = null;
        if (s != null && locations.containsKey(s))
            loading = (Point) locations.get(s);

        if (loading != null)
            loading = new Point(loading.x, loading.y);
        if (loading == null)
            placeNewNode(c);
        else
            placeNewNode(c, loading);
    }
   
    public Point getLocationForLine(GraphNode n, GraphNode src)
    {
        Point result;

        JPanel jpanel;
        NodePanel panel = null;
        Point p;
        if ((jpanel = (JPanel) getComponent(n)) != null 
                && jpanel instanceof NodePanel)
            panel = (NodePanel) jpanel;
        if (panel != null)
            p = panel.getLocationForLine(src);
        else if (jpanel != null)
            p = new Point(
                    jpanel.getLocation().x + jpanel.getWidth() / 2,
                    jpanel.getLocation().y + jpanel.getHeight() / 2);
        else
            p = new Point(0, 0);
        return p;
    }
        
    
    protected void placeNewNode(Component c)
    {
        Point initial;
        
        GraphNode n = getNode(c);
        if (n != null && n instanceof GraphEdge)
        {
            GraphNode f = ((GraphEdge)n).getFrom();
            GraphNode t = ((GraphEdge)n).getTo();
            initial = new Point(
                    Math.abs((getLocationForLine(f, t).x + getLocationForLine(t, f).x) / 2), 
                    Math.abs((getLocationForLine(f, t).y + getLocationForLine(t, f).y) / 2));
        } else if (n != null && !isMoveable(c)) {
            initial = new Point(0, 0);
        } else {
            initial = new Point(curcol * H_SPACE + H_MARGIN, currow * V_SPACE + V_MARGIN);
            if (columns > 0 && curcol >= columns - 1)
            {
                // row limit NYI
                currow++;
                curcol = 0;
            } else
                curcol++;
        }
        placeNewNode(c, initial);
    }
     
    protected Point findClearSpaceAlongLine(Component c, Point p1, Point p2, int spacing)
    {
        Point p, solution;
        int count;
        
        Dimension d = c.getPreferredSize();
        
        if (p1.x < H_MARGIN)
            p1.x = H_MARGIN;
        if (p1.y < V_MARGIN)
            p1.y = V_MARGIN;
        if (p2.x < H_MARGIN)
            p2.x = H_MARGIN;
        if (p2.y < V_MARGIN)
            p2.y = V_MARGIN;
        
        int temp = (p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y) * (p1.y - p2.y);
        int checks = (int) (Math.sqrt(temp) / spacing);
        if (checks < 1)
            checks = 1;
        int ychange = - (p1.y - p2.y) / checks;
        int xchange = - (p1.x - p2.x) / checks;
        p = new Point(p1.x, p1.y);
        for (count = 0; count < checks; count++)
        {
            if (!findConflict(c, new Rectangle(p.x, p.y, d.width, d.height)))
            {
                return p;
            }
            p = new Point(p.x + xchange, p.y + ychange);
        }
        return null;
    }
    
    // checks box around point
    protected Point findClearSpaceThorough(Component c, Point initial, int out)
    {
        Rectangle search = new Rectangle(initial.x - out * H_SPACE2, 
                initial.y - out * V_SPACE2, 2 * out * H_SPACE2, 2 * out * V_SPACE2);
        Point p = findClearSpaceAlongLine(c, new Point(search.x, search.y), 
                new Point(search.x + search.width, search.y), H_SPACE2);
        
        if (p != null)
            return p;
        p = findClearSpaceAlongLine(c, new Point(search.x + search.width, search.y), 
                new Point(search.x + search.width, search.y + search.height), V_SPACE2);
        if (p != null)
            return p;
        p = findClearSpaceAlongLine(c, new Point(search.x + search.width, search.y + search.height), 
                new Point(search.x, search.y + search.width), H_SPACE2);
        if (p != null)
            return p;
        p = findClearSpaceAlongLine(c, new Point(search.x, search.y), 
                new Point(search.x + search.width, search.y), V_SPACE2); 
        if (p != null)
            return p;
        return null;
    }
     
    // just checks 90 degree lines
    protected Point findClearSpaceQuick(Component c, Point initial, int out)
    {
        Point p;
        GraphNode n = getNode(c);
        p = new Point(initial.x + out * H_SPACE, initial.y);
        Dimension d = c.getPreferredSize();
        
        if (p.x >= 0 && p.y >= 0 && !findConflict(c, new Rectangle(p.x, p.y, d.width, d.height)))
            return p;
        if (n != null && n instanceof GraphEdge)
        {
            p = new Point(initial.x, initial.y + out * V_SPACE);
            if (p.x >= 0 && p.y >= 0 && !findConflict(c, new Rectangle(p.x, p.y, d.width, d.height)))
                return p;
        }
        p = new Point(initial.x - out * H_SPACE, initial.y);
        if (p.x >= 0 && p.y >= 0 && !findConflict(c, new Rectangle(p.x, p.y, d.width, d.height)))
            return p;
        if (n != null && n instanceof GraphEdge)
        {
            p = new Point(initial.x, initial.y - out * V_SPACE);
            if (p.x >= 0 && p.y >= 0 && !findConflict(c, new Rectangle(p.x, p.y, d.width, d.height)))
                return p;
        }
        return null;
    }
           
    
    // finds a clear space as close as possible to the given point
    protected Point findClearSpace(Component c, Point initial)
    {
        Point p = new Point(Math.max(initial.x, 0), Math.max(initial.y, 0));
        Rectangle search;
        int out = 1;
        GraphNode n = getNode(c);
        Dimension d = c.getPreferredSize();

        if (!findConflict(c, new Rectangle(p.x, p.y, d.width, d.height)))
            return p;
        while (out < 1000)  // this limit should never be reached but is there to prevent an infinite loop
        {
            if (thorough_placement || (n != null && n instanceof GraphEdge))
                p = findClearSpaceThorough(c, initial, out);
            else p = findClearSpaceQuick(c, initial, out);
            if (p != null)
                return p;
            out++;
        }
        return null;
    }
     
    protected boolean hasBeenPlaced(Component c)
    {
        if (getSaveName(c) != null && locations.containsKey(getSaveName(c)))
            return true;
        if (getNode(c) != null && !(isMoveable(c)))
            return true;
        return false;
    }
     
    protected void placeNewNode(Component c, Point initial)
    {
        Point p = new Point(initial.x, initial.y);
        
        GraphNode n = getNode(c);
        
        Dimension d = c.getPreferredSize();
        if (isMoveable(c))
        {
            p = findClearSpace(c, p);
            c.setLocation(p);
        }
        storeNodePlacement(c);
        if (p.x + d.width + H_MARGIN - origin.x > preferred.width)
            preferred = new Dimension(p.x + d.width + H_MARGIN - origin.x, preferred.height);
        if (p.y + d.height + V_MARGIN - origin.y > preferred.height)
            preferred = new Dimension(preferred.width, p.y + d.height + V_MARGIN - origin.y);
    }
    
    public void resetNodePlacing()
    {
        curcol = currow = 0;
        locations = new Hashtable();
        placeNodes();
    }
    
    protected void saveNodePlacements()
    {
        if (!willsave)
            return;
        try {
            FileOutputStream file = new FileOutputStream(savefile);
            ObjectOutput out = new ObjectOutputStream(file);
            out.writeObject(locations);
        } catch (IOException e) {
            System.err.println("Cannot open " + savefile + " for output");
            return;
        }

    }
   
    protected void loadNodePlacements()
    {
        if (!willsave)
            return;
        try {
            FileInputStream file = new FileInputStream(savefile);
            ObjectInputStream in = new ObjectInputStream(file);
            locations = (Hashtable) in.readObject();
        } catch (IOException e) {
            System.err.println("Cannot open " + savefile + " for input");
            locations = new Hashtable();
            return;
        } catch (ClassNotFoundException e2) {
            System.err.println("Somethings really screwed up...");
            return;
        }
    }
    
    public void placeNodes()
    {   
        currow = curcol = 0;
        Iterator i = nodes.values().iterator();
        layedoutonce = true;
        while (i.hasNext())
        {
            Component c = (Component) i.next();
            c.setSize(c.getPreferredSize());
        }
        
        i = nodes.keySet().iterator();
        while (i.hasNext())
        {
            GraphNode n = (GraphNode) i.next();
            if (n != null)
            {
                if (n == notanode || n instanceof GraphEdge)
                {
                    continue;
                }
                restoreNodePlacement(getComponent(n));
            }
        }
         
        i = nodes.keySet().iterator();
        while (i.hasNext())
        {
            GraphNode n = (GraphNode) i.next();
            if (n != null)
            {
                if (!(n instanceof GraphEdge))
                {
                    continue;
                }
                restoreNodePlacement(getComponent(n));
            }
        }
        saveNodePlacements();
    }
    
    public synchronized void updateCfg()
    {
        
    }
    
    public synchronized void saveCfg()
    {
        
    }
    
    protected class ComponentComparator implements Comparator
    {
        public boolean equals(Object obj)
        {
            return (obj instanceof ComponentComparator);
        }
        
        public int compare(Object obj1, Object obj2)
        {
            if (obj1.equals(obj2))
                return 0;
            if (obj1 instanceof Component && obj2 instanceof Component)
                return ((Component) obj1).getName().compareTo(((Component) obj2).getName());
            else
                throw new ClassCastException();
        }
    }
 
}
