/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/************************************************************
 * GraphEdge.java
 ************************************************************/
 
package utilities;

/* Global imports */
import java.awt.*;
import java.io.*;
import java.util.*;
import javax.swing.*;

/**
 * GraphEdge
 * <P>
 * This is a superclass for the generic purpose of drawing
 *  edges in a graph.  It works with GraphNode.
 *  This is a generalization of the taems Interrelationship.
 */
public class GraphEdge extends GraphNode implements Serializable, Cloneable
{
  protected GraphNode from = null;
  protected GraphNode to = null;
  protected boolean drawarrows = true;
  
  
  /**
   * Default constructor
   * @param l The edge's label
   * @param nodeFrom The node the edge originates from (may be null)
   * @param nodeTo The node the edge leads to (may be null)
   */
  public GraphEdge(String l, GraphNode nodeFrom, GraphNode nodeTo)
  {
      super(l);
      setDrawArrows(true);
      setFrom(nodeFrom);
      setTo(nodeTo);
  }
  
  /**
   * labelless constructor
   * @param nodeFrom The node the edge originates from (may be null)
   * @param nodeTo The node the edge leads to (may be null)
   */
  public GraphEdge(GraphNode nodeFrom, GraphNode nodeTo)
  {
      this(null, nodeFrom, nodeTo);
  }
  
  /**
   * constructor that sets only the label
   * @param l the label
   */
  public GraphEdge(String l)
  {
      this(l, null, null);
  }
  
  /**
   * blank constructor
   */
  public GraphEdge()
  {
      this(null, null, null);
  }
  
  /**
   * accessor functions
   */   
  
  /**
   * attachers
   */
  public GraphNode getFrom() { return from; }
  public GraphNode getTo() { return to; }

  public void setFrom(GraphNode n)
  {
      if (from != null)
      {
          from.removeOutEdge(this);
          removeEdge(from);
      }
      from = n;
      if (from != null)
      {
          n.addOutEdge(this);
          addEdge(from);
      }
  }
  
  public void setDrawArrows(boolean b)
  {
      drawarrows = b;
  }
  
  public boolean getDrawArrows()
  {
      return drawarrows;
  }
  
  public void setTo(GraphNode n)
  {
      if (to != null)
      {
          to.removeInEdge(this);
          removeEdge(to);
      }
      to = n;
      if (to != null)
      {
          to.addInEdge(this);
          addEdge(to);
      }
  }
  
  protected void setEndpoints (GraphNode f, GraphNode t)
  {
      if ((f instanceof GraphEdge) || (t instanceof GraphEdge))
      {
          System.err.println("Error: Cannot attach a graph edge to another graph edge");
          return;
      }
      setFrom(f);
      setTo(t);
  }
  /**
   * Detaches the node from the structure its in.
   */
  public void excise()
  {
      super.excise();
      setEndpoints(null, null);
  }
  /**
   * Determines if an object matches this one.
   * @see GraphNode#matches
   */
  public boolean matches(Object o)
  {
      if (o == null)
          return true;
      if (o.getClass().isInstance(this))
          return super.equals(o);
      return false;
  }
  
  /**
   * Clone
   */
  public Object clone()
  {
      GraphEdge cloned = new GraphEdge(getLabel(), getTo(), getFrom());
      
      return cloned;
  }


/*****************************************
 *            Drawing stuff              *
 *****************************************/

  public JPanel getDefaultPanel()
  {
      return new EdgeNodePanel(this, getLabel());
  }
  
}
  
