/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/************************************************************
 * Graph.java
 ************************************************************/

package utilities;

/* Global imports */
import javax.swing.*;
import java.beans.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.net.*;

import utilities.cfg.*;

/**
 * This class encapsulates the display of a graph, using GraphNodes,
 *  GraphEdges, and derived classes.
 * It is <i>not</i> an efficient class; there is redundant data storage,
 *  things like that, that made certain other things easier.  However,
 *  any speed loss due to this is negligable compared to the speed loss
 *  from swing.  We're using java, so what does it matter, anyway?
 */
public class Graph extends JSplitPane implements Serializable, Cloneable,
                                                 ConfiguredObject
{
  protected Vector nodes = new Vector();
  protected String label = "";
  protected Hashtable locations = new Hashtable();
  private int sequence = -1;
  protected GraphLayers layers;
  public static Graph globalhack;
  
  /**
   * default constructor
   * @param l The graph's label
   */
  public Graph(String l)
  {
      label = l;
      globalhack = this;
      initGraphics();
  }
  /**
   * blank constructor
   */
  public Graph()
  {
      this("");
  }
   
  /**
   * accessors
   */
  public String getLabel() { return label; }
  public void setLabel(String l) 
  { 
      label = l;
  }
  
  public GraphLayers getLayers()
  {
      return layers;
  }
  
  String default_layer = "default";
  
  public void setDefaultLayer(String s)
  {
      if (s != null)
          default_layer = s;
  }
     
  public void addNode(GraphNode n)
  {
      JPanel p = n.getDefaultPanel();
      if (p instanceof NodePanel)
      {
          ((NodePanel)p).setLabel(n.getLabel());
      }
      addNode(n, p, default_layer);
  }
      
  /**
   * Adds a node to the graph
   */
  public void addNode(GraphNode n, JPanel p, String layer)
  {
      if (!nodes.contains(n))
      {
          CfgManager manager = CfgManager.getManager(this);
          if (manager != null)
          {
              manager.addConfiguredObject(n, manager.getObjectKey(this) + 
                      "/nodes/" + n.getLabel());
          }
          nodes.addElement(n);
          GraphNodeFinderEnumeration.invalidateCache();
      }
      layers.addToLayer(layer, n, p);
      //display.add(p, m);
      display.invalidate();
  }
      
  public void addTree(GraphNode n)
  {
      Enumeration e = new GraphNodeFinderEnumeration(n, null);
      while (e.hasMoreElements())
      {
          GraphNode m = (GraphNode) e.nextElement();
          addNode(m);
      }
  }
 
  /**
   * Removes a node from the graph
   */
  public void removeNode(GraphNode n)
  {
      layers.eliminate(n);
      //display.remove(n.getPanel()); 
      nodes.removeElement(n);
      GraphNodeFinderEnumeration.invalidateCache();
      display.invalidate();
  }
  
  /**
   * Removes a node and excises it from its surrounding connections
   * @see GraphNode#excise
   */
  public void exciseNode(GraphNode n)
  {
      removeNode(n);
      n.excise();
  }
  
  /**
   * returns an enumeration of the nodes contained in the graph
   * note that this will only return the high level nodes, use
   * getAllNodes to return everything in the graph.
   */
  public Enumeration getNodes()
  {
      return nodes.elements();
  }
  
  /**
   * Returns all nodes in the graph
   */
  public Enumeration getAllNodes()
  {
      // changed to find ALL nodes, not just GraphNodes
      //return findNodes(new GraphNode());
      return findNodes(null);
  }
  
  /**
   * returns an enumeration of nodes matching n
   */
  public Enumeration findNodes(GraphNode n)
  {
      // use the cache system for the most common special case
      if (n == null)
      {
          if (!GraphNodeFinderEnumeration.checkCache(sequence))
              sequence = GraphNodeFinderEnumeration.cacheGraph(getNodes(), null);
          return new GraphNodeFinderEnumeration(getNodes(), null, sequence);
      } else
          return new GraphNodeFinderEnumeration(getNodes(), n);
  }
  
  /**
   * returns a node matching n
   */
  public GraphNode findNode(GraphNode n)
  {
      Enumeration e = findNodes(n);
      
      if (e.hasMoreElements())
          return (GraphNode) e.nextElement();
      else
          return null;
  }
  
  /**
   * merges a Graph g into this one
   */
  public void mergeGraph(Graph g)
  {
      Enumeration e = g.getNodes();
      while (e.hasMoreElements())
      {
          mergeNode((GraphNode) e.nextElement());
      }
  }
  
  /**
   * unmerges a graph based on g from the graph
   */
  public void unmergeGraph(Graph g)
  {
      Enumeration e = g.getNodes();
      while (e.hasMoreElements())
      {
          unmergeNode((GraphNode) e.nextElement());
      }
  }
  
  /**
   * merges in one node
   */
  protected void mergeNode(GraphNode n)
  {
      nodes.addElement(n);
  }
  
  /**
   * removes one node
   */
  protected void unmergeNode(GraphNode n)
  {
      nodes.removeElement(n);
  }
  
  /**
   * clones the object
   */
  public Object clone()
  {
      Graph cloned = null;
      
      try {
          cloned = (Graph) super.clone();
      } catch (Exception e) {
          System.out.println("Clone Error: " + e);
      }
      
      if (label != null)
          cloned.setLabel(new String(label));
      else
          cloned.setLabel("");
      
      Enumeration e = getNodes();
      cloned.nodes = new Vector();
      while (e.hasMoreElements())
      {
          cloned.addNode((GraphNode) ((GraphNode) e.nextElement()).clone());
      }
      
      return cloned;
  }
  
  public String toString()
  {
      return label;
  }
  
  /**************************************************
   *                Drawing Stuff                   *
   **************************************************/
   
  protected static final int H_SPACE = 50;
  protected static final int V_SPACE = 50;
  protected static final int H_SPACE2 = 50;
  protected static final int V_SPACE2 = 50;
  protected static final int H_MARGIN = 50;
  protected static final int V_MARGIN = 50;
  
  protected JTabbedPane display;
  protected JTextArea text = new JTextArea();
  GraphNode selected = null;
  boolean reset = false;
  boolean load = true;
  // this shouldn't exist for long than one month
  protected boolean dolayersdisplayhack = true;
  
  protected JPanel createCanvas(String name)
  {
      JPanel temp = new GraphCanvas(name);
      temp.setLayout(new GraphLayout(name, text));
      return temp;
  }
  protected void dldh(boolean b)
  {
      dolayersdisplayhack = b;
  }
  
  public void addLayer(String s)
  {
      layers.addLayer(s);
  }
  
  public void addToLayer(String layer, GraphNode n, JPanel p)
  {
      layers.addToLayer(layer, n, p);
  }
  
  protected void initGraphics()
  {
      display = new JTabbedPane();
      layers = new GraphLayers(display);
      //layers.addLayer(default_layer);
      JScrollPane sp;
      setOrientation(HORIZONTAL_SPLIT);
      setLabel(getLabel());
      setRightComponent(sp = new JScrollPane());
      sp.setMinimumSize(new Dimension(0, 0));
      sp.setViewportView(display);
      JPanel p = new JPanel();
      sp = new JScrollPane();
      sp.setMinimumSize(new Dimension(0, 0));
      //text = ((GraphLayout)(display.getLayout())).getText(); 
      //text = new JTextArea();
      sp.setViewportView(text);
      sp.setPreferredSize(new Dimension(200, sp.getHeight()));
      p.add(sp);
      if (dolayersdisplayhack)
      {
          p.setLayout(new GridLayout(3, 1));
          try {
              String http = "http://mas.cs.umass.edu/images/masl.gif";
              if (System.getProperty("MASL") != null) 
	              http = System.getProperty("MASL");
	          JPanel r = new JPanel();
              JPanel plugh = new JPanel();
              JImageComponent s = new JImageComponent(new URL(http));
              s.setMinimumSize(new Dimension(0, 0));
              plugh.setLayout(new GridLayout(1, 1));
              plugh.add(s);
              r.add(plugh);
              r.setPreferredSize(s.getPreferredSize());
              p.add(r);
          } catch (MalformedURLException e) { 
              System.err.println("Error finding image: " + e); 
          }
          JPanel q = new JPanel();
          p.add(q);
          layers.setDisplay(display);
          setLeftComponent(p);
      } else {
          setLeftComponent(sp);
      }

  }
  
  public void addMenuItem(JMenuItem item)
  {
      Enumeration e = getLayers().getLayers();
      while (e.hasMoreElements())
      {
          String s = (String) e.nextElement();
          GraphLayer g = getLayers().getLayer(s);
          JPanel p = g.getPanel();
          if (p.getLayout() instanceof GraphLayout)
              ((GraphLayout)(p.getLayout())).addMenuItem(item);
      }
  }
  
  public void updateCfg()
  {
      
  }
  
  public void saveCfg()
  {
      
  }
  
  /**
   * This class encapsulates the placement of nodes and the mouse handling
   */
  public class GraphCanvas extends JPanel
  {
    private transient Image offscreen;
    Dimension dim = getMinimumSize();
    public GraphNode selectedinternal;
    private boolean thorough_placement = false;
    String canvas_name = null;
    
    public GraphCanvas(String n)
    {
        super();
        canvas_name = n;
    }
    
    public GraphCanvas()
    {
        this(null);
    }
    
    public boolean isOptimizedDrawingEnabled()
    {
        return false;
    }
    
    public String toString()
    {
        return new String("GraphCanvas '" + canvas_name + "'");
    }
    
    public void paint(Graphics g) 
    {
        // got to repaint the whole thing so that all of the lines are drawn
        RepaintManager.currentManager(this).markCompletelyDirty(this);
        super.paint(g);
        RepaintManager.currentManager(this).markCompletelyClean(this);
    }
    
    public void paintEdge(Graphics g, GraphEdge edge)
    {
        if (!layers.isObjectVisible(edge))
            return;
        int np = 0;
        float []px = new float[5];
        float []py = new float[5];

        g.setColor(Color.black);
        GraphNode from, to;
        NodePanel frompanel = null, topanel = null, edgepanel = null;
        
        JPanel t;
        
        from = edge.getFrom();
        to = edge.getTo();
        if ((t = layers.getTopPanel(from)) != null && t instanceof NodePanel)
            frompanel = (NodePanel) t;
        if ((t = layers.getTopPanel(to)) != null && t instanceof NodePanel)
            topanel = (NodePanel) t;
        if ((t = layers.getTopPanel(edge)) != null && t instanceof NodePanel)
            edgepanel = (NodePanel) t;

        if (frompanel != null && layers.isObjectVisible(from))
        {
            Point tmp;
            if (topanel != null && layers.isObjectVisible(to))
            {
                tmp = frompanel.getLocationForLine(to);
                px[np] = tmp.x;
                py[np] = tmp.y;
                np++;
                if (!tmp.equals(frompanel.getLocationForLine()))
                {
                    g.setColor(Color.black);
                    g.fillRoundRect(tmp.x - 4, tmp.y - 4, 8, 8, 8, 8);
                }
            }
        }

        px[np] = edgepanel.getLocationForLine().x;
        py[np] = edgepanel.getLocationForLine().y;
        np++;

        g.setColor(Color.black);

        if (topanel != null && layers.isObjectVisible(to))
        {
            Point tmp;
            if (frompanel != null && layers.isObjectVisible(from))
            {
                tmp = topanel.getLocationForLine(from);
                px[np] = tmp.x;
                py[np] = tmp.y;
                np++;
                if (!tmp.equals(topanel.getLocationForLine()))
                {
                    g.setColor(Color.black);
                    g.fillRoundRect(tmp.x - 2, tmp.y - 2, 4, 4, 4, 4);
                }
            }
        }
        if (np <= 1)
            return;

        Spline spline = new Spline(px, py, np);
        spline.Generate();
        spline.draw(g);
        if (edge.getDrawArrows())
        {
            Point p1 = spline.getPoint((float)0.15);
            Point p2 = spline.getPoint((float)0.25);
            Arrow.drawArrowhead(g, p1.x, p1.y, p2.x, p2.y);
            p1 = spline.getPoint((float)0.75);
            p2 = spline.getPoint((float)0.85);
            Arrow.drawArrowhead(g, p1.x, p1.y, p2.x, p2.y);
        }
    }  

    public void paintLines(Graphics g)
    {
        g.setColor(Color.black);
        Enumeration e = getAllNodes();
        while (e.hasMoreElements())
        {
            GraphNode n = (GraphNode) e.nextElement();
            JPanel t;
            NodePanel npanel = null;
            if ((t = layers.getTopPanel(n)) != null && t instanceof NodePanel)
                npanel = (NodePanel) t;
            if (npanel == null || !layers.isObjectVisible(n))
                continue;
            if (n instanceof GraphEdge)
            {
                paintEdge(g, (GraphEdge) n);
                continue;
            }
                
            Enumeration f = n.getUndirEdges();
            // yeah, I know, this will paint all the lines twice...
            while (f.hasMoreElements())
            {
                GraphNode node = (GraphNode) f.nextElement();
                NodePanel nodepanel = null;
                if ((t = layers.getTopPanel(node)) != null && t instanceof
                        NodePanel)
                    nodepanel = (NodePanel) t;
                if (nodepanel == null || node instanceof GraphEdge || 
                        !layers.isObjectVisible(node))
                    continue;
                Point tmp = npanel.getLocationForLine(node);
                g.drawLine(nodepanel.getLocationForLine().x, 
                        nodepanel.getLocationForLine().y, tmp.x, tmp.y);
                if (!tmp.equals(npanel.getLocationForLine()))
                {
                    g.setColor(Color.black);
                    g.fillRoundRect(tmp.x - 2, tmp.y - 2, 4, 4, 4, 4);
                }
            }
        }
    }

    public void paintComponent(Graphics g)
    {
        super.paintComponent(g);
        /*
        Enumeration e = getAllNodes();
        while (e.hasMoreElements())
        {
            GraphNode n = (GraphNode) e.nextElement();
            if (layers.isObjectVisible(n))
                layers.getTopPanel(n).paintComponent(g);
        }
        */
        Enumeration e = getAllNodes();
        while (e.hasMoreElements())
        {
            GraphNode n = (GraphNode) e.nextElement();
            JPanel p = layers.getTopPanel(n);
            if (p instanceof NodePanel)
                ((NodePanel) p).paintBackImage(g);
            //if (n instanceof ContainerGraphNode && layers.isObjectVisible(n))
            //{
            //    ((ContainerGraphNode)n).paintBackImage(g);
            //}
        }
        paintLines(g);
    }
    
  }
  
}
        
