/************************************************************
 * File: Distribution.java
 ************************************************************/

package utilities;

import java.util.*;
import java.io.*;

// A distribution is simply a vector of floats with a special output
//   function
public class Distribution extends Vector implements Serializable {
  
  /**
   * Makes an empty distribution
   */
  public Distribution() {
    super();
  }

  /**
   * Makes a distribution with one entry
   */
  public Distribution(Float Value, Float Distribution) {
    this();

    appendTerm(Value, Distribution);
  }

  /**
   * Makes a distribution with one entry
   */
  public Distribution(float Value, float Distribution) {
    this(new Float(Value), new Float(Distribution));
  }

  /**
   * Makes a distribution with one entry
   */
  public Distribution(double Value, double Distribution) {
    this((float)Value, (float)Distribution);
  }

  /**
   * Parses a string of the form "5 0.2 10 0.5 20 0.3" into
   * a distribution object.  Will also trip leading
   * and trailing parentheses if present.
   * <P>
   * This will crash if you feed it an incorrectly formatted
   * string.
   * @param str The string representation of the distribution
   */
  public Distribution(String str) {

      str = str.trim();
      if (str.startsWith("(")) {
          str = str.substring(1, str.length());
      }
      if (str.endsWith(")")) {
          str = str.substring(0, str.length()-1);
      }
      str = str.trim();

      StringTokenizer tok = new StringTokenizer(str);
      while (tok.hasMoreElements()) {
          Float v = Float.valueOf(tok.nextToken());
          Float d = Float.valueOf(tok.nextToken());
          appendTerm(v, d);
      }
  }

  /**
   * getArray method. This method is used by the simulator to get the 
   * distribution in a float array. This method is called in the 
   * simulator.ExecuteEvent constructor.
   * @return float [] array
   * @see  ExecuteEvent#ExecuteEvent(int, int, Agent, String, Distribution, Distribution, Distribution) 
   */
  public float[] getArray()
  {
    float[] distribution = new float[this.size()];
    int counter=0;
    Enumeration e;
    e = elements();
    while (e.hasMoreElements()) {
      distribution[counter+1]=((Float)e.nextElement()).floatValue();
      distribution[counter]=((Float)e.nextElement()).floatValue();
      counter = counter+2;
    }
    return(distribution);
  }


  /**
   * getHashtable, converts the Distribution is an hashtable
   * where the pair is for as <value, frequency>
   * @return a hashtable
   */
  public Hashtable getHashtable() {
    Hashtable ht= new Hashtable();
    Enumeration e = elements();
    while (e.hasMoreElements()) 
      ht.put((Float)e.nextElement(), (Float)e.nextElement());
    return(ht);
  }

  /**
   * Searches for the supplied value in the distribution
   * @param v The value to look for
   * @return The value's probability (0 if not found)
   */
  public float findValue(float v)
  {
    Enumeration e;
    e = elements();
    float value, prob;

    while (e.hasMoreElements()) {
      value = ((Float)e.nextElement()).floatValue();
      prob = ((Float)e.nextElement()).floatValue();

      if (value == v) return prob;
    }

    return (float)0;
  }

  /**
   * Searches for the closest match to the supplied value
   * in the distribution.  In case of a tie, the element
   * with the greater probability is chosen.
   * @param v The value to look for
   * @return The value
   */
  public float findClosestValue(float v) {
    Enumeration e;
    e = elements();
    float closestvalue, closestprob; 
    float value, prob;

    closestvalue = Float.POSITIVE_INFINITY;
    closestprob = Float.NEGATIVE_INFINITY; 
    while (e.hasMoreElements()) {
      value = ((Float)e.nextElement()).floatValue();
      prob = ((Float)e.nextElement()).floatValue();

      if (Math.abs(value - v) < Math.abs(closestvalue - v)) {
        closestvalue = value;
        closestprob = prob;

      } else if (Math.abs(value - v) == Math.abs(closestvalue - v)) {
        if (prob > closestprob) {
          closestvalue = value;
          closestprob = prob;
        }
      }
    }

    return (float)closestvalue;
  }

  /**
   * Determines if the value is contained in the Distribution
   * @param v The value to look for
   * @param t The threshold (+/-) around each value to be considered a match
   * @return True, if the value is contained
   */
  public boolean containsValue(float v, float t)
  {
    Enumeration e;
    e = elements();
    float value, prob;

    while (e.hasMoreElements()) {
      value = ((Float)e.nextElement()).floatValue();

      if (((value - t) <= v) && ((value + t) >= v)) return true;
    }

    return false;
  }

  /**
   * calculateMin() method computes the minimal value, you can get from
   * a distribution instance. Useful for the diagnostic component.
   */
  public float calculateMin()
  {
    Enumeration e;
    e = elements();
    float min = ((Float)elementAt(0)).floatValue();
    float cur;

    while (e.hasMoreElements()) {
      cur = ((Float)e.nextElement()).floatValue();
      if (min > cur)
	min = cur;
      cur = ((Float)e.nextElement()).floatValue();
    }
    return min;
  }

  /**
   * calculateMax() method computes the maximal value from a Distribution
   * instance.
   */
  public float calculateMax()
  {
    Enumeration e;
    e = elements();
    float max = ((Float)elementAt(0)).floatValue();
    float cur;

    while (e.hasMoreElements()) {
      cur = ((Float)e.nextElement()).floatValue();
      if (max < cur)
	max = cur;
      cur = ((Float)e.nextElement()).floatValue();
    }
    return max;
  }

  /**
   * Compute the average of a given Distribution
   */
  public float calculateAvg()
  {
    Enumeration e;
    e = elements();
    float avg = (float)0.0;
    float value, prob;

    while (e.hasMoreElements()) {
      value = ((Float)e.nextElement()).floatValue();
      prob = ((Float)e.nextElement()).floatValue();
      avg += value * prob;
    }
    return avg;
  }

  /**
   * Compute the most likely outcome of a given Distribution
   */
  public float calculateMostLikely()
  {
    Enumeration e;
    e = elements();
    float mvalue = 0, mprob = 0;

    while (e.hasMoreElements()) {
      float value = ((Float)e.nextElement()).floatValue();
      float prob = ((Float)e.nextElement()).floatValue();
      if (prob > mprob) {
	mprob = prob;
	mvalue = value;
      }
    }

    return mvalue;
  }

  /**
   * Standart deviation : Sum(f X^2)/ Sum(f) - X_bar^2
   * Easy computation....
   */
  public float calculateDeviation() {
    float mean = this.calculateAvg();
    Enumeration e = elements();
    float value, prob, result, somme = 0, somme_prob = 0;

    while (e.hasMoreElements()) {
      value = ((Float)e.nextElement()).floatValue();
      prob = ((Float)e.nextElement()).floatValue();
      somme = somme + (prob * (value * value));
      somme_prob = prob + somme_prob;
    }
    
    result = (float) Math.sqrt((somme / somme_prob) - (mean * mean));
    return(result);
  }

  /** 
   * Calculate the total density represented by the distribution.
   * @return the total density
   * Example:<BR>
   * [0.0, 0.1, 1.0, 0.2, 10.0, 0.3] = total density 0.6 <BR>
   */
  public float calculateTotalDensity() {
    float prob = 0;
    
    for (int i = 0; i < size(); i++) {
      i++;
      prob += ((Float)elementAt(i)).floatValue();
    }

    return prob;
  }

  /** 
   * Apply a density weight to a Distribution.  Changes
   * are made to a clone, which is returned, the original is
   * left unchanged.
   * @param density
   * Example:<BR>
   * Dc = [0.0, 1.0] Dd = [1.0, 0.5] Dq = [10.0, 0.5] <BR>
   * Applying a density of 0.5 give this result: <BR>
   * Dc = [0.0, 0.5] Dd = [1.0, 0.25] Dq = [10.0, 0.25] <BR>
   */
  public Distribution applyDensity(float density) {
    Distribution d = (Distribution)this.clone();
    float value, prob;
    
    for (int i = 0; i < d.size(); i++) {
      i++;
      prob = ((Float)d.elementAt(i)).floatValue();
      d.setElementAt(new Float(prob * density), i);
    }

    return(d);
  }

  /** 
   * Apply a power to the value of a Distribution.  Changes
   * are made to a clone, which is returned, the original is
   * left unchanged.
   * @param power
   * Example:<BR>
   * Dc = [0.0, 1.0] Dd = [1.0, 0.5] Dq = [10.0, 0.5] <BR>
   * Applying a power of 0.5 give this result: <BR>
   * Dc = [0.0, 1.0] Dd = [0.5, 0.5] Dq = [5.0, 0.5] <BR>
   */
  public Distribution applyPower(float power) {
    Distribution d = (Distribution)this.clone();
    float value, prob;
    
    for (int i = 0; i < d.size(); i++) {
      value = ((Float)d.elementAt(i)).floatValue();
      d.setElementAt(new Float(value * power), i);
      i++;
    }

    return(d);
  }

  /** 
   * Apply a offset to the value of a Distribution.  Changes
   * are made to a clone, which is returned, the original is
   * left unchanged.
   * @param offset
   * Example:<BR>
   * Dc = [0.0, 1.0] Dd = [1.0, 0.5] Dq = [10.0, 0.5] <BR>
   * Applying a power of 2.5 give this result: <BR>
   * Dc = [2.5, 1.0] Dd = [3.5, 0.5] Dq = [12.5, 0.5] <BR>
   */
  public Distribution applyOffset(float offset) {
    Distribution d = (Distribution)this.clone();
    float value, prob;
    
    for (int i = 0; i < d.size(); i++) {
      value = ((Float)d.elementAt(i)).floatValue();
      d.setElementAt(new Float(value + offset), i);
      i++;
    }

    return(d);
  }

  /**
   * Appends a new term onto the distribution, no normalization
   */
  public void appendTerm(Float v, Float d) {
    this.addElement(v);
    this.addElement(d);
  }

  /**
   * Appends a new term onto the distribution, no normalization
   */
  public void appendTerm(float v, float d) {
    this.addElement(new Float(v));
    this.addElement(new Float(d));
  }

  /**
   * Appends a new term onto the distribution, no normalization
   */
  public void appendTerm(double v, double d) {
    this.addElement(new Float(v));
    this.addElement(new Float(d));
  }

  /**
   * Appends a distribution to another one, no normalization
   */
  public Distribution appendDistribution(Distribution d) {
    Distribution d1 = (Distribution)this.clone();
    Enumeration e = d.elements();
    Float value, prob;

    while (e.hasMoreElements()) {
      value = (Float)e.nextElement();
      prob = (Float)e.nextElement();
      d1.insertElementAt(prob, 0);
      d1.insertElementAt(value, 0);
    }

    return(d1);
  }

  /**
   * Computes the overall joint probability for a vector of
   * distributions.  Values are summed and compacted.
   */
  public static Distribution computeJointDistribution(Vector v) {
    Distribution newd = new Distribution();

    subComputeJointDistribution(0, 1, v, 0, newd);
    newd.compact();

    return(newd);
  }

  public static Distribution computeJointDistribution(Distribution d1, Distribution d2) {
    Vector v = new Vector();
    v.addElement(d1);
    v.addElement(d2);
    return computeJointDistribution(v);
  }

  /**
   * Recursive part of cJD
   */
  private static void subComputeJointDistribution(float v, float p, Vector dists, int index, Distribution output) {
    float nv, np;

    if (index < dists.size()) {
      Distribution d = (Distribution)dists.elementAt(index);
      Enumeration e = d.elements();
      while (e.hasMoreElements()) {
        nv = ((Float)e.nextElement()).floatValue();
        np = ((Float)e.nextElement()).floatValue();
        subComputeJointDistribution(v + nv, p * np, dists, index + 1, output);
      }

    } else {
      output.appendTerm(new Float(v), new Float(p));
    }
  }


  /**
   * Computes the overall joint probability for a vector of
   * distributions.  Values are differenced and compacted.
   */
  public static Distribution computeDifferenceJointDistribution(Vector v) {
    Distribution newd = new Distribution();

    subComputeDifferenceJointDistribution(0, 1, v, 0, newd);
    newd.compact();

    return(newd);
  }

  public static Distribution computeDifferenceJointDistribution(Distribution d1, Distribution d2) {
    Vector v = new Vector();
    v.addElement(d1);
    v.addElement(d2);
    return computeDifferenceJointDistribution(v);
  }

  /**
   * Recursive part of cJD
   */
  private static void subComputeDifferenceJointDistribution(float v, float p, Vector dists, int index, Distribution output) {
    float nv, np;
    float v1,p1;
    if (index < dists.size()) {
      Distribution d = (Distribution)dists.elementAt(index);
      Enumeration e = d.elements();
      while (e.hasMoreElements()) {
        nv = ((Float)e.nextElement()).floatValue();
        np = ((Float)e.nextElement()).floatValue();
	if (v == 0.0) {
	    v1 = nv;
	    p1 = np;
	}
	else {
	    v1 = v - nv;
	    p1 = p * np;
	}
        subComputeDifferenceJointDistribution(v1, p1, dists, index + 1, output);
      }

    } else {
      output.appendTerm(new Float(v), new Float(p));
    }
  }


  /**
   * Computes the overall joint probability for a vector of
   * distributions.  Values are maxed and compacted.
   */
  public static Distribution computeMaxJointDistribution(Vector v) {
    Distribution newd = new Distribution();

    subComputeMaxJointDistribution(0, 1, v, 0, newd);
    newd.compact();

    return(newd);
  }

  public static Distribution computeMaxJointDistribution(Distribution d1, Distribution d2) {
    Vector v = new Vector();
    v.addElement(d1);
    v.addElement(d2);
    return computeMaxJointDistribution(v);
  }

  /**
   * Recursive part of cMJD
   */
  private static void subComputeMaxJointDistribution(float v, float p, Vector dists, int index, Distribution output) {
    float nv, np, fv;

    if (index < dists.size()) {
      Distribution d = (Distribution)dists.elementAt(index);
      Enumeration e = d.elements();
      while (e.hasMoreElements()) {
        nv = ((Float)e.nextElement()).floatValue();
        np = ((Float)e.nextElement()).floatValue();
	if (v > nv) { fv = v; } else { fv = nv; }
        subComputeMaxJointDistribution(fv, p * np, dists, index + 1, output);
      }

    } else {
      output.appendTerm(new Float(v), new Float(p));
      output.compact();
    }
  }

  /**
   * Computes the overall joint probability for a vector of
   * distributions.  Values are maxed and compacted.
   */
  public static Distribution computeMinJointDistribution(Vector v) {
    Distribution newd = new Distribution();

    subComputeMinJointDistribution(Integer.MAX_VALUE, 1, v, 0, newd);
    newd.compact();

    return(newd);
  }

  public static Distribution computeMinJointDistribution(Distribution d1, Distribution d2) {
    Vector v = new Vector();
    v.addElement(d1);
    v.addElement(d2);
    return computeMinJointDistribution(v);
  }

  /**
   * Recursive part of cMJD
   */
  private static void subComputeMinJointDistribution(float v, float p, Vector dists, int index, Distribution output) {
    float nv, np, fv;
    if (index < dists.size()) {
      Distribution d = (Distribution)dists.elementAt(index);
      Enumeration e = d.elements();
      while (e.hasMoreElements()) {
        nv = ((Float)e.nextElement()).floatValue();
        np = ((Float)e.nextElement()).floatValue();
	if (v < nv) { 
	    fv = v; } 
	else {
	    fv = nv; }
        subComputeMinJointDistribution(fv, p * np, dists, index + 1, output);
      }

    } else {
      output.appendTerm(new Float(v), new Float(p));
      output.compact();
    }
  }

  /**
   * Compacts the distribution by combining like valued terms
   */
  public void compact() {

    for (int i = 0; i < size(); i += 2) {
      Float value = (Float)elementAt(i);
      Float prob = (Float)elementAt(i + 1);

      for (int j = size() - 2; j > i; j -= 2) {
        Float v = (Float)elementAt(j);

        if (value.equals(v)) {
          Float p = (Float)elementAt(j + 1);
          prob = new Float(prob.floatValue() + p.floatValue());
          removeElementAt(j + 1);
          removeElementAt(j);
        }
      }

      setElementAt(prob, i + 1);
    }
  }


  /**
   * Normalizes the probabilities to 1
   */
  public void normalize() {
    Enumeration e = elements();
    float totalprob = 0;
    Float prob;

    // Get total
    while (e.hasMoreElements()) {
      e.nextElement();
      totalprob += ((Float)e.nextElement()).floatValue();
    }

    // Normalize em
    for (int i = 1; i < size(); i += 2) {
       prob = (Float)elementAt(i);
       setElementAt(new Float(prob.floatValue() / totalprob), i);
     }
  }
 

  /**
   * Clusters the distribution by combining terms to reach 
   * at most a given number of points
   */
  public void cluster(int pointsNumber) {

    compact();

    Distribution original = (Distribution)this.clone();

    if (pointsNumber >= size()/2 )
    {
        // do nothing, distribution is the right size.
	return;
    }

    // Clearing up current distribution : clustered distrib will be put in there.
    removeAllElements();

    if (pointsNumber == 1)
    {
	appendTerm(new Float(original.calculateMostLikely()), new Float(1));
    }
    else if (pointsNumber == 2)
    {
	float minVal = original.calculateMin();
	float minDis = original.findValue(original.calculateMin());
	float avgVal = original.calculateMostLikely();
	float avgDis = original.findValue(original.calculateMostLikely());
	float maxVal = original.calculateMax();
	float maxDis = original.findValue(original.calculateMax());
	
	if (minDis > maxDis) {
	    appendTerm(new Float(minVal), new Float(minDis));
	    appendTerm(new Float(avgVal), new Float(1 - minDis));
	} else {
	    appendTerm(new Float(avgVal), new Float(1 - maxDis));
	    appendTerm(new Float(maxVal), new Float(maxDis));
	}
	compact();

    }
    else
    {

	// We find the pointsNumber values with the greatest densities,
	// and then lump everything else into one of those buckets.

	// First we grab the min and the max values to prevent
	// completely skewing the distribution.

	float minValue = original.calculateMin();
	float maxValue = original.calculateMax();
	float mostLikelyValue = original.calculateMostLikely();

	appendTerm(new Float(minValue), new Float(0));
	appendTerm(new Float(maxValue), new Float(0));
	appendTerm(new Float(mostLikelyValue), new Float(0));

	// Sort by distribution in increasing order (so we can take
	// the pointsNumber - 3 last density/value pairs).

	original.sortByDistribution();

        // Put the pointsNumber - 3 values that have the largest densities into
        // our array.  Note, since we resorted by density, we have
        // to check each time to see if the item is already in the
        // result distribution

        int k = 3;
        for (int i = original.size()-2; (i > 0 && k < pointsNumber); i=i-2) {

	    // Value of pair number i is at i (value, density)
	    float val = ((Float)original.elementAt(i)).floatValue();

	    if (val != minValue &&
		val != maxValue &&
		val != mostLikelyValue)
	    {
		appendTerm(new Float(val), new Float(0));
		k++;
	    }
	}

	// This is necessary in case either the max or min is also the most likely
	compact();

        // Resort our pointsNumber best in ascending order.
	sortByValue();

	//System.err.println("Second dis (buckets) sorted : " + output());

	for (int j = 0; j < original.size(); j=j+2) {

	    float delta2 = Float.MAX_VALUE;
	    boolean found_right_bucket = false;

	    // Value and distribution of element being lumped in one of the final buckets
	    float movedVal = ((Float)original.elementAt(j)).floatValue();
	    float movedDis = ((Float)original.elementAt(j+1)).floatValue();

	    //System.err.println("Lumping : [" + movedVal + " " + movedDis + "]");

	    // Value and distribution of previous bucket examined
	    float prevBucketDis = 0;

	    // Value and distribution of current bucket being examined
	    float bucketVal = 0;
	    float bucketDis = 0;

	    for (int i = 0; (found_right_bucket == false && i < size()); i=i+2) {

		if (i > 0) prevBucketDis = ((Float)elementAt(i-1)).floatValue();
		bucketVal = ((Float)elementAt(i)).floatValue();
		bucketDis = ((Float)elementAt(i+1)).floatValue();

		float delta1 = bucketVal - movedVal;
		if (delta1 < 0) delta1 *= -1;

		if (delta1 == delta2) {

		    setElementAt(new Float(prevBucketDis + (movedDis / 2)), i-1);
		    setElementAt(new Float(bucketDis + (movedDis / 2)), i+1);
		    found_right_bucket = true;
		} else if (delta1 > delta2) {

		    setElementAt(new Float(prevBucketDis + movedDis), i-1);
		    found_right_bucket = true;
		}
		delta2 = delta1;
	    }

	    if (found_right_bucket == false) {
		bucketDis = ((Float)elementAt(size()-1)).floatValue();
		setElementAt(new Float(bucketDis + movedDis), size()-1);
	    }

	//System.err.println("Second dis (buckets) sorted : " + output());

	}
    }
}


  /**
   * Should be removed if nobody uses it. Was used in compare().
   */
  public void sort() {
    sortByValue();
  }

  /**
   * Sort the distribution by value. I'm not sure anyone will use this
   * but I need it to compare 2 Distributions
   */
  public void sortByValue() {
    Enumeration e;
    e = elements();
    Float value;
    Float frequency=new Float(0);

    for (int i = 0; i < size()-2; i=i+2) {
      int enumr = i;
      value = (Float)elementAt(enumr);
      for (int j = i; j < size(); j=j+2) {
	if (((Float)elementAt(j)).floatValue() < value.floatValue()) {
	  enumr = j;
	  value = (Float)elementAt(enumr);
	  frequency =(Float)elementAt(enumr+1);
	  }
	}
	if (i != enumr) { // swap
	  setElementAt(elementAt(i), enumr); //value
	  setElementAt(elementAt(i+1), enumr+1); //frequency
	  setElementAt(value,i); //value
	  setElementAt(frequency,i+1);
	}
    }
  }

  /**
   * Sort the distribution by frequency. I'm not sure anyone will use this
   * either but I need it for the cluster() method.
   */
  public void sortByDistribution() {
    Enumeration e;
    e = elements();
    Float value=new Float(0);
    Float frequency;

    for (int i = 0; i < size()-2; i=i+2) {
      int enumr = i;
      frequency = (Float)elementAt(enumr+1);
      for (int j = i; j < size(); j=j+2) {
	if (((Float)elementAt(j+1)).floatValue() < frequency.floatValue()) {
	  enumr = j;
	  value = (Float)elementAt(enumr);
	  frequency =(Float)elementAt(enumr+1);
	  }
	}
	if (i != enumr) { // swap
	  setElementAt(elementAt(i), enumr); //value
	  setElementAt(elementAt(i+1), enumr+1); //frequency
	  setElementAt(value,i); //value
	  setElementAt(frequency,i+1);
	}
    }
  }

  /**
   * This algorithm assumes no equal values (e.g. they're compacted)
   */
  public boolean equals(Distribution d) {
    Float value, prob, dvalue, dprob;
    boolean found;

    if (d == null)
      return false;

    if (size() != d.size())
      return false;

    Enumeration e = elements();
    while (e.hasMoreElements()) {
      value = (Float)e.nextElement();
      prob = (Float)e.nextElement();
      found = false;
      for(int i = 0; i < d.size(); i+=2) {
        dvalue = (Float)d.elementAt(i);
        dprob = (Float)d.elementAt(i+1);
        if (value.equals(dvalue) && prob.equals(dprob)) {
          found = true;
          break;
        }
      }
      if (!found) return false;
    }

    return true;
  }

  /**
   * Statistic described in our AAAI paper to compare two distributions.
   * It's based on the difference of area between the cumulative probability 
   * plots of the distributions.
   * @return a value significatif depending of the errors between two 
   * Distributions
   */
  public double compare(Distribution b) {
    return(compare(this,b));
  }

  /**
   * Statistic described in our AAAI paper to compare two distributions.
   * It's based on the difference of area between the cumulative probability 
   * plots of the distributions.
   * @return a value significatif depending of the errors between two 
   * Distributions
   */
  private double compare(Distribution a, Distribution b) {
    double accumval=0.0, distance,accsum=0.0;
    Hashtable accumhash0=new Hashtable();
    Hashtable accumhash1=new Hashtable();
    Hashtable hash0 = a.getHashtable();
    Hashtable hash1 = b.getHashtable();
    Distribution listOfAllValues = a.appendDistribution(b);
    
    listOfAllValues.compact();
    listOfAllValues.sortByValue();

    for (int i=0; i<listOfAllValues.size(); i+=2){
      if (hash0.containsKey(listOfAllValues.elementAt(i))){
	double tmpval=((Float)hash0.get(listOfAllValues.elementAt(i))).doubleValue();
	accsum += tmpval;
      }
      accumhash0.put(listOfAllValues.elementAt(i), new Double(accsum));
    }

    accsum=0.0;
    for (int i=0; i<listOfAllValues.size(); i+=2){
      if (hash1.containsKey(listOfAllValues.elementAt(i))){
	double tmpval=((Float)hash1.get(listOfAllValues.elementAt(i))).doubleValue();
	accsum += tmpval;
      }
      accumhash1.put(listOfAllValues.elementAt(i), new Double(accsum));
    }
    
    Enumeration e=accumhash0.keys();
    while (e.hasMoreElements()){
      Float tmpkey=(Float)e.nextElement();
      Double tmpval=(Double)accumhash0.get(tmpkey);
    }

    e=accumhash1.keys();
    while (e.hasMoreElements()){
      Float tmpkey=(Float)e.nextElement();
      Double tmpval=(Double)accumhash1.get(tmpkey);
    }


    double finalsum=0.0;
    for (int i=0; i<listOfAllValues.size()-2; i+=2){
      double intersum= Math.abs(((Double)accumhash0.get(listOfAllValues.elementAt(i))).doubleValue() - ((Double)accumhash1.get(listOfAllValues.elementAt(i))).doubleValue());
      double intersum2=(((Float)listOfAllValues.elementAt(i+2)).doubleValue() - ((Float)listOfAllValues.elementAt(i)).doubleValue()) * intersum;
      finalsum += intersum2;
    }

    double lastdiv=((Float)listOfAllValues.elementAt(listOfAllValues.size()-2)).doubleValue()- ((Float)listOfAllValues.firstElement()).doubleValue();
    

    distance=finalsum/lastdiv;
    return(distance);
    
  }


  /**
   * Stringify in a TTaemsy way
   */
  public String output()
  {
    // Output vector in the format: V[1] V[0] V[3] V[2] V[5] V[4] ... 
    StringBuffer sb = new StringBuffer(""); 
    
    Enumeration e;
    e = elements();
    while (e.hasMoreElements())
      {
	float n1;
	float n2;
	long n3;
	n1 = ((Float)e.nextElement()).floatValue();
	if (e.hasMoreElements())
	  n2 = ((Float)e.nextElement()).floatValue();
	else
	  return "Error";
	
	sb.append(n1);
	sb.append(" ");
	sb.append(n2);
	if (e.hasMoreElements())
	  sb.append(" ");
      }
    return sb.toString();
  }
}
