/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Converter.java
 ***************************************************************************************/

package utilities;

/* Global Inclues */
import java.io.*;
import java.util.*;

/**
 * Converts stuff to and from strings
 */
public class Converter {

    /**
     * This method converts string data to other formats,
     * as specified by their property info.  Useful when reading
     * data from the config file, or wherever.  Currently supported
     * formats are String, Long, Integer, Float, Double, Short, Boolean,
     * Vector (str1,str2,str3...), and Hashtable (key1,obj1,key2,obj2...).
     * <P>
     * For those items which are lists (e.g. Vector, Hashtable), you
     * may optionally set the object delimiter in the property.  To do
     * this, follow this example:
     * <P>
     *    <DD><TT>MyVector: DELIM=; Item 1;Item 2;Item 3;...</TT>
     * <P>
     * The default delimiter is a comma, note that the contents of
     * aggregate types (Vector, Hashtable) will be converted to Strings.
     * <P>
     * This function and its companion unTypeProperty may be useful to
     * other components doing string/data processing.
     * @param property The data to change
     * @param type The type to change it to
     * @return The typed, converted data, or null if error
     * @see #unTypeProperty
     */
    public static Object reTypeProperty(String property, String type) {
        Object newp = null;
        String delim = ",";

        if (property instanceof String) {

            // Check for new delimiter
            if (((String)property).startsWith("DELIM=")) {
                StringTokenizer tmp = new StringTokenizer(property);
                delim = tmp.nextToken();
                property = (property).substring(delim.length());
                property = (property).trim();
                delim = delim.substring(6);
            }
      
            // All the types we support
            if ((type == null) || type.equalsIgnoreCase("String")) {
                newp = property;
	
            } else if (type.equalsIgnoreCase("Long")) {
                newp = Long.valueOf(property);
	
            } else if (type.equalsIgnoreCase("Integer")) {
                newp = Integer.valueOf(property);
	
            } else if (type.equalsIgnoreCase("Float")) {
                newp = Float.valueOf(property);
	
            } else if (type.equalsIgnoreCase("Double")) {
                newp = Double.valueOf(property);
	
            } else if (type.equalsIgnoreCase("Short")) {
                newp = Short.valueOf(property);
	
            } else if (type.equalsIgnoreCase("Boolean")) {
                newp = Boolean.valueOf(property);
	
            } else if (type.equalsIgnoreCase("Distribution")) {
                StringTokenizer t = new StringTokenizer(property, delim);
                Distribution temp = new Distribution();
	
                while (t.hasMoreTokens()) {
                    try { 
                        Float v = Float.valueOf(t.nextToken());
                        Float d = Float.valueOf(t.nextToken());
                        temp.appendTerm(v, d);
                    } catch (Exception e) {
                        System.err.println(e);
                        break;
                    }
                }

                newp = temp;

            } else if (type.equalsIgnoreCase("Vector")) {
                StringTokenizer t = new StringTokenizer(property, delim);
                Vector temp = new Vector();
	
                while (t.hasMoreTokens())
                    temp.addElement(t.nextToken().trim());
	
                newp = temp;
	
            } else if (type.equalsIgnoreCase("Hashtable")) {
                StringTokenizer t = new StringTokenizer(property, delim);
                Hashtable temp = new Hashtable();
	
                while (t.hasMoreTokens()) {
                    String k, o;
	  
                    k = t.nextToken().trim();
                    if (t.hasMoreTokens())
                        o = t.nextToken().trim();
                    else
                        o = "Invalid hashtable specification";
	  
                    temp.put(k, o);
                }
	
                newp = temp;
	
            } else {
                Log.getDefault().log("Error, unsupported conversion type " + type, Log.LOG_INFO);
            }
        }
    
        return newp;
    }

    /**
     * This method returns the property type name, in string form,
     * of the supplied property
     * @param property The data to look at
     * @return The type name for property, or null if unrecognized
     */
    public static String getPropertyType(Object property) {
        if (property == null) return null;
        String type = null;

        // Convert em
        if ((property instanceof String) ||
            (property instanceof Long) ||
            (property instanceof Integer) ||
            (property instanceof Float) ||
            (property instanceof Double) ||
            (property instanceof Short) ||
            (property instanceof Boolean) ||
            (property instanceof Distribution) ||
            (property instanceof Vector) ||
            (property instanceof Hashtable)) {

            type = property.getClass().getName();
            type = type.substring(type.lastIndexOf(".") + 1);
        }
    
        return type;
    }

    /**
     * This method converts typed data to string representation.  This
     * function essentially reverses the behavior of reTypeProperty
     * @param property The data to change
     * @param delim The delim to use, if any
     * @return The untyped, stringified data, or null if error
     * @see #reTypeProperty
     */
    public static String unTypeProperty(Object property, String delim) {
        String newp = null;

        if (delim == null)
            delim = ",";

        if (property == null)
            return null;

        // Convert em
        if ((property instanceof String) ||
            (property instanceof Long) ||
            (property instanceof Integer) ||
            (property instanceof Float) ||
            (property instanceof Double) ||
            (property instanceof Short) ||
            (property instanceof Boolean)) {
            newp = property.toString();

        } else if (property instanceof Distribution) {
            //newp = "DELIM=" + delim + " ";
            newp = "";
            delim = " ";
            Enumeration e = ((Vector)property).elements();
            while (e.hasMoreElements()) {
                newp = newp + e.nextElement().toString();
                if (e.hasMoreElements())
                    newp = newp + delim;
            }

        } else if (property instanceof Vector) {
            newp = "DELIM=" + delim + " ";
            Enumeration e = ((Vector)property).elements();
            while (e.hasMoreElements()) {
                newp = newp + e.nextElement().toString();
                if (e.hasMoreElements())
                    newp = newp + delim;
            }
        } else if (property instanceof Hashtable) {
            newp = "DELIM=" + delim + " ";
            Enumeration e = ((Hashtable)property).keys();
            while (e.hasMoreElements()) {
                Object key = e.nextElement();
                newp = newp + key.toString() + delim + ((Hashtable)property).get(key).toString();
                if (e.hasMoreElements())
                    newp = newp + delim;
            }

        } else {
            Log.getDefault().log("Error, unsupported conversion type " + property.getClass(), Log.LOG_INFO);
        }

        return newp;
    }
}
