/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 * Connection.java
 ***************************************************************************************/

package utilities;

import java.io.*;
import java.util.*;
import java.net.*;

/**
 * Communicate uses this class to store connection information.
 */
public class Connection {
  /* Connection Protocols */
  public static final int RAW = 0;
  public static final int KQML = 1;
  public static final int LENGTHPREFIX = 2;

  protected transient Socket socket;
  protected transient BufferedReader input = null;
  protected transient BufferedWriter output = null;
  protected boolean def = false;
  protected String delim = Message.MSG_DELIM;
  protected int type;
  protected boolean accepted;

  /**
   * Constructor, uses default type-specific delimiters.
   * @param s The socket the connection is on
   * @param t The type of connection.  RAW, KQML, or LENGTHPREFIX.
   * @param a Did we accepted this connection? (we initiated if false)
   */
  public Connection(Socket s, int t, boolean a) throws IOException { 
    socket = s;
    type = t;
    accepted = a;

    switch (t) {
    case RAW:
      setDelim(""+'\0');
      break;
    case KQML:
      setDelim(Message.MSG_DELIM);
      break;
    }

    if (socket != null) {
        input = new BufferedReader(new InputStreamReader(socket.getInputStream()));
        output = new BufferedWriter(new OutputStreamWriter(socket.getOutputStream()));
    }
  }

  /**
   * Constructor
   * @param s The socket the connection is on
   * @param t The type of connection
   * @param d The message delimieter
   * @param a Did we accepted this connection? (we initiated if false)
   */
  public Connection(Socket s, int t, String d, boolean a) throws IOException { 
    socket = s;
    type = t;
    setDelim(d);
    accepted = a;

    if (socket != null) {
        input = new BufferedReader(new InputStreamReader(socket.getInputStream()));
        output = new BufferedWriter(new OutputStreamWriter(socket.getOutputStream()));
    }
  }

  /**
   * Returns the connection (pseudo-protocol) type
   * @return the type
   */
  public int getType() { return type; }

  /**
   * Sets the connection (pseudo-protocol) type
   * @param t the new type
   */
  public void setType(int t) { type = t; }

  /**
   * Returns the host name of machine attached to this connection
   * @return The host name
   */
  public String getRemoteHostName() {
      if (socket != null)
          return socket.getInetAddress().getHostName();
      else
          return null;
  }

  /**
   * Returns the host name of the local machine
   * @return The host name
   */
  public String getLocalHostName() {
      if (socket != null)
          return socket.getLocalAddress().getHostName();
      else
          return null;
  }

  /**
   * Returns the remote port number
   * @return The port number
   */
  public int getRemotePort() {
      if (socket != null)
          return socket.getPort();
      else
          return -1;
  }

  /**
   * Returns the local port number
   * @return The port number
   */
  public int getLocalPort() {
      if (socket != null)
          return socket.getLocalPort();
      else
          return -1;
  }

  /**
   * Returns the input reader attached to the socket
   * @return The stream
   */
  public BufferedReader getInput() { return input; }

  /**
   * Returns the input reader attached to the socket
   * @return The stream
   */
  public BufferedWriter getOutput() { return output; }

  /**
   * Gets the socket
   * @return The socket
   */
  public Socket getSocket() { return socket; }

  /**
   * Sets the message delimiter
   * @param d The message delimiter
   * @see Message#receive
   */
  public void setDelim(String d) { delim = d; }

  /**
   * Gets the message delimiter
   * @return the message delimiter
   */
  public String getDelim() { return delim; }

  /**
   * Closes the currently open socket
   */
  public synchronized void close() throws IOException {

    if (socket != null) {
      output.flush();
      output.close();
      output = null;
      input.close();
      input = null;
      socket.close();
      socket = null;
    }
  }

  /**
   * Tells weather this connection was accepted (e.g. picked up
   * by a server port listener) or established (e.g. actively opened
   * by the local agent).
   * @returns True if this connection was accepted
   */
  public boolean wasAccepted() { return accepted; }

  /**
   * Sets the default state
   * @param d The new state
   */
  public void setDefault(boolean d) { def = d; }

  /**
   * Gets the default state
   * @returns True if this connection is the default one
   */
  public boolean isDefault() { return def; }

  /**
   * Stringify me
   * @return A string representation of the connection
   */
  public String toString() {
    
    return "(Connection) Rem>" + getRemoteHostName() + ":" + getRemotePort()
      + " Loc>" + getLocalHostName() + ":" + getLocalPort() 
      + " [Type:" + getType()
      + " Dflt:" + isDefault()
      + " Acpt:" + wasAccepted()
      + "]";
  }
}
