/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

package utilities;

import java.util.*;
import java.awt.*;
import java.awt.geom.*;
import utilities.UTimer;

public class BetterArea extends Area
{
    static int default_depth = 6;
    static Hashtable CLists = new Hashtable();
    static BetterArea junk = new BetterArea();
    
    static {
        CLists.put(new Integer(default_depth),
                calculateControlList(default_depth));
    }
    
    int depth;
    
    public BetterArea()
    {
        super();
        depth = default_depth;
    }
    
    public BetterArea(Shape s)
    {
        super(s);
        depth = default_depth;
    }
   
                            
    static protected CListElement calculateControlList(int depth)
    {
        depth = Math.min(depth, 16);
        int wall = 1 << depth;
        CListElement parent = junk. new CListElement(0, 0, wall, wall);
        parent.next = calculateControlList(parent, 1, depth);
        parent.skip = null;
        return parent;
    }
    
    static protected CListElement calculateControlList(CListElement parent, 
            int curdepth, int maxdepth)
    {
        if (curdepth > maxdepth)
            return parent.skip;
        int neww = parent.w >> 1;
        int newh = parent.h >> 1;
        if (neww == 0 || newh == 0)
            return parent.skip;
        CListElement e11 = junk. new CListElement(
                parent.x, parent.y, neww, newh);
        CListElement e12 = junk. new CListElement(
                parent.x + neww, parent.y, neww, newh);
        CListElement e21 = junk. new CListElement(
                parent.x, parent.y + newh, neww, newh);
        CListElement e22 = junk. new CListElement(
                parent.x + neww, parent.y + newh, neww, newh);
        e11.skip = e12;
        e12.skip = e21;
        e21.skip = e22;
        e22.skip = parent.skip;
        e11.next = calculateControlList(e11, curdepth + 1, maxdepth);
        e12.next = calculateControlList(e12, curdepth + 1, maxdepth);
        e21.next = calculateControlList(e21, curdepth + 1, maxdepth);
        e22.next = calculateControlList(e22, curdepth + 1, maxdepth);
        return e11;
    }
   
    protected static CListElement getCList(int depth)
    {
        CListElement e = (CListElement) CLists.get(new Integer(depth));
        if (e == null)
        {
            CLists.put(new Integer(depth),
                    e = calculateControlList(depth));
        }
        return e;
    }
    
    public static void main(String args[])
    {
        Ellipse2D e = new Ellipse2D.Double(10.0, 10.0, 20.0, 20.0);
        final BetterArea b = new BetterArea(e);
        System.out.println("circle of radius 10.0 - ");
        System.out.println("include_partials: true");
        b.include_partials = true;
        for (int i = 0; i <= 7; i++)
        {
            final int temp = i;
            b.printcachestats = true;
            UTimer.time(100, new UTimer("findArea with partials") {
                        public void action() 
                        { 
                            b.setDepth(temp); 
                            temparea = b.findArea();
                            b.flushCache();
                            b.printcachestats = false;
                        }
                    }, true);
            System.out.println("  computed area with depth of " 
                    + i + ": " + temparea);
        }
        System.out.println("include_partials: false");
        b.include_partials = false;
        for (int i = 0; i <= 7; i++)
        {
            final int temp = i;
            b.printcachestats = true;
            UTimer.time(100, new UTimer("findArea without partials") {
                        public void action() 
                        { 
                            b.setDepth(temp);
                            temparea = b.findArea();
                            b.flushCache();
                            b.printcachestats = false;
                        }
                    }, true);
            System.out.println("computed area with depth of " 
                    + i + ": " + temparea + "\n");
        }
        System.out.println("actual area: " + Math.PI * 10.0 * 10.0);
    }
    
    private static double temparea = 0.0;
    
    public void setDepth(int i) { depth = i; }
    public int getDepth() { return depth; }
    
    protected boolean include_partials = true;
    protected boolean use_intersects = false;
    
    public double findArea()
    {
        if (isEmpty())
            return 0.0;
        CListElement clist = getCList(getDepth());
        Rectangle2D bounds = getBounds2D();
        double scalex, scaley;
        scalex = bounds.getWidth() / ((double) clist.w);
        scaley = bounds.getHeight() / ((double) clist.h);
        CListElement cur = clist;
        double total = 0.0;
        while (cur != null)
        {
            Rectangle2D rect = cur.getRectangle(scalex, scaley, bounds.getX(),
                    bounds.getY());
            if (checkContains(rect))
            {
                double addval = rect.getWidth() * rect.getHeight();
                total += addval;
                cur = cur.skip;
            } else if (include_partials && cur.next == cur.skip 
                    && checkIntersects(rect)) {
                double addval = rect.getWidth() * rect.getHeight();
                total += addval;
                cur = cur.skip;
            } else {
                cur = cur.next;
            }
        }
        return total;
    }
  
    protected boolean use_contains_cache = true;
    
    protected boolean checkContains(Rectangle2D rect)
    {
        if (use_intersects)
        {
            return contains(rect);
        }
        
        double x1, x2, y1, y2;
        x1 = rect.getX();
        x2 = x1 + rect.getWidth();
        y1 = rect.getY();
        y2 = y1 + rect.getHeight();
        if (use_contains_cache)
        {
            if (!containsCacheing(x1, y1))
                return false;
            else if (!containsCacheing(x2, y1))
                return false;
            else if (!containsCacheing(x1, y2))
                return false;
            else if (!containsCacheing(x2, y2))
                return false;
        } else {
            if (!contains(x1, y1))
                return false;
            else if (!contains(x2, y1))
                return false;
            else if (!contains(x1, y2))
                return false;
            else if (!contains(x2, y2))
                return false;
        }
        return true;
    }
    
    private Hashtable ccache = new Hashtable();
   
    protected void flushCache()
    {
        if (ccache.size() > 0)
        {
            printCacheStats();
            ccache = new Hashtable();
            containscount = 0;
            hitcount = 0;
            misscount = 0;
        }
    }
   
    int containscount = 0;
    int hitcount = 0;
    int misscount = 0;
    boolean printcachestats = false;
    
    protected void printCacheStats()
    {
        if (printcachestats && (hitcount > 0 || misscount > 0))
            System.out.println("point cache hits: " + hitcount + ", misses: " +
                    misscount);
    }
    
    protected boolean containsCacheing(double x, double y)
    {
        Point p = new Point((int) x, (int) y);
        boolean b;
        if (ccache.containsKey(p))
        {
            hitcount++;
            return ((Boolean) ccache.get(p)).booleanValue();
        }
        misscount++;
        b = contains(x, y);
        ccache.put(p, new Boolean(b));
        return b;
    }
    
    protected boolean checkIntersects(Rectangle2D rect)
    {
        if (use_intersects)
        {
            return intersects(rect);
        } else {
            //return (contains(rect.getX() + 0.5 * rect.getWidth(), rect.getY() +
            //            0.5 * rect.getHeight()));
            double x1, x2, y1, y2;
            x1 = rect.getX();
            x2 = x1 + rect.getWidth();
            y1 = rect.getY();
            y2 = y1 + rect.getHeight();
            if (use_contains_cache)
            {
                if (containsCacheing(x1, y1))
                    return true;
                else if (containsCacheing(x2, y1))
                    return true;
                else if (containsCacheing(x1, y2))
                    return true;
                else if (containsCacheing(x2, y2))
                    return true;
            } else {
                if (contains(x1, y1))
                    return true;
                else if (contains(x2, y1))
                    return true;
                else if (contains(x1, y2))
                    return true;
                else if (contains(x2, y2))
                    return true;
            }
            return false;
        }
    }
       
    public void add(Area rhs)
    {
        flushCache();
        super.add(rhs);
    }
    
    public void exclusiveOr(Area rhs)
    {
        flushCache();
        super.add(rhs);
    }
    
    public void intersect(Area rhs)
    {
        flushCache();
        super.intersect(rhs);
    }
    
    public void reset()
    {
        flushCache();
        super.reset();
    }
    
    public void subtract(Area rhs)
    {
        flushCache();
        super.subtract(rhs);
    }
    
    public void transform(AffineTransform t)
    {
        flushCache();
        super.transform(t);
    }
 
    public void dumpPath()
    {
        PathIterator i = getPathIterator(null);
        int winding = i.getWindingRule();
        int whereto;
        float [] array = new float[6];
        
        System.err.println("dumping path:");
        System.err.println("winding " + winding);
        while (!i.isDone())
        {
            whereto = i.currentSegment(array);
            if (whereto == PathIterator.SEG_CLOSE)
                System.err.println("  SEG_CLOSE");
            else if (whereto == PathIterator.SEG_MOVETO)
                System.err.println("  SEG_MOVETO: " + array[0] + ", " +
                        array[1]);
            else if (whereto == PathIterator.SEG_LINETO)
                System.err.println("  SEG_LINETO: " + array[0] + ", " + 
                        array[1]);
            else if (whereto == PathIterator.SEG_QUADTO)
                System.err.println("  SEG_QUADTO: " + array[0] + ", " + 
                        array[1] + ", " + array[2] + ", " + array[3]);
            else if (whereto == PathIterator.SEG_CUBICTO)
                System.err.println("  SEG_CUBICTO: " + array[0] + ", " + 
                        array[1] + ", " + array[2] + ", " + array[3] + 
                        ", " + array[4] + ", " + array[5]);
            i.next();
        }
        System.err.println("done");
    }
    
    public Iterator getClosedSubpaths()
    {
        //dumpPath();
        
        PathIterator i = getPathIterator(null);
        Vector subshapes = new Vector();
        int winding = i.getWindingRule();
        int whereto;
        GeneralPath cur = null;
        float [] pathstart = null;
        
        while (!i.isDone())
        {
            
            float [] array = new float[6];
            boolean close = false;
            
            whereto = i.currentSegment(array);
            if (whereto == PathIterator.SEG_CLOSE)
            {
                if (cur != null)
                    cur.closePath();
                close = true;
                //System.err.println("closing");
            } else if (whereto == PathIterator.SEG_MOVETO) {
                //System.err.println("moveto " + array[0] + ", " + array[1]);
                pathstart = array;
                cur = new GeneralPath(winding);
                cur.moveTo(array[0], array[1]);
                i.next();
                continue;
            } else {
                if (cur == null)
                {
                    System.err.println("WARNING: path without moveto - there'll"
                           + "probably be an exception soon");
                    cur = new GeneralPath(winding);
                }
                if (whereto == PathIterator.SEG_LINETO)
                {
                    //System.err.println("lineto " + array[0] + ", " + array[1]);
                    cur.lineTo(array[0], array[1]);
                } else if (whereto == PathIterator.SEG_QUADTO) {
                    //System.err.println("quadto " + array[0] + ", " + array[1]);
                    cur.quadTo(array[0], array[1], array[2], array[3]);
                } else if (whereto == PathIterator.SEG_CUBICTO) {
                    //System.err.println("curveto " + array[0] + ", " + array[1]);
                    cur.curveTo(array[0], array[1], array[2], array[3],
                            array[4], array[5]);
                }
                if (pathstart == null)
                    pathstart = array;
                else {
                    /*
                    boolean equal = true;
                    for (int c = 0; c < pathstart.length; c++)
                    {
                        if (array[c] != pathstart[c])
                        {
                            equal = false;
                            break;
                        }
                    }
                    if (equal)
                    {
                        //System.err.println("closing2");
                        close = true;
                    }
                    */
                }
            }
            if (close && cur != null)
            {
                subshapes.add(cur);
                cur = null;
                //System.err.println("closed");
                close = false;
            }
            i.next();
        }
        //System.err.println("BetterArea subshapes: " + subshapes.size());
        return subshapes.iterator();
    }
    
    public boolean hasMultipleClosedSubpaths()
    {
        float [] pathstart = new float[6];
        float [] array = new float[6];
        PathIterator i = getPathIterator(null);
        int count = 0;
        while (!i.isDone())
        {
            int whereto = i.currentSegment(array);
            if (whereto == PathIterator.SEG_MOVETO)
                i.currentSegment(pathstart);
            else if (whereto == PathIterator.SEG_CLOSE)
                count++;
            else {
                boolean equal = true;
                for (int c = 0; c < pathstart.length; c++)
                {
                    if (array[c] != pathstart[c])
                    {
                        equal = false;
                        break;
                    }
                }
                if (equal)
                    count++;
            }
            if (count > 1)
                return true;
            i.next();
        }
        return false;
    }
    
    public Object clone()
    {
        Object o = super.clone();
        return new BetterArea((Shape) o);
    }
    
    protected class CListElement
    {
        public int x, y, w, h;
        public CListElement 
            next = null, skip = null;
        
        public CListElement(int tx, int ty, int tw, int th)
        {
            x = tx;
            y = ty;
            w = tw;
            h = th;
        }
        
        public Rectangle2D getRectangle(double scalex, double scaley, 
                double x1, double y1)
        {
            double tx = ((double) x) * scalex + x1;
            double ty = ((double) y) * scaley + y1;
            double tw = ((double) w) * scalex;
            double th = ((double) h) * scaley;
            return new Rectangle2D.Double(tx, ty, tw, th);
        }
        
        public Rectangle2D getRectangle(double scalex, double scaley)
        {
            return getRectangle(scalex, scaley, 0.0, 0.0);
        }
        
        public String toStringWithoutLinks()
        {
            return new String( "x " + x + ", y " + y + ", w " + w + ", h " + h);
        }
        
        public String toString()
        {
            String s = toStringWithoutLinks();
            s =  new String(s + ", next [");
            if (next == null)
                s = new String(s + "null], skip [");
            else
                s = new String(s + next.toStringWithoutLinks() + "], skip [");
            if (skip == null)
                s = new String(s + "null]");
            else
                s = new String(s + skip.toStringWithoutLinks() + "]");
            return s;
        }
    }
}

