/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
* $Revision: 1.2 $
* Author : Regis Vincent [vincent@cs.umass.edu]
* Creation date: 28 Jan 99 19:30
* Last file update: $Date: 2005/01/19 19:00:23 $
*****************************************************************/

package simulator.sensor;

/* Global Includes */
import java.io.*;
import java.util.*;

/* Local Includes */
import utilities.TaemsRandom;
import utilities.Log;

/**
 * SensorDeviation represents the different biais the simulator
 * could apply on a sensor before sending the value to the agent
 */

public class SensorDeviation {
  protected static final int PLUS=0;
  protected static final int MINUS=1;
  protected static final int EQUAL=2;
  
  protected static final int VALUE_FIXED=10;
  protected static final int PERCENT_FIXED=11;
  protected static final int VALUE_RANDOM=12;
  protected static final int PERCENT_RANDOM=13;

  private double probability;
  private double min,max;
  private int type;
  private int function;
  private boolean fixed;
  private double currentDeviation = -1;

  /**
   * Define a sensor deviations
   * @param p - proability associated with this deviation
   * @param token - function (PLUS, MINUS, EQUAL)
   * @param min - minimum
   * @param max - maximum
   * @param type - (PERCENT, VALUE)
   */
  public SensorDeviation(double p, int token, double min, double max, int type) {
    probability = p;
    function = token;
    this.min = min;
    this.max = max;
    this.type = type;
    if ((type == VALUE_FIXED) || (type == PERCENT_FIXED))
      fixed = true;
    else
      fixed = false;
  }
    
    
  public double getProbability() { return(probability); }
  public int getFunction() { return(function); }
  public boolean getFixed() { return(fixed); }
  public double getMax() { return(max); }
  public double getMin() { return(min); }
  public int getType() { return(type); }

  public double getDeviation() {
    TaemsRandom r = TaemsRandom.getDefault();
    if (fixed && (currentDeviation != -1)) 
      return(currentDeviation);
    else {
      currentDeviation =  min + (max - min)*r.nextFloat();
      return(currentDeviation);
    }
  }
      

  public static int convertToken(String s) {
    if (s.equals("+"))
      return (PLUS);
    if (s.equals("-"))
      return (MINUS);
    return(EQUAL);
  }

  public String reverseToken(int v) {
    if (v == EQUAL) 
      return("=");
    if (v == PLUS) 
      return("+");
    if (v == MINUS) 
      return("-");
    return("?E?");
  }

  /**
   * unit is (# for fixed percent, ! for fixed value, 
   * % random percent   , ~ for random value )
   */
  public static int convertDeviation(String s) {
    if (s.equals("%"))
      return (PERCENT_RANDOM);
    if (s.equals("~"))
      return (VALUE_RANDOM);
    if (s.equals("#"))
      return (PERCENT_FIXED);
    return(VALUE_FIXED);
  }
  
  public String reverseDeviation(int v) {
    if (v == PERCENT_RANDOM) 
      return("%");
    if (v == VALUE_RANDOM) 
      return("~");
    if (v == PERCENT_FIXED) 
      return("#");
    if (v == VALUE_FIXED) 
      return("!");
    return("?E?");
  }

  /**
   * Read the string error : +5% or +/-5% and get the type of error
   * Currently only +, -, +/- are defined
   * @param errorString 
   * @param value is the error value returned
   * @return Vector of Deviation
   */
  public static Vector parseLine(String errorString) {
    int type=0;
    String substring = "",value;
    Vector v=new Vector();
    int start,end;

    if (errorString.equals("*")) {
      SensorDeviation s = new SensorDeviation(1, SensorDeviation.PLUS,
					      0, 0, SensorDeviation.VALUE_FIXED);
      v.addElement(s);
      return(v);
    } 
    if ((errorString.endsWith(")")) && (errorString.startsWith("("))) {
      start = 1;
      end = errorString.length() - 1;
      value = errorString.substring(start, end);
      try {
	StringTokenizer tok = new StringTokenizer(value,")");
	while (tok.hasMoreElements()) {
	  substring = (String)tok.nextElement();
	  if (substring.startsWith("(")) 
	    substring = substring.substring(1,substring.length());
	  StringTokenizer tok2 = new StringTokenizer(substring);
	  if (tok2.countTokens() != 5) {
	    Log.getDefault().log("Not the right number of tokens: " + substring, 0);
	    return null;
	  }
	  double prob = (Double.valueOf((String)tok2.nextElement())).doubleValue();
	  String token = (String)tok2.nextElement();
	  double min = (Double.valueOf((String)tok2.nextElement())).doubleValue();
	  double max = (Double.valueOf((String)tok2.nextElement())).doubleValue();
	  String deviation = (String)tok2.nextElement();
	  v.addElement(new SensorDeviation(prob,
	  			   SensorDeviation.convertToken(token),
	  			   min,max,
	  			   SensorDeviation.convertDeviation(deviation)));
	}
      }
      catch (NumberFormatException e) {
	Log.getDefault().log("Syntax error in directive " + errorString, 0);
	return(null);
      }
      return(v);
    }
    return(null);
  }


  public void setProbability(double v) { probability = v; }
  public void setFunction(int v) { function = v; }
  public void setFixed(boolean v) { fixed = v; }
  public void setMax(double v) { max = v; }
  public void setMin(double v) { min = v; }
  public void setType(int v) { type = v; }

  public String toString() {
    String s = "(" + probability + " " + reverseToken(function) + 
      " " + min + " " + max + " " + reverseDeviation(type) + ")";
    return(s);
  }
}
