/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
* Author : Regis Vincent [vincent@cs.umass.edu]
* Creation date: 25 Feb 99 15:41
* ResourceSensor.java
*****************************************************************/
package simulator.sensor;

/* Global Includes */
import java.util.Hashtable;
import java.util.Vector;
import java.util.StringTokenizer;

/* Local Includes */
import simulator.*;
import utilities.TaemsRandom;
import utilities.Distribution;
import utilities.Log;
import simulator.locale.*;
/**
 * Sense any kind of resource, if this resource is defined in the
 * simulator. Of course, it applies a deviation from the real value.
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Deviation:String - The Deviation to apply, check 
 * <LI>Resource:String - The name of the resources
 * </UL>
 * @see simulator.sensor.SensorDeviation
 */

public class ResourceSensor extends Sensor {
  protected String resource;
  protected Vector deviation;
  protected Hashtable cachedValues;

  /**
   * Constructor.
   */
  public ResourceSensor(String n, String d) {
    super(n, d);
  }

  /**
   * Init, usually you will retrieve input properties here, and setup
   * any state you need to.  
   * @see agent.simplest.State#reTypeProperty
   */  
  public void init() {
     // Level
    if (getData().containsKey("Deviation"))
      deviation = SensorDeviation.parseLine((String)getData().get("Deviation"));

    if (getData().containsKey("Resource"))
      resource = (String)getData().get("Resource");

	/*
	  if (getData().containsKey("Locale"))
      name = (String)getData().get("Locale")+ "/" + name;
	*/

    cachedValues = new Hashtable();
    super.init();
  }

  public void pulse() {
    cachedValues.clear();
  }

  /**
   * Accessor for the deviation
   * getDeviation() get the error deviation
   * @return Vector
   */
  public Vector getDeviation() { return deviation; }
  
  /**
   * setDeviation, obviously set the deviation vector
   *
   */
  public void setDeviation(Vector v) { deviation  = v; }

    public Object getObjectSensed(Locale loc) {
        utilities.Log log=utilities.Log.getDefault();
        log.log("getObjectSensed: looking for "+resource+" in "+loc.getName(), 4);
        try {
            return loc.getResource(resource);
        } catch (simulator.locale.Locale.ResourceDoesNotExist x) {
            log.log("getObjectSensed: could not find "+resource+" there", 4);
            return null;
        }
    }
    
  /**
   * make the real sensing
   * @return Object  if it was sucessfull, null otherwise
   */
  public Object sense(Locale loc, String data) throws SensingErrorException  {
    Hashtable runtimeData = simulator.script.Script.parseDataLine(data);
    Vector deviationToApply;
    String dataKey;

    if (runtimeData.containsKey("Deviation"))
       deviationToApply = SensorDeviation.parseLine((String)runtimeData.get("Deviation"));
    else {
       deviationToApply = deviation;
       if (getData().get("Deviation") == null) 
	   throw new SensingErrorException("ResourceSensor: Sensor Definition Error");
       runtimeData.put("Deviation", getData().get("Deviation"));
    }
    
     if (deviationToApply == null)
      throw new SensingErrorException("ResourceSensor: Deviation Format Error");

    if (runtimeData.containsKey("Resource"))
      resource = (String)runtimeData.get("Resource");
    else if (getData().containsKey("Resource")) {
      resource = (String)getData().get("Resource");
      runtimeData.put("Resource", resource);
    }

    if (resource == null) 
      throw new SensingErrorException("ResourceSensor: Resource not defined");
    dataKey=generateKey(runtimeData);
    if (cachedValues.containsKey(dataKey)) {
	return((Double)cachedValues.get(dataKey));
    }
    else {
	Resources res = (Resources)getObjectSensed(loc);
	if(res==null) throw new SensingErrorException("ResourceSensor: Can't find resource to sense!");
	
	Double v = new Double(res.getCurrentValue());
	if (v == null)
	    throw new SensingErrorException("ResourceSensor: Format Error " + resource);
	v = applyDeviation(v, deviationToApply );
	cachedValues.put(generateKey(runtimeData),v);
	return(v);
    }	
  }

  /**
   * Apply a deviation on the real value of the sensor.
   * @param Double d real computed value
   * @param Vector of SensorDeviation
   * @return the new value (after deviation)
   */
  protected Double applyDeviation(Double d, Vector devVector) {
    TaemsRandom r = TaemsRandom.getDefault();
    Distribution D=new Distribution();
    for(int i =0 ; i < devVector.size(); i++) {
      SensorDeviation ds = (SensorDeviation)devVector.elementAt(i);
      D.insertElementAt(new Float(ds.getProbability()),0);
      D.insertElementAt(new Float(i),0);
    }
    double dev=0;
    r.setDistribution(D);
    int index = (int)(r.nextValue());
    r.unsetDistribution();
    SensorDeviation ds = (SensorDeviation)devVector.elementAt(index);
    
    if ((ds.getType() == SensorDeviation.PERCENT_RANDOM) || 
        (ds.getType() == SensorDeviation.PERCENT_FIXED)) 
      dev = d.doubleValue() * ds.getDeviation() / 100;
    else
      dev = ds.getDeviation();
    if (ds.getFunction() == SensorDeviation.EQUAL) 
      return(new Double(dev));
    if (ds.getFunction() == SensorDeviation.PLUS) 
      return(new Double(dev + d.doubleValue()));
    if (ds.getFunction() == SensorDeviation.MINUS) 
      return(new Double(d.doubleValue() - dev));
    return(d);
  }

    void clearCache() {
	cachedValues.clear();
    }

 public String toString() {
    String answer = getName();
    
    if (getData().containsKey("Resource")) 
      answer = name + ", Resource:String:" + (String)getData().get("Resource")
	+ " " ;

    if (deviation != null)
      for(int i =0 ; i < deviation.size(); i++) {
	SensorDeviation ds = (SensorDeviation)deviation.elementAt(i);
	answer = answer + ", Deviation[" + i + "] = " + ds + " ";
      }

    return(answer);
  }
}
