/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	SetSensorDeviation.java
 ***************************************************************************************/

package simulator.script;

import simulator.*;
import simulator.sensor.*;
import java.util.*;

/**
 * Sets the deviation distribution on a sensor.
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Sensor:String - Sensor query string
 * <LI>Deviation:String - Deviation distribution, encoded in the same
 * manner as in the sensor script file.
 * <LI>Agent:String - Owner of the sensor (optional, if not supplied the
 * assertion checks the simulator's global sensor table)
 * </UL>
 */
public class SetSensorDeviation extends Reaction {
  String sensor;
  String deviation;
  String name = null;

  /**
   * Constructor.
   */
  public SetSensorDeviation(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get it
    sensor = (String)getData().get("Sensor");

    // Get it
    deviation = (String)getData().get("Deviation");

    // Get it
    name = (String)getData().get("Agent");

    super.init();
  }

  /**
   * Gets a ref to the agent
   * @param n Name of the agent to look for
   * @return The Agent, or null if not found
   */
  public Agent findAgent(String n) {

    synchronized (Simulator.getAgentTable()) {
      return Simulator.getSimulator().findAgent(n);
    }
  }

  /**
   * Finds a sensor
   * @param s The sensor query string (if null, uses global sensor table)
   * @param a The name of the agent
   * @return The sensor, or null if not found
   */
  public ResourceSensor findSensor(String s, String a) {
    Sensor sensor = null;
    Sensing sensing = null;
    Agent agent = null;

    // Look for the agent (if any)
    if (name != null) {
      agent = findAgent(a);
    } else {
      a = "*";
    }

    // Look for the sensing controller
    if (agent == null) {
      sensing = Simulator.getSimulator().getGlobalSensing();
    } else {
      sensing = agent.getSensing();
    }

    // Find the sensor
    if (sensing == null) return null;
    sensor = sensing.getSensor(s);

    if (sensor instanceof simulator.sensor.ResourceSensor)
      return((ResourceSensor)sensor);
    return null;
  }

  /**
   * Realize
   */
  public boolean realize() {

    // Find the sensor
    ResourceSensor s = findSensor(sensor, name);
    if (s == null) return false;

    // Then make the deviation
    Vector v = SensorDeviation.parseLine(deviation);
    if (v == null) return false;
    s.setDeviation(v);

    return true;
  }
}
