/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	SetResourceLevel.java
 ***************************************************************************************/

package simulator.script;

import simulator.*;
import java.util.Vector;

/**
 * Sets the current level of a resource to some fixed level, using a 
 * ResourceEvent.
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Resource:String - The name of the resource to check
 * <LI>Dur:Integer - The desired duration of the event forcing the
 * level to a specific value (optional, default 0)
 * <LI>Level:Float - The level to set the current level to.  If
 * Level is set to "Max" or "Min" the level will be set to the 
 * maximum and minimum values for the resource, respectively.
 * </UL>
 */
public class SetResourceLevel extends Reaction {
  String op;
  float level;
  int dur = 0;
  Resources rsrc;
  Vector events = new Vector();

  /**
   * Constructor.
   */
  public SetResourceLevel(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get the level
    String str = (String)getData().get("Level");
    if (str.equalsIgnoreCase("Max"))
      level = Float.POSITIVE_INFINITY;
    else if (str.equalsIgnoreCase("Min"))
      level = Float.NEGATIVE_INFINITY;
    else
      level = Float.valueOf(str).floatValue();

    // Duration
    if (getData().containsKey("Dur"))
      dur = Integer.parseInt((String)getData().get("Dur"));

    // Get the resource
    rsrc = findResource();

    super.init();
  }

  /**
   * Finds the resource object
   */
  public Resources findResource() {
      utilities.Log logger=utilities.Log.getDefault();
      String rsc=(String)getData().get("Resource");
      try {
	  return (Resources)Simulator.getSimulator().getResource(rsc);
      } catch (simulator.locale.Locale.ResourceDoesNotExist x) {
	  logger.log("(SCRIPT) Error in Resource path "+rsc, 1);
	  return null;
      }
  }

  
  /**
   * Creates the event we'll be using
   */
  public ResourceEvent createEvent() {

    return new ResourceEvent(0, dur, rsrc, level, ResourceEvent.NO_CHANGE, ResourceEvent.NO_CHANGE, true);
  }

  /**
   * Performs the action
   */
  public boolean realize() {

    if (rsrc == null)
      findResource();
    if (rsrc == null)
      return false;

    if (level == Float.POSITIVE_INFINITY)
      level = rsrc.getMaximum();
    else if (level == Float.NEGATIVE_INFINITY)
      level = rsrc.getMinimum();

    ResourceEvent e = createEvent();
    EventQueue.getQueue().addEvent(e);
    events.addElement(e);

    return true;
  }
}
