/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	SensorQuery.java
 ***************************************************************************************/

package simulator.script;

import simulator.*;
import simulator.locale.*;
import utilities.*;
//import java.util.*;

/**
 * Queries a sensor and compares the results to those specified
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Sensor:String - Sensor query string
 * <LI>Result:String - The result to compare to
 * <LI>Type:String - The result's type (uses utilities.Converter#reTypeProperty
 * to convert before comparison) (default String, optional)
 * <LI>Agent:String - Owner of the sensor (optional, if not supplied the
 * assertion checks the simulator's global sensor table)
 * </UL>
 * @see utilities.Converter#reTypeProperty
 */
public class SensorQuery extends Assertion {
  String sensor;
  Object result;
  String name = "*";
  String type = "String";
    Locale locale = Simulator.getSimulator().getLocale("Global");
  /**
   * Constructor.
   */
  public SensorQuery(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get it
    sensor = (String)getData().get("Sensor");

    // Get it
    if (getData().containsKey("Type"))
      type = (String)getData().get("Type");

    // Get it
    name = (String)getData().get("Agent");

    if (getData().containsKey("Locale"))
	locale = Simulator.getSimulator().getLocale((String)getData().get("Locale"));

    // Get it
    result = Converter.reTypeProperty((String)getData().get("Result"), type);

    super.init();
  }

  /**
   * Gets a ref to the agent
   * @param n Name of the agent to look for
   * @return The Agent, or null if not found
   */
  public Agent findAgent(String n) {

    synchronized (Simulator.getAgentTable()) {
      return Simulator.getSimulator().findAgent(n);
    }
  }

  /**
   * Checks the assertion.
   */
  public boolean check() {

    // Find the sensor
    Sensing sen = null;
    if (name == null)
      sen = Simulator.getSimulator().getGlobalSensing();
    else {
      Agent a = findAgent(name);
      if (a == null) return false;
      sen = a.getSensing();
    }

    // Then query it
    Object r = querySensor(sen, sensor, type);
    if ((r != null) && r.equals(result))
      return true;

    return false;
  }

  /**
   * Queries the sensor and types the output
   * @param s The sensor string to use
   * @param t The type to cast it to
   */
  public Object querySensor(Sensing sen, String s, String t) {
    Object o = null;

    try {
      if (sen == null) return null;
      String r = sen.runSensing(locale, s);
      if (r == null) return null;
      
      o = Converter.reTypeProperty(r, t);

    } catch (simulator.sensor.SensingErrorException e) { }

    return o;
  }
}
