/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	SendMessage.java
 ***************************************************************************************/

package simulator.script;

import utilities.*;
import simulator.*;
import java.util.*;

/**
 * Sends a KQML message to an agent.
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Agent:String - The name of the recipient
 * <LI>Source:String - The name of the source (optional, default "simulator")
 * <LI>Perf:String - The message performative (optional, default "tell")
 * <LI>CW:String - The message content word (optional, default "ScriptMessage")
 * <LI>CD:String - The message content data (optional, default "")
 * <LI>Control:Boolean - Set to true if this should be marked as a control
 * message (optional, default false)
 * <LI>Other:Hashtable - Hashtable of any other KQML parameters you wish
 * to add to the message (optional).  Make sure you don't use the "," as
 * a separator, use the DELIM feature as described in reTypeProperty.
 * </UL>
 */
public class SendMessage extends Reaction {
  String agent;
  String source = "simulator";
  String perf = "tell";
  String CW = "ScriptMessage";
  String CD = "";
  boolean control = false;
  Hashtable other = null;

  /**
   * Constructor.
   */
  public SendMessage(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Agent
    agent = (String)getData().get("Agent");

    // Source
    if (getData().containsKey("Source"))
      source = (String)getData().get("Source");

    // Perf
    if (getData().containsKey("Perf"))
      perf = (String)getData().get("Perf");

    // Content word
    if (getData().containsKey("CW"))
      CW = (String)getData().get("CW");

    // Content data
    if (getData().containsKey("CD"))
      CD = (String)getData().get("CD");

    // Control
    if (getData().containsKey("Control"))
      control = Boolean.valueOf((String)getData().get("Control")).booleanValue();

    // Other junk
    if (getData().containsKey("Other"))
	other = (Hashtable)Converter.reTypeProperty((String)getData().get("Other"), "Hashtable");

    super.init();
  }

  /**
   * Gets a ref to the agent
   * @param n Name of the agent to look for
   * @return The Agent, or null if not found
   */
  public Agent findAgent(String n) {

    synchronized (Simulator.getAgentTable()) {
      return Simulator.getSimulator().findAgent(n);
    }
  }

  /**
   * Performs the action
   */
  public boolean realize() {

    Agent a = findAgent(agent);
    if (a == null) return false;

    Agent s = null;
    if (!source.equalsIgnoreCase("simulator")) {
      s = findAgent(source);
      if (s == null) return false;
    }

    KQMLMessage msg = new KQMLMessage(perf, "(" + CW + " " + CD + ")", agent);
    if (s != null)
      msg.setSourceAddr(s.getName());
    
    if (control)
      msg.addField("type", "control");

    if (other != null) {
      Enumeration e = other.keys();
      while (e.hasMoreElements()) {
	String key = (String)e.nextElement();
	msg.addField(key, (String)other.get(key));
      }
    }

    a.sendMsg(msg);

    return true;
  }
}
