/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ScriptPulseType.java
 ***************************************************************************************/

package simulator.script;

import simulator.*;
import java.util.*;

/**
 * Checks the current type of pulse being sent out to the scripts.  These
 * are all defined as final variables in the Scripter class.
 * <P>
 * Be careful when using the POSTEVENTS type in conjunction with reactions
 * that can effect the event queue.  The simulator event processing 
 * mechanism will continue to process the same time slot until no more
 * events exist in that time slot.  If your reaction adds an event 
 * at POSTEVENTS, the simulator will get stuck on that time forever.  If
 * you need to add an event at this time, you may also want to use the
 * TimeChanged assertion to ensure that the rule will only fire once
 * per time slot.
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Type:String - Type of pulse to match (PREEVENTS, POSTEVENTS, PERIODIC)
 * (optional, default PREEVENTS)  You can combine multiple types with
 * a '|' (e.g. "PREEVENTS|POSTEVENTS")
 * </UL>
 * @see simulator.script.Scripter
 */
public class ScriptPulseType extends Assertion {
  int type = Scripter.PREEVENTS;

  /**
   * Constructor.
   */
  public ScriptPulseType(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Type
    StringTokenizer tok = new StringTokenizer((String)getData().get("Type"), "|");
    if (tok.hasMoreTokens())
      type = 0;
    while (tok.hasMoreTokens()) {
      String str = tok.nextToken().trim();

      if (str.equalsIgnoreCase("PREEVENTS"))
	type |= Scripter.PREEVENTS;
      else if (str.equalsIgnoreCase("PERIODIC"))
	type |= Scripter.PERIODIC;
      else if (str.equalsIgnoreCase("POSTEVENTS"))
	type |= Scripter.POSTEVENTS;
      else if (str.equalsIgnoreCase("OTHER"))
	type |= Scripter.OTHER;
    }

    super.init();
  }

  /**
   * Checks the assertion.
   */
  public boolean check() {

    return ((type & Simulator.getSimulator().getScripter().getPulseType()) > 0);
  }
}
