/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Script.java
 ***************************************************************************************/

package simulator.script;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.awt.*;

/* Local Includes */
import simulator.*;
import utilities.*;

/**
 * Generic script class from which other Scripts must be defined.  Use
 * a derivation to actually get behaviors out of your scripts, this one
 * doesn't do anything.
 * <P>
 * By convention, we hope that event authors place the supported data
 * descriptions in the header comments for their classes.  Take a look at
 * some of the other source files for examples of how to do this.
 * <B>Data:</B><BR>
 * <UL>
 * <LI>Color:Boolean - Update the display with colors. (optional, default
 * to true) This may slow down simulation if there are a lot of scripts.
 * </UL>
 */
public class Script {
  /**
   * Script item state codes
   */
  public static final int NORMAL = 0;
  public static final int ACTIVE = 1;
  public static final int ERROR  = 2;
  public static final int UNKNOWN = 3;

  private static int NUM = 0;
  private Container display;
  protected Log logger;

  private String name;
  protected Vector assertions = new Vector(5);
  protected Vector reactions = new Vector(5);
  protected Hashtable labels = new Hashtable(10);
  private Hashtable data;
  private int activations = 0;
  private int completions = 0;
  private boolean usecolor = true;

  /**
   * Constructor
   */
  public Script(String n, String d) {
    logger = Log.getDefault();

    if ((n != null) && (!n.equals("")))
      name = n;
    else
      name = "Script " + NUM++;

    data = parseDataLine(d);
  }
  
  /**
   * Init.  This should be called after all assertions and recations have
   * been added.  Make sure your derivations make a super call to this
   * class.
   */  
  public void init() {    

    // Update the colors?
    if (data.containsKey("Color")) {
      usecolor = Boolean.valueOf((String)data.get("Color")).booleanValue();
    }

    //  Setup the display
    display = new ScrollPane();
    Panel panel = new Panel();
    panel.setLayout(new VFlowLayout(VFlowLayout.TOP));
    panel.setFont (new Font ("Helvetica", Font.BOLD , 14));
    makeDisplay(panel);
    display.add(panel);
    display = panel; // cause stupid thang aint stretchin'

    // Update colors
    updateColor(null, UNKNOWN);
  }		

  /**
   * Parses the input data line, converting it into a hashtable
   * of string data.  Script, Assertion and Reaction all use this
   * method to parse the Data at the end of their description
   * line into a more usable form.  Here's an example of the data
   * this method will parse:
   * <P>
   * <CENTER><TT>&lt;cfg info&gt;, Name1: Value1, Name2 : Value2,Name3:Value3</TT></CENTER>
   * <P>
   * The above line will be parsed into a hashtable with each of the Name*
   * values being used as a key pointing to the appropriate Value* item.
   * <P>
   * Each object currently parses its data line in its constructor.  The
   * object can use the getData() call to get a handle to the resulting
   * data table, which will contain the keys and values in String form.
   * The object is responsible for further processing the String values
   * into a useful format.
   * <P>
   * With the current parsing technique, you can't have names or values
   * with embedded commas.  If this proves to be a problem I can make
   * it smarter.
   * @param line The intput data line
   * @return The hashtable of data
   */
  public static Hashtable parseDataLine(String line) {
    Hashtable hash = new Hashtable(5);

    if (line == null) return new Hashtable(0);

    // Tokenizer it
    StringTokenizer tok = new StringTokenizer(line, ",");    
    while (tok.hasMoreTokens()) {
      // Get the next value pair
      String str = tok.nextToken().trim();
      if (str.indexOf(":") == -1)
	hash.put(str, "");
      else {
	String name = str.substring(0, str.indexOf(":")).trim();
	String value = str.substring(str.indexOf(":")+1).trim();
	hash.put(name, value);
      }
    }

    return hash;
  }

  /**
   * This method should perfor the assertion checks in whichever
   * manner is appropriate.  You'll need to override this to get your
   * scripts to do anything, since this instance does no assertion
   * checking.
   * <P>
   * This instance updates the colors of the script and reactions,
   * make a super call to it at the beginning of your pulse derivation.
   */
  public void pulse() { 

    updateColor(this, NORMAL);

    Enumeration e = reactions.elements();
    while (e.hasMoreElements())
      updateColor(e.nextElement(), NORMAL);
  }

  /**
   * Starts the reactions (e.g. calls Reaction.realize)
   * @see simulator.script.Reaction#realize
   */
  public void start() {
    Enumeration e = reactions.elements();

    activations++;

    ((Label)labels.get(this)).setText(this.toString());
    updateColor(this, ACTIVE);

    logger.log("(SCRIPT) " + getName() + " starting", 4);
    while (e.hasMoreElements()) {
      Reaction r = (Reaction)e.nextElement();
      if (! r.realize()) {
	logger.log("(SCRIPT) Error realizing reaction " + r, 0);
	updateColor(r, ERROR);
      } else {
	logger.log("(SCRIPT) Reaction " + r + " realized.", 2);
	updateColor(r, ACTIVE);
      }
    }
  }

  /**
   * Resets the script by terminating any actions and resetting
   * the internal counters
   */
  public void reset() {

    logger.log("(SCRIPT) " + getName() + " resetting ", 2);

    activations = 0;

    // Update colors
    updateColor(null, UNKNOWN);
  }

  /**
   * Adds a assertion to the script, initializing it in the process.
   * @param a The Assertion to add
   */
  public void addAssertion(Assertion a) {

    logger.log("(SCRIPT) " + getName() + " adding assertion " + a, 2);
    assertions.addElement(a);
    a.init();
  }

  /**
   * Adds a reaction to the script, initializing it in the process.
   * @param e The reaction to add
   */
  public void addReaction(Reaction r) {

    logger.log("(SCRIPT) " + getName() + " adding reaction " + r, 2);
    reactions.addElement(r);
    r.init();
  }

  /**
   * Returns the number of times the script has been activated
   * (the number of times start has been called).
   * @return The number of activations.
   */
  public int getActivations() { return activations; }

  /**
   * Returns the script's name.  This should be unique among the set
   * being used in the simulator (although it doesn't necessarily have
   * to be).
   * @return The name
   */
  public String getName() {
    return name;
  }

  /**
   * Returns the input data table
   * @return The data table, as parsed from the input data line
   */
  public Hashtable getData() { return data; }

  /**
   * Returns the display
   * @return A panel
   */
  public Container getDisplay() {
    return display;
  }

  /**
   * Updates the color of an item's label
   * @param i The item to update (e.g. Script, Assertion or Reaction)
   * If null, all items are set to this color.
   * @param s The state to change it to
   */
  public void updateColor(Object i, int s) {
    if (! usecolor) return;
    Color c;
    Label l;

    // Find the color
    switch (s) {
    case NORMAL:
      c = Color.black;
      break;
    case ACTIVE:
      c = Color.red;
      break;
    case ERROR:
      c = Color.green;
      break;
    case UNKNOWN:
    default:
      c = Color.gray;
    }

    if (i != null) {
      // Set color for individual item
      l = (Label)labels.get(i);
      if ((l != null) && (l.getForeground() != c))
	l.setForeground(c);

    } else {
      // ..or all items
      Enumeration e = labels.keys();
      while (e.hasMoreElements()) {
	l = (Label)labels.get(e.nextElement());
	if ((l != null) && (l.getForeground() != c))
	  l.setForeground(c);
      }
    }
  }

  /**
   * Same as above, but works on vectors of the script objects.
   * @param v The vector to process
   * @param s The state to change it to
   */
  public void updateColor(Vector v, int s) {

    if (v == null) {
      logger.log("(SCRIPT) Error, null vector encountered", 4);
      return;
    }

    Enumeration e = v.elements();
    while (e.hasMoreElements())
      updateColor(e.nextElement(), s);
  }

  /**
   * Creates the display for the script
   */
  public synchronized void makeDisplay(Container c) {
    Enumeration e;
    Label l;

    // Drop old stuff
    c.removeAll();

    // Add itself
    c.add(l = new Label(this.toString()));
    labels.put(this, l);

    // Add assertions
    e = assertions.elements();
    while (e.hasMoreElements()) {
      Assertion a = (Assertion)e.nextElement();
      c.add(l = new Label("   < " + a));
      labels.put(a, l);
    }

    // Add reactions
    e = reactions.elements();
    while (e.hasMoreElements()) {
      Reaction r = (Reaction)e.nextElement();
      c.add(l = new Label("   > " + r));
      labels.put(r, l);
    }

    // Update
    c.validate();
  }

  /**
   * Stringify
   */
  public String toString() {

    return "[Script] " + getName() +
      " (A:" + assertions.size() +
      " R:" + reactions.size() +
      " F:" + activations +
      ")";
  }
}
