/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ResourceLevel.java
 ***************************************************************************************/

package simulator.script;

import simulator.*;

import simulator.locale.*;

/**
 * Tests the current level of a resource in the simulation environment.
 * Comparison test is "CurLevel Op Level" (e.g. CurTime < GivenTime).
 * <P>
 * <B>Data</B><BR>
 * <UL>
 * <LI>Resource:String - The name of the resource to check
 * <LI>Op:String - Comparison operator (e.g. "==", ">", "<", ">=", "<=", "!=")
 * <LI>Level:Float - The target level to compare the current level with.  If
 * Level is set to "Max" or "Min" the level will be compared to the 
 * maximum and minimum values for the resource, respectively.
 * </UL>
 */
public class ResourceLevel extends Assertion {
  String op;
  float level;
  Resources rsrc;

  /**
   * Constructor.
   */
  public ResourceLevel(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get the operator
    op = (String)getData().get("Op");

    // Get the level
    String str = (String)getData().get("Level");
    if (str.equalsIgnoreCase("Max"))
      level = Float.POSITIVE_INFINITY;
    else if (str.equalsIgnoreCase("Min"))
      level = Float.NEGATIVE_INFINITY;
    else
      level = Float.valueOf(str).floatValue();

    // Get the resource
    rsrc = findResource();

    super.init();
  }

  /**
   * Finds the resource object
   */
  public Resources findResource() {
      utilities.Log logger=utilities.Log.getDefault();
      String rsc=(String)getData().get("Resource");
      try {
	  return (Resources)Simulator.getSimulator().getResource(rsc);
      } catch (simulator.locale.Locale.ResourceDoesNotExist x) {
	  logger.log("(SCRIPT) Error in Resource path "+rsc, 1);
	  return null;
      }
  }

  /**
   * Checks the assertion.
   */
  public boolean check() {

    if (rsrc == null)
      findResource();
    if (rsrc == null)
      return false;

    float rlevel = rsrc.getCurrentValue();
    float clevel = level;

    if (clevel == Float.POSITIVE_INFINITY)
      clevel = rsrc.getMaximum();
    else if (clevel == Float.NEGATIVE_INFINITY)
      clevel = rsrc.getMinimum();

    if (op.equals("==")) {
      return (rlevel == clevel);

    } if (op.equals(">")) {
      return (rlevel > clevel);

    } if (op.equals("<")) {
      return (rlevel < clevel);

    } if (op.equals(">=")) {
      return (rlevel >= clevel);

    } if (op.equals("<=")) {
      return (rlevel <= clevel);

    } if (op.equals("!=")) {
      return (rlevel != clevel);
    }

    return false;
  }
}
