/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ExpressionScript.java
 ***************************************************************************************/

package simulator.script;

/* Global Includes */
import java.util.*;

/* Local Includes */
import utilities.Log;

/**
 * This script allows the author to combine assertions in an arbitrary
 * manner, as defined by a simple boolean expression.  The BNF notion
 * of the grammer supported by this engine is as follows:
 * <P>
 * <PRE>
 *     Exp := '(' Term ')'
 *    Term := # BinOp #
 *    Term := UnOp #
 *   BinOp := '|' | '&'
 *    UnOp := '!'
 *       # := <integer assertion symbol 0..n>
 *       # := Exp
 * </PRE>
 * <P>
 * The binary and unary operator(s) have their usual meanings.  The integer
 * elements symbolize assertions added to the script.  The first assertion
 * is 0, the second is 1, third is 2, etc, so unlike some other scripts
 * the order of the assertions in the config file is important.  When
 * pulsed, the script first determines the check values for each assertion
 * and then replaces their symbolic placeholders in the expression with
 * those check values.  The evaluation of the resulting expression controls
 * weather or not the script fires.
 * <P>
 * <DD>Ex. <TT>((0 & 1) | ((0 & 3) & (! 2)))</TT>
 * <P>
 * <B>Data:</B><BR>
 * <UL>
 * <LI>Expression:String - A fully parenthesized boolean expression using
 * the above syntax.
 * <LI>Fire:Integer - The number of times the script may fire.  Value must be
 * an integer or "*", with "*" indicating the script may fire indefinately.
 * (optional, defaults to 1)
 * </UL>
 */
public class ExpressionScript extends AndScript {
  Vector expression;

  /**
   * Constructor
   */
  public ExpressionScript(String n, String d) {
    super(n, d);
  }
  
  /**
   * Init
   */
  public void init() {
    Hashtable data = getData();
    
    // Get the short circuit op
    expression = infixToPostfix(stringToStack((String)data.get("Expression")));
    if (expression == null)
      logger.log("Error parsing expression: " + data.get("Expression"), 0);

    super.init();
  }

  /**
   * Converts the string of characters to a stack of tokens (Strings)
   * @param The intput String
   * @return The output Stack, or null if error
   */
  public Stack stringToStack(String str) {
    Stack exp = new Stack();
    StringBuffer buf = null;
    char chars[] = str.toCharArray();

    for (int i = chars.length - 1; i >= 0; i--) {
      char c = chars[i];

      switch (c) {
	// Single length tokens
      case '&':
      case '|':
      case '!':
      case '(':
      case ')':
	if (buf != null) {
	  exp.push(buf.toString());
	  buf = null;
	}
	exp.push(String.valueOf(c));
	break;

	// Numbers
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	if (buf != null) {
	  buf = buf.insert(0, c);
	} else
	  buf = new StringBuffer(String.valueOf(c));
	break;

	// Whitespace
      case ' ':
      case '\t':
	if (buf != null) {
	  exp.push(buf.toString());
	  buf = null;
	}
	break;

      default:
	logger.log("Unknown token \'" + c + "\' in expression", 0);
	return null;
      }
    }

    return exp;
  }

  /**
   * Converts and infix expression to a postfix one.
   * @param s The stack of tokens
   * @return The converted vector of tokens, or null if error
   */
  public Vector infixToPostfix(Stack s) {
    Vector v = new Vector(10);
    Stack tmp = new Stack();

    if (s == null) return null;

    try {
      while (! s.empty()) {
	String str = (String)s.pop();
	char c = (str.toCharArray())[0];

	switch (c) {
	  // Numbers
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  v.addElement(Integer.valueOf(str));
	  break;
	  
	  // Operators
	case '&':
	case '|':
	case '!':
	case '(':
	  tmp.push(str);
	  break;
	case ')':
	  v.addElement(tmp.pop());
	  tmp.pop();
	  break;
	  
	default:
	  logger.log("Unexpected token \'" + str + "\' in expression", 0);
	  return null;
	}
      }
    } catch (EmptyStackException e) {
      logger.log("Error parsing expression", 0);
      return null;
    }

    return v;
  }

  /**
   * Converts the symbolic expression into an actual stack
   * @param exp The postfix expression
   * @param res The listing of term results
   * @return The resulting expression
   */
  public Stack symbolicToActual(Vector exp, Vector res) {
    Stack stack = new Stack();

    if (exp == null) return null;
    if (res == null) return null;

    for (int idx = exp.size() - 1; idx >= 0; idx--) {
      Object o = exp.elementAt(idx);
      
      if (o instanceof Integer) {
	// Convert symbols to their boolean results
	int i = ((Integer)o).intValue();
	if ((res.size() > i) && (res.elementAt(i) != null))
	  stack.push(res.elementAt(i));
	else {
	  logger.log("Value for assertion " + i + " not found", 0);
	  return null;
	}
      
      } else {
	// Push everything else
	stack.push(o);
      }
    }

    return stack;
  }

  /**
   * Evaluates the postfix expression, given the expression and
   * a Vector of truth (Boolean) values
   * @param exp The postfix expression
   * @return True if the expression is true, false otherwise
   */
  public boolean evaluatePostfix(Stack exp) {
    Stack tmp = new Stack();
    Boolean a, b;

    if (exp == null) return false;

    try {
      while (! exp.empty()) {
	Object o = exp.pop();

	if (o instanceof Boolean) {
	  // Data - push it
	  tmp.push(o);

	} else {
	  // Operator - pop the right number and evaulate, push results
	  char c = (((String)o).toCharArray())[0];
	  
	  switch (c) {
	  case '&':
	    a = (Boolean)tmp.pop();
	    b = (Boolean)tmp.pop();
	    tmp.push(new Boolean(a.booleanValue() && b.booleanValue()));
	    break;
	  case '|':
	    a = (Boolean)tmp.pop();
	    b = (Boolean)tmp.pop();
	    tmp.push(new Boolean(a.booleanValue() || b.booleanValue()));
	    break;
	  case '!':
	    a = (Boolean)tmp.pop();
	    tmp.push(new Boolean(! a.booleanValue()));
	    break;
	  
	  default:
	    logger.log("Unknown token \'" + o + "\' in expression", 0);
	    return false;
	  }
	}
      }

      return ((Boolean)tmp.pop()).booleanValue();

    } catch (EmptyStackException e) {
      logger.log("Error evaluating expression", 0);
      return false;
    }
  }

  /**
   * Checks the assertions.  If one is true, the reactions are
   * fired. 
   */
  public void pulse() {

    if ((fire != Integer.MAX_VALUE) && (getActivations() >= fire)) {
      updateColor(assertions, Script.UNKNOWN);
      super.pulse();
      return;
    }

    Enumeration e = assertions.elements();
    boolean react = true;

    logger.log("(SCRIPT) " + getName() + " checking assertions", 4);
    
    // Create the results
    Vector res = new Vector();
    while (e.hasMoreElements()) {
      Assertion a = (Assertion)e.nextElement();
      Boolean b = new Boolean(a.check());
      if (b.booleanValue()) {
	updateColor(a, Script.ACTIVE);
	logger.log("(SCRIPT) Assertion " + a + " passed", 5);
      } else {
	updateColor(a, Script.NORMAL);
	logger.log("(SCRIPT) Assertion " + a + " failed", 5);
      }
      res.addElement(b);
    }
    
    // Make the expression
    Stack stack = symbolicToActual(expression, res);
    logger.log("(SCRIPT) Expresssion " + stack + "...", 5);
    react = evaluatePostfix(stack);
    logger.log("(SCRIPT) ...is " + react, 5);

    if (react)
      start();
    else
      super.pulse();
  }
}
