/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ExecuteEventDimension.java
 ***************************************************************************************/

package simulator.script;

import utilities.Clock;
import taems.*;
import simulator.*;
import java.util.*;

/**
 * Determines if a certain execute event is active.
 * Comparison test is "CurValue Op Value" (e.g. CurQuality < GivenQuality).
 * <P>
 * <B>Data</B><BR>
 * <LI>Method:String - The name of the method to look for
 * <LI>Op:String - Comparison operator (e.g. "==", ">", "<", ">=", "<=", "!=")
 * <LI>Dimension:String - The dimension to look at (e.g. quality, cost,
 * duration, resources..)
 * <LI>Value:Float - The value to compare the current value to
 * </UL>
 */
public class ExecuteEventDimension extends ExecuteEventActive {
  protected String op, dim;
  protected float value;

  /**
   * Constructor.
   */
  public ExecuteEventDimension(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get the operator
    op = (String)getData().get("Op");

    // Get the dimension
    dim = (String)getData().get("Dimension");
    
    // Get the value
    value = Float.valueOf((String)getData().get("Value")).floatValue();

    super.init();
  }

  /**
   * Checks the assertion.
   */
  public boolean check() {

    ExecuteEvent e = findExecuteEvent(name, agent);
    if (e != null) {
    
      float total;
      if (dim.equalsIgnoreCase("duration")) {
	total = e.getStart() - Clock.getTime();
      } else
	total = e.getDimension(dim);

      if (total != Float.NaN) {
	if (op.equals("==")) {
	  return (total == value);
	  
	} if (op.equals(">")) {
	  return (total > value);
	  
	} if (op.equals("<")) {
	  return (total < value);
	  
	} if (op.equals(">=")) {
	  return (total >= value);
	  
	} if (op.equals("<=")) {
	  return (total <= value);
	  
	} if (op.equals("!=")) {
	  return (total != value);
	}
      }
    }

    return false;
  }
}
