/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ExecuteEventActive.java
 ***************************************************************************************/

package simulator.script;

import utilities.Clock;
import taems.*;
import simulator.*;
import java.util.*;

/**
 * Determines if a certain execute event is active.
 * <P>
 * <B>Data</B><BR>
 * <LI>Method:String - The name of the method to look for
 * <LI>Agent:String - The name of the agent owning the method (optional)
 * </UL>
 */
public class ExecuteEventActive extends Assertion {
  protected String name;
  protected String agent = null;
  private EventQueue queue;

  /**
   * Constructor.
   */
  public ExecuteEventActive(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get the method name
    name = (String)getData().get("Method");
    
    // Get the agent
    agent = (String)getData().get("Agent");
    
    // Get the queue
    queue = EventQueue.getQueue();

    super.init();
  }

  /**
   * Checks the assertion.
   */
  public boolean check() {

    return eventActive(findExecuteEvent(name, agent));
  }

  /**
   * Determines if the event is active (starttime >= time && finishtime <= time)
   * @param id The id number to search for.
   * @return True if the event is active
   */
  public boolean checkByID(long i) {

    Event e = queue.findEvent(i);
    return eventActive(e);
  }

  /**
   * Determines if the event is active (starttime >= time && finishtime <= time)
   * @param e The event to check
   * @return True if the event is active
   */
  public boolean eventActive(Event e) {

    if (e != null) {
      int time = Clock.getTime();
      if ((e.getStart() >= time) && (e.getFinish() <= time))
	return true;
    }

    return false;
  }

  /**
   * Determines if the event is starting (starttime == time)
   * @param e The event to check
   * @return True if the event is starting
   */
  public boolean eventStarting(Event e) {

    if (e != null) {
      int time = Clock.getTime();
      if (e.getStart() == time)
	return true;
    }

    return false;
  }

  /**
   * Determines if the event is completing (finishtime == time)
   * @param e The event to check
   * @return True if the event is completing
   */
  public boolean eventCompleting(Event e) {

    if (e != null) {
      int time = Clock.getTime();
      if (e.getFinish() == time)
	return true;
    }

    return false;
  }

  /**
   * Searches for a specific action in the current time slot.  Uses
   * TTaems.tasksMatch to determine if the method names match.
   * <P>
   * If this method is slowing everything down, let me know and I'll
   * try to add some sort of event caching thing.
   * @param name The name of the action to search for
   * @return The action, or null if not found
   */
  public ExecuteEvent findExecuteEvent(String name, String agentname) {

    if (!queue.hasTimeSlot(Clock.getTime())) return null;

    Vector v = queue.getTimeSlot(Clock.getTime());
    Enumeration en = v.elements();

    taems.Agent agent = new taems.Agent(agentname);
    taems.Node tmp = new Node(name, agent);

    while (en.hasMoreElements()) {
      Event e = (Event)en.nextElement();
      if (e instanceof ExecuteEvent) {
	//	if (Taems.findNode(new Method(((ExecuteEvent)e).methodName, name))
	if ((((ExecuteEvent)e).getMethod().matches(tmp)))
	  return (ExecuteEvent)e;
      }
    }

    return null;
  }
}
