/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ExecuteEventAbort.java
 ***************************************************************************************/

package simulator.script;

import utilities.Clock;
import taems.*;
import simulator.*;
import java.util.*;

/**
 * Aborts a certain execute event.
 * <P>
 * <B>Data</B><BR>
 * <LI>Method:String - The name of the method to look for
 * </UL>
 */
public class ExecuteEventAbort extends Reaction {
  protected String name;
  protected String agent = null;
  private EventQueue queue;

  /**
   * Constructor.
   */
  public ExecuteEventAbort(String d) {
    super(d);
  }
  
  /**
   * Init.
   */  
  public void init() {
    
    // Get the method name
    name = (String)getData().get("Method");
    
    // Get the agent
    agent = (String)getData().get("Agent");  

    // Get the queue
    queue = EventQueue.getQueue();

    super.init();
  }

  /**
   * Starts the reaction
   * @return True if it was sucessfull, false otherwise
   */
  public boolean realize() {

    return abortEvent(findExecuteEvent(name, agent));
  }

  /**
   * Attempts to extend an event
   * @param e The event to check
   * @param d The amount to extend by
   * @return True if the event was extended
   */
  public boolean extendEvent(Event e, int d) {

    if (e != null) {
      return queue.extendEvent(e, d);
    }

    return false;
  }

  /**
   * Attempts to delay an event
   * @param e The event to check
   * @param d The amount to delay by
   * @return True if the event was delayed
   */
  public boolean delayEvent(Event e, int d) {

    if (e != null) {
      return queue.delayEvent(e, d);
    }

    return false;
  }

  /**
   * Attempts to abort an event
   * @param e The event to check
   * @return True if the event was aborted
   */
  public boolean abortEvent(Event e) {

    if (e != null) {
      e.abort(String.valueOf(e.getID()));
      queue.removeEvent(e);
      return true;
    }

    return false;
  }

  /**
   * Searches for a specific action in the current time slot.  Uses
   * TTaems.tasksMatch to determine if the method names match.
   * <P>
   * If this method is slowing everything down, let me know and I'll
   * try to add some sort of event caching thing.
   * @param name The name of the action to search for
   * @return The action, or null if not found
   */
  public ExecuteEvent findExecuteEvent(String name, String agentname) {

    if (!queue.hasTimeSlot(Clock.getTime())) return null;

    Vector v = queue.getTimeSlot(Clock.getTime());
    Enumeration en = v.elements();

    taems.Agent agent = new taems.Agent(agentname);
    taems.Node tmp = new Node(name, agent);

    while (en.hasMoreElements()) {
      Event e = (Event)en.nextElement();
      if (e instanceof ExecuteEvent) {
	//	if (Taems.findNode(new Method(((ExecuteEvent)e).methodName, name))
	if ((((ExecuteEvent)e).getMethod().matches(tmp)))
	  return (ExecuteEvent)e;
      }
    }

    return null;
  }
}
