/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Assertion.java
 ***************************************************************************************/

package simulator.script;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.awt.*;

/* Local Includes */
import simulator.*;
import utilities.*;

/**
 * The base class for all Script Assertions - those things or states which
 * indicate that a script may activate itself.  Each Script will keep track
 * of one or more Assertions during its lifetime.  As the Assertions change
 * state (are true or false), the semantics of the Script dictate what
 * reactions should take place.
 * <P>
 * The pertinant method here is check, which is periodically called by 
 * Script to determine the state of the Assertion.
 * <P>
 * By convention, we hope that event authors place the supported data
 * descriptions in the header comments for their classes.  Take a look at
 * some of the other source files for examples of how to do this.
 */
public class Assertion {
  private Hashtable data;

  /**
   * Constructor.
   */
  public Assertion(String d) {

    data = Script.parseDataLine(d);
  }
  
  /**
   * Init, usually you will retrieve input properties here, and setup
   * any state you need to.  Called once by the Script the Assertion
   * belongs to, as it is added to the Script.
   * @see utilities.Converter#reTypeProperty
   */  
  public void init() {
  }

  /**
   * Checks the assertion.  Override this method to change the
   * topic of the Assertion.
   * @return True if the state monitored by the Assertion is
   * found to be true, false otherwise.
   */
  public boolean check() {
    return false;
  }

  /**
   * Returns the input data table
   */
  public Hashtable getData() { return data; }

  /**
   * Returns the name of the object
   */
  public String getName() {
	String name = getClass().getName();
	return name.substring(name.lastIndexOf(".") + 1);
  }

  /**
   * Stringify
   */
  public String toString() {
	StringBuffer sb = new StringBuffer(getName() + " - ");
	Enumeration e = getData().keys();
	while (e.hasMoreElements()) {
		Object key = e.nextElement();
		sb.append(key + ":" + getData().get(key));
		if (e.hasMoreElements())
			sb.append("; ");
	}
	return sb.toString();
  }
}
