/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	AndScript.java
 ***************************************************************************************/

package simulator.script;

/* Global Includes */
import java.util.*;

/* Local Includes */
import utilities.Log;

/**
 * This class uses the AND semantics for Assertion checking - they 
 * must all be true for the reactions to fire.
 * <P>
 * <B>Data:</B><BR>
 * <UL>
 * <LI>ShortCircuit:Boolean - If this flag is present, the check call will 
 * not stop checking after the first true is detected. An explicit
 * ShortCircuit token may also be used. (optional, defaults to True) Note that
 * when you use ShortCircuit, some assertions will not be checked, so you may
 * decrease analysis time through intelligent ordering of your assertions (e.g.
 * put the least likely assertion first)
 * <LI>Fire:Integer - The number of times the script may fire.  Value must be
 * an integer or "*", with "*" indicating the script may fire indefinately.
 * (optional, defaults to 1)
 * </UL>
 */
public class AndScript extends Script {
  boolean sc = true;
  int fire = 1;

  /**
   * Constructor
   */
  public AndScript(String n, String d) {
    super(n, d);
  }
  
  /**
   * Init
   */
  public void init() {
    Hashtable data = getData();
    
    // Get the short circuit op
    if (data.containsKey("ShortCircuit")) {
      sc = Boolean.valueOf((String)data.get("ShortCircuit")).booleanValue();
    }

    // Fires
    if (data.containsKey("Fire")) {
      String str = (String)data.get("Fire");
      if (str.equalsIgnoreCase("*"))
	fire = Integer.MAX_VALUE;
      else if (str != null)
	fire = Integer.parseInt(str);
    }

    super.init();
  }

  /**
   * Checks the assertions.  If one is true, the reactions are
   * fired. 
   */
  public void pulse() {

    if ((fire != Integer.MAX_VALUE) && (getActivations() >= fire)) {
      updateColor(assertions, Script.UNKNOWN);
      super.pulse();
      return;
    }

    Enumeration e = assertions.elements();
    boolean react = true;

    logger.log("(SCRIPT) " + getName() + " checking assertions", 4);
    
    while (e.hasMoreElements()) {
      Assertion a = (Assertion)e.nextElement();
      if (a.check()) {
	updateColor(a, Script.ACTIVE);
	logger.log("(SCRIPT) Assertion " + a + " passed", 5);

      } else {
	updateColor(a, Script.NORMAL);
	logger.log("(SCRIPT) Assertion " + a + " failed", 5);
	react = false;
	if (sc && (e.hasMoreElements())) {
	  logger.log("(SCRIPT) Short circuiting AND", 5);
	  while (e.hasMoreElements())
	    updateColor(e.nextElement(), Script.UNKNOWN);
	  break;
	}
      }
    }

    if (react)
      start();
    else
      super.pulse();
  }
}
