/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
 * $Source: /nfs/shelob/cvs/masl/Mass/modules/simulator/simulator/locale/Locale.java,v $
 * $Revision: 1.2 $
 ******************************************************************
 *
 *                   Locale.java
 *
 *     Author : Peter Amstutz [amstutz@cs.umass.edu]
 *              
 *
 * Creation date: 15 June 99 12:43
 * Last file update: $Date: 2005/01/19 19:00:06 $
 *****************************************************************/

package simulator.locale;

import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.awt.*;
import java.io.*;
import java.lang.Class.*;

import simulator.*;
import simulator.locale.*;
import utilities.*;
import utilities.cfg.*;

/** 
 * <B>Module</B>: simulator.locale<p>
 * <B>Copyright</B>: UMASS - MASL 1999<P>
 * @version $Revision: 1.2 $
 * @author  Peter Amstutz (amstutz@cs.umass.edu)<P>
 * <B> Description:</B>
 * This is the base class for locales and linkages for the locale
 * resource model.  Each instance is linked to certain resources
 * associated with that locale, in addition, Locales may contain
 * locales (sublocales) and link to other locales via a special
 * subclass of locale called a linkage.  An API is provided for
 * working with the data encapsulated in this object.  
*/
public class Locale extends ContainerGraphNode implements ConfiguredObject
{ 
    private int prop_mark=0;
    private Hashtable sublocales = new Hashtable();
    private Hashtable linkages = new Hashtable();
    private Hashtable resources = new Hashtable();

    protected String name;
    
    protected Locale parent;
    protected GraphEdge edgetoparent = null;
    utilities.Log log=utilities.Log.getDefault();
   
    /**
     * This blank class is used for signaling an exception (that this
     * locale and none of its sublocales contain this resource) in the
     * resource search method 
    */
    public class ResourceDoesNotExist extends Exception { 
		public ResourceDoesNotExist() { super(); }
    };

    /**
     * This blank class is used for signaling an exception (that this
     * locale and none of its sublocales contain this as a sublocale)
     * in the locale search method 
    */
    public class LocaleDoesNotExist extends Exception { 
		public LocaleDoesNotExist() { super(); }
    };

    /**
     * Basic constructor, does some generic initialization.  Mostly
     * here to keep other methods happy that seem to expect a
     * constructor with no arguments and raises a fuss if they don't
     * find one -- this includes subclasses that call their
     * superclass's constructor.  
    */
    public Locale() {
		super();
		name="";
		setParent(null);
		edgetoparent = null;
    }
    
    /**
       The actual useful constructor.  Calls the simple constructor to
       set up a (blank) display panel, sets name, parent (very
       important to know!) and draws to the panel.  
       @param <b>nm</b> the name
       @param <b>par</b> the parent Locale
    */
    public Locale (String nm, Locale par) {
		this();
		name=nm;
		setLabel(name);
		setParent(par);
    }
   
    /**
     * Sets the parent, and does some graph stuff
     */
    public void setParent(Locale p)
    {
        parent = p;
        if (edgetoparent != null)
        {
            edgetoparent.excise();
        }
        if (parent == null)
            edgetoparent = null;
        else
            edgetoparent = new GraphEdge("", parent, this);
    }
    
    /** 
		Makes a resource object be considered to be "in" this locale,
		that is, things local to this locale will affect it.  
		@param <b>n</b> The resource name, and will be used as a key to
		find this resource in the search functions
		@param <b>rsc</b> The resource object.  Because it is of the
		generic type object, we are by no means restricted to storing
		just instances of class "Resources" here -- other metadata
		(perhaps to augument the LocationMapping function) can be
		stored here as well 
    */
    public void addResource(String n, Object rsc) {
		if(!resources.containsKey(n))
		{
			resources.put(n, rsc);
            ContainerNodePanel p = (ContainerNodePanel)
                Simulator.getSimulator().getGraph().getLayers().getTopPanel(this);
            if (p != null)
                addContained(p, (GraphNode) rsc);
		}
    }

    /** 
		Creates a linkage object from this instance object to another.
		@param <b>nm</b> The name of the linkage
		@param <b>dst</b> The destination locale
		@param <b>type</b> The linkage "type" - a subclass of linkage that
		is dynamically loaded.  In IHome, for example, this might be a
		door or a window -- specifically the "Door" class which
		subclasses Linkage.  The prefix simulator.locale is implicit
		and there should be no ".class" or ".java" suffix, just the
		class name, i.e. Door class, simulator/locale/Door.java,
		simulator/locale/Door.class, or simulator.locale.Door should
		all be refered to as simply "Door" 
		@return The new linkage created.
    */
    public Linkage linkageTo(String nm, Locale dst, String type) {
		if(!linkages.containsKey(nm)) {
			Object[] args = new Object[3];
			Class[] cargs = new Class[3];
			args[0]=nm;
			args[1]=this;
			args[2]=dst;
			cargs[0]=String.class;
			cargs[1]=Locale.class;
			cargs[2]=Locale.class;
			Linkage ln = (Linkage)LoadClass.load("simulator.locale."+type, args, cargs);
			if(ln!=null) linkages.put(nm, ln);
			
			dst.addLinkage(ln);
			
			return ln;
		}
		return null;
    }

	public void addLinkage(Linkage l) {
		if(!linkages.containsKey(l.getName())) {
			linkages.put(l.getName(), l);
		}
	}

    /** 
		This searches for a resource of a given name.  First the
		current instance is searched, then the parent is searched,
		then the parent's parent and so on outward (if thinking in
		terms of venn diagrams) or up the tree until reaching the
		global locale.
		@param <b>rsc</b> is the resource name/key giving in addResource()
		@return the object found with that key
		@exception ResourceDoesNotExist is thrown if the resource cannot be found 
	*/
    public Object getResource(String rsc) throws ResourceDoesNotExist {
		StringTokenizer e = new StringTokenizer(rsc, "/");
		if(e.countTokens()>1) {
			String s;
			Object r=null;
			Locale lcur=null;
			while(e.hasMoreTokens()) {
				s=e.nextToken().trim();
				try {
					lcur=getSubLocale(s);
				} catch (LocaleDoesNotExist x) {
					try {
						r=lcur.getResource(s);
					} catch (ResourceDoesNotExist y) {
						if(!rsc.equals("")) 
							throw new ResourceDoesNotExist();
					}
				}
			}
			if(r==null) {
				throw new ResourceDoesNotExist();
			}
			return r;
		}
		else {
			if(resources.containsKey(rsc.trim()))
				return resources.get(rsc.trim());
			else if(parent!=null)
				return parent.getResource(rsc.trim());
			else throw new ResourceDoesNotExist();
		}
    }
    /**
       Name accessor
       @return locale name
	*/
    public String getName() { 
		return name; 
    }

    /**
       Creates a new sublocale.
       @param <b>nm</b> is the new locales's name and its key in the sublocale table
       @param <b>type</b> is the new locale type, the class name of some
       class that is a subclass Locale.  See the discssion for the type 
       parameter of linkageTo() for more information on what exactly is expected.
       @return the locale object that has just been created
       @see simulator.locale.Locale#linkageTo
    */
    public Locale newSubLocale(String nm, String type) {
		Object[] args = new Object[2];
		args[0]=nm;
		args[1]=this;
		Locale l = (Locale)LoadClass.load("simulator.locale."+type, args); // new Locale(nm, this);
		if(l!=null) sublocales.put(nm, l);
		return l;
    }

	/**
	   Finds a sublocale.  Parses a slash-delimited path
	   (ex: Global/Livingroom) indicating the path to follow (that is,
	   start at the Global locale, and return Livingroom sublocale).
	   If a locale name isn't found in the current sublocale table,
	   then each sublocale will be recursively searched.  This makes
	   it posible to have "shortcuts", that is Global/Livingroom can
	   be referred to as just "Livingroom" <i>iff</i> the Livingroom
	   locale name is unique -- this shortcut can be used anywhere in
	   the path, although for clarity should probably only be used
	   when if is quite obvious what locale is intended.
	   @param <b>nm</b> is the locale name or path searched for
	   @return the locale that matches this name or path
	   @exception LocaleDoesNotExist is thrown when the locale cannot
	   be found.  
	*/
    public Locale getSubLocale(String nm) throws LocaleDoesNotExist {
		StringTokenizer e = new StringTokenizer(nm, "/");	
		if(e.countTokens()>1) {
			String s;
			Locale lcur=this;
			while(e.hasMoreTokens()) {
				s=e.nextToken();
				lcur=lcur.getSubLocale(s);
			}
			if(lcur==null) {
				throw new LocaleDoesNotExist();
			}
			return lcur;
		}
		else {
			if(sublocales.containsKey(nm))
				return (Locale)sublocales.get(nm);
			else {
				Enumeration sl = getSubLocales();
				while(sl.hasMoreElements()) {
					try {
						return ((Locale)sl.nextElement()).getSubLocale(nm);
					} catch (LocaleDoesNotExist foo) { }
				}
				throw new LocaleDoesNotExist();
			}
		}
    }

	/**
	   @return an enumeration of the resource table for this locale
	*/
    public Enumeration getResources() {
		return resources.elements();
    }

	/**
	   @return an enumeration of the sublocale table for this locale
	*/
    public Enumeration getSubLocales() {
		return sublocales.elements();
    }

	/**
	   @return an enumeration of the linkage table for this locale
	*/
    public Enumeration getLinkages() {
		return linkages.elements();
    }

	/**
	   @return The parent Locale node for this locale, if it exists (null otherwise)
	*/
	public Locale getParent() {
		return parent;
	}
    
    public boolean matches(Locale l)
    {
        if (l == null)
            return true;
        if (!getName().startsWith(l.getName()))
            return false;
        if (l.getClass().isInstance(this))
            return true;
        return false;
    }

	/**
	   @return the propagation mark on this node
	*/
    public boolean getPropMark() {
		return (prop_mark > 0);		
    }

	/**
	   sets the propagation mark on this node
	*/
    public void setPropMark() {
		prop_mark++;
    }

	/**
	   clears the propagation mark on this node
	*/
    public void clearPropMark() {
		if(prop_mark>0) prop_mark--;
    }

	/**
	   To better model interdependent resource networks where a change
	   in one will affect another, when a resource changes you will
	   want to call this method.  This calls propagateResourceChange()
	   on the sublocales, linkages, and parent locales, the idea being
	   to recursivly touch every node that this touchs.  Each node
	   touched then has an opportunity to update its resources to
	   reflect the resource change.  For example, noise in one room
	   will be heard in (at decreasing levels) in connecting rooms.
	   Marks are placed on each visited node to avoid cycles.
	   @param <b>src</b> The Locale from which pRC() was called from.
	   This is to allow for implementing gated linkages (which might
	   allow a pRC() through one way, but not the other) 
	   @param <b>changed</b> The object that has changed
	   @param <b>how</b> Some object indicating _how_ the object has
	   changed.  For example, for a resource change this might be a
	   Float of the magnatude of the change -- such as noise level
	   changing from 30 to 20 would be -10.  
	*/
    public void propagateResourceChange(Locale src, Object changed, Object how, simulator.Event ev) {
		// subclasses which implement representations
		// of locales (roooms, network hosts, etc)
		// (optionally) get some information from
		// the change object and use it to possibly
		// update their own status.

	//		log.log("(LOCALE) propagateResourceChange touched "+name, 4);

		setPropMark();

		Enumeration e = getSubLocales();
		while(e.hasMoreElements()) {
			Locale fool =  (Locale)e.nextElement();
			if(! fool.getPropMark()) 
				fool.propagateResourceChange(this, changed, how, ev);
		}

		e = getLinkages();
		while(e.hasMoreElements()) {
			Linkage fool =  (Linkage)e.nextElement();
			fool.propagateResourceChange(this, changed, how, ev);
		}

		if(parent!=null && !parent.getPropMark()) 
			parent.propagateResourceChange(this, changed, how, ev);
    }

	/**
	   This should be called after propagateResourceChange() is
	   finished doing its business.  It walks the tree, cleaning up
	   all the marks so the next propagateResourceChange() run doesn't
	   get confused.
	*/
    public void clearPropMarks() {
		clearPropMark();

		Enumeration e = getSubLocales();
		while(e.hasMoreElements()) {
			Locale fool =  (Locale)e.nextElement();
			if(fool.getPropMark()) 
				fool.clearPropMarks();
		}

		e = getLinkages();
		while(e.hasMoreElements()) {
			Linkage fool =  (Linkage)e.nextElement();
			fool.clearPropMarks();
		}	

		if(parent!=null && !parent.getPropMark()) 
			parent.clearPropMarks();
    }
    
    /**
     * returns a string representing the contents of the object, of course
     */
    public String toString()
    {
        String s = getLabel() + "\n";
        Enumeration e = getContained();
        while (e.hasMoreElements())
        {
            s = s + e.nextElement().toString() + "\n";
        }
        return s;
    }

	/**
	   This function is distinct from propagateResourceChange() for
	   one very important reason: propagateEnvironmentalModel() will
	   be called from the global locale exactly once every time unit,
	   and is intended for modeling persistant or environmental
	   effects.  
	   @param <b>src</b> The Locale from which pEM() was called from.
	   This is to allow for implementing gated linkages (which might
	   allow a pEM() through one way, but not the other) 
	*/
	public void propagateEnvironmentalModel(Locale src) {
		// subclasses which implement representations
		// of locales (roooms, network hosts, etc)
		// want to use this to model environmental
		// effects that should be calculated exactly once
		// each time pulse, after all agent events have
		// been processed

		log.log("(LOCALE) propagateEnvironmentalModel touched "+name, 4);

		setPropMark();
		
		Enumeration e = getSubLocales();
		while(e.hasMoreElements()) {
			Locale fool =  (Locale)e.nextElement();
			if(! fool.getPropMark()) 
				fool.propagateEnvironmentalModel(this);
		}

		e = getLinkages();
		while(e.hasMoreElements()) {
			Linkage fool =  (Linkage)e.nextElement();
			fool.propagateEnvironmentalModel(this);
		}

		if(parent!=null && !parent.getPropMark()) 
			parent.propagateEnvironmentalModel(this);
	}

    public GraphEdge getEdgeToParent()
    {
        return edgetoparent;
    }
}
