/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
 * $Source: /nfs/shelob/cvs/masl/Mass/modules/simulator/simulator/locale/Linkage.java,v $
 * $Revision: 1.2 $
 ******************************************************************
 *
 *                   Linkage.java
 *
 *     Author : Peter Amstutz [amstutz@cs.umass.edu]
 *              
 *
 * Creation date: 15 June 99 12:43
 * Last file update: $Date: 2005/01/19 19:00:06 $
 *****************************************************************/

package simulator.locale;

import java.awt.*;
import utilities.*;
import utilities.VFlowLayout;
import java.util.Enumeration;
import java.util.Vector;

/** 
 * <B>Module</B>: simulator.locale<p>
 * <B>Copyright</B>: UMASS - MASL 1999<P>
 * @version $Revision: 1.2 $
 * @author  Peter Amstutz (amstutz@cs.umass.edu)<P>
 * <B> Description:</B>
 *
 * This is the class from which linkages should be derived.  It
 * subclasses from Locale so it has full capability to have its own
 * sublocales, resources and even linkages, but it it not recommended
 * you use this feature too much ;-) Of primary interest should be
 * resources.  For example, a doorway might have a resource value for
 * the rate of air flow between two rooms: when
 * propagateResourceChange() touches this linkage, it can then change
 * the temperature of these rooms based on temperature diffusion (and
 * some ad-hoc physics ;-)
 *
 * Note: Linkages are can be either bidirectional, unidirectional, or
 * n-way directional (one root pointing to many children) it is up to
 * the way your subclasses represent and how they implement
 * propagateResourceChange() to determine which behavior is desired.  */
public class Linkage extends Locale {
    protected Locale src, dst;
    protected GraphEdge edge;
    
	Vector endpoints = new Vector();

	public Linkage() {
	}

	/**
	   A terribly exciting constructor.
	   @param <b>nm</b> is the name
	   @param <b>s</b> is the source Locale
	   @param <b>d</b> is the destination Locale
	 */
    public Linkage(String nm, Locale s, Locale d) {
		name=nm;
		parent=s;
		edge = new GraphEdge(name);
		edge.setDrawArrows(false);
		endpoints.addElement(d);
		
		setLabel(name);
		setFrom(s);
		setTo(d);       
    }
    
    /**
     * Accessors
     */
    public void setTo(Locale l)
    {
        dst = l;
        edge.setTo(l);
    }
    
    public void setFrom(Locale l)
    {
        src = l;
        edge.setFrom(l);
    }
       
    /*public boolean isHidden()
    {
        return edge.isHidden();
    }
    
    public void setHidden(boolean b)
    {
        edge.setHidden(b);
    }*/
    
    public GraphEdge getEdge()
    {
        return edge;
    }
    
	/**
	   A terribly exciting constructor.
	   @param <b>nm</b> is the name
	   @param <b>s</b> is the source Locale
	   @param <b>d</b> is an array of destination Locales
	 */
    public Linkage(String nm, Locale s, Locale[] d) {
		name=nm;
		parent=s;
		for(int i=0; i<d.length; i++) 
			endpoints.addElement(d[i]);
    }

	/** 
		@return The Locale this linkage links to
	*/
    public Enumeration linksTo() {
		return endpoints.elements();
    }

	/** 
		@return The Locale this linkage links from
	*/
    public Locale linksFrom() {
		return parent;
    }

	public Enumeration otherSide(Locale thisside)
	{
		if(parent==thisside) return endpoints.elements();
		else {
			Vector v = new Vector(1);
			v.addElement(parent);
			return v.elements();
		}
	}

    /**
	   Interesting things can be done here in subclasses, however the
	   base case is just to pass the resource change to its
	   destination.  Subclasses which implement connecting objects
	   such as doorways, network connections etc might do something
	   here (such as update their bandwidth depending on the status of
	   connected hosts) then should call
	   super.propagateResourceChange()
	*/
    public void propagateResourceChange(Locale src, Object changed, Object how, simulator.Event ev) {
		if(! parent.getPropMark()) parent.propagateResourceChange(this, changed, how, ev);
		Enumeration e = linksTo();
		while(e.hasMoreElements()) {
			Locale l=(Locale)e.nextElement();
			if(! l.getPropMark()) l.propagateResourceChange(this, changed, how, ev);
		}
    }

    public void clearPropMarks() {
		if(parent.getPropMark()) parent.clearPropMarks();
		Enumeration e = linksTo();
		while(e.hasMoreElements()) {
			Locale l=(Locale)e.nextElement();
			if(l.getPropMark()) l.clearPropMarks();
		}
    }

    public void propagateEnvironmentalModel(Locale src) {
		if(! parent.getPropMark()) parent.propagateEnvironmentalModel(this);
		Enumeration e = linksTo();
		while(e.hasMoreElements()) {
			Locale l=(Locale)e.nextElement();
			if(! l.getPropMark()) l.propagateEnvironmentalModel(this);
		}
    }
}
