/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Simulator.java
 ***************************************************************************************/
package simulator;

/* Global Inclues */
import java.applet.*;
import java.io.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import javax.swing.*;

/* Local Includes */
import simulator.*;

/* Utilities */
import utilities.*;
import utilities.cfg.*;

/* Generator imports */
import taems.*;

import simulator.locale.*;

/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @author Regis Vincent (vincent@cs.umass.edu), Bryan Horling (bhorling@cs.umass.edu)<P>
 * <B> Description:</B>
 * The simulator main class. This class has only one instance and contains 
 * all the variables useful for the simulator.
 */
public class Simulator extends Panel implements  Runnable 
{
    private final static Dimension DIMENSION = new Dimension(670, 460);
    private final static String VERSION = new String(" (C) UMASS/MASL 1997,1998,1999,2000          - MASS Version 1.40 alpha - ");
    private static final int RESOURCE_WINDOW_INITIAL_WIDTH = 750;
    private static final int RESOURCE_WINDOW_INITIAL_HEIGHT = 400;

    private static int TIMEOUT = 300000;
    private static int PORT = 0;
    public static Simulator simulator;
    private static Hashtable agentTable = new Hashtable();
    
    private boolean complexNLEFlag = false; /*Default setup for apply complex NLE*/
    private RootServer server;
    private Scripter scripts;
    private Locale global_locale = new Locale("Global", null);
    private LocationMapping locmap;

    private EventQueue queue;
    private Log logger;
    public TaemsRandom random;
    private boolean running = false;
    private boolean autostart = false;
    private int autostart_agents = 0;
    private boolean text_inter = false;
    private int steps = 0;
    private int agentcount = 0;
    
    private CfgManager cfg;

    /**
     * it's the Sensing mechanism, this is generated by the Sensor
     * configuration file, to simulate agent's sensors. Please
     * check 
     * <A HREF="http://mas.cs.umass.edu/research/mass/simulatorHOWTO.html#sensor"> the sensing page</A>
     */
    private Sensing       globalSensing = new Sensing();

    /**
     * Taems object, this variable contains the complete Taems object.
     */
    public static Taems global_view = new Taems("Objective (Merged View)");

    /**
     * global_view_frame is the Frame window of the simulator. 
     All the GUI is a child of this frame.
     */
    public static JFrame global_view_frame = null;

    /**
     * it's a Vector of all resources defined in the configuration file (
     * <I>simulator.cfg</I>). Please
     * check <A HREF="http://mas.cs.umass.edu/research/mass/simulatorHOWTO.html#configure"> the configuration page</A>
     */
    public Hashtable resources = new Hashtable();

    /*
     * gui variables
     */
    private static SimulatorWindowManager windowmanager = new SimulatorWindowManager();
    private static ProgressBarPanel progress = new ProgressBarPanel();
    private JDialog displayAbout;
    private Graph graph;
    private TextField steptoField;
    private JButton pauseButton, steptoButton, stepButton, quitButton,
    aboutButton, resetButton;
    private Label agentcountlabel;
    private Panel display;
    private Label eventLabel, actionLabel, statusLabel, randomLabel;
    private Integer mergelock = new Integer("1");

    /*-----------------------------------------*
     *           static functions              *
     *-----------------------------------------*/
    
    /**
     * Main function. This function read the configuration file, all generator 
     * grammars and the resources scripting file.
     */
    public static void main(String args[]) 
    {
        if ((args.length > 0) && (args[0].equals("-v"))) 
        {
            System.out.println(Simulator.VERSION);
            Runtime.getRuntime().exit(0);
        }

        progress.display();
        progress.startLoading(0,11);

        progress.showProgress("Initialization...");
        JFrame frame = new JFrame("Agent list");
        String configfile;

        FileInputStream cfgstream = null;
        ConfigFileParser parser;

        if (args.length > 0)
            configfile = args[0];
        else
            configfile = new String("config/simulator.cfg");

        progress.showProgress("Reading Configuration File: " + configfile);
        System.out.println("Reading Configuration File: " + configfile);
        try {
            cfgstream = new FileInputStream(configfile);
        } catch (Exception e) {
            System.err.println(e);
        }

        if (cfgstream == null)
            return;

        progress.showProgress("Parsing configuration file");
        parser = new ConfigFileParser(cfgstream);

        progress.showProgress("Creating Parser");
        try {
            simulator = parser.Input();
        } catch (Exception e) {
            e.printStackTrace();
        }
        simulator.displayEnvironment();
        progress.showProgress("Creating TAEMS Structures");

        frame.getContentPane().add("Center", simulator);
        frame.setSize(DIMENSION);
        windowmanager.addWindow(frame);

        progress.dispose();

        simulator.run();
    }
    
    /**
     * Accessor on the Agent Table
     */
    public static Hashtable getAgentTable() 
    {
        return agentTable;
    }

    /**
     * Returns the current simulator.
     * @return The global simulator object
     */
    public static Simulator getSimulator() 
    {
        return simulator;
    }
    
    /**
     * Util function that adds a component to a GridBag layout
     */
    private static void addComponent (Container container, Component component, 
            int gridx, int gridy, int gridwidth, int gridheight, int fill, 
            int anchor) 
    {

        LayoutManager lm = container.getLayout();
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = gridx;
        gbc.gridy = gridy;
        gbc.gridwidth = gridwidth;
        gbc.gridheight = gridheight;
        gbc.fill = fill;
        gbc.anchor = anchor;
        container.add(component, gbc);
    }
    
    /**
     * getTimeout gives the maximum time that the simulator wait for an 
     * agent.
     */
    public static int getTimeout() 
    { 
        return(TIMEOUT);
    }
    
    /**
     * Get the flag for the simulation, about applying the NLE's using
     * the quality of the facilitator
     */
    public static boolean getComplexNLE() 
    { 
        return (simulator.complexNLEFlag);
    }

    
    /*-----------------------------------------*
     *          non-static functions           *
     *-----------------------------------------*/
    
    /**
     * constructor for the simulator.
     * This initializes most of the simulator, including the GUI and 
     * various internal components.  It should probably never be called outsied
     * of this class, except by someone who really knows what they are doing.
    */
    public Simulator()
    {
        this("sim.cfg");
    }
    
    public Simulator(String cfgfilename) 
    {
        setSize(DIMENSION);
        setLayout(new BorderLayout());
        
        progress.showProgress("Load configuration file(s)");
        //cfg = new CfgManager(configfile);

        progress.showProgress("Logger initialization");
        logger = Log.getDefault();
        if (logger == null) {
            logger = new Log(System.err);
            Log.setDefault(logger);
        }
	logger.setName("MASS");

        progress.showProgress("Random Generator initialization");
        random = TaemsRandom.getDefault();
        if (random == null) {
            random = new TaemsRandom();
            TaemsRandom.setDefault(random);
        }

        progress.showProgress("Graphical User Interface");
        Panel south = new Panel();
        south.setLayout(new GridBagLayout());
        south.setFont (new Font ("Helvetica", Font.PLAIN, 12));
        south.setBackground(Color.white);

        // Add the clock
        Clock.getClock().reset();

        initWindowManager();

        Panel container = new Panel();
        container.setFont (new Font ("Helvetica", Font.BOLD , 12));
        addComponent(south, container, 0, 0, 5, 1, GridBagConstraints.HORIZONTAL, GridBagConstraints.CENTER);
        // And the logo
        try {
            String http="http://mas.cs.umass.edu/images/masl.gif";
            if (System.getProperty("MASL") != null) 
                http=System.getProperty("MASL");
            addComponent(south, new ImageComponent(new URL(http)), 6, 0, 1, 4, GridBagConstraints.HORIZONTAL, GridBagConstraints.EAST);
        } catch (MalformedURLException e) { 
            logger.log("Error finding image: " + e); 
        }
        add("South", south);
        try {
            Panel tempPanel = new Panel();
            tempPanel.setBackground(Color.white);
            String http="http://mas.cs.umass.edu/research/mass/graphics/simulate/simulator.GIF";
            if (System.getProperty("MASS") != null) 
                http=System.getProperty("MASS");
            tempPanel.add(new ImageComponent(new URL(http)));
            add("North", tempPanel);
        } catch (MalformedURLException e) { 
            logger.log("Error finding image: " + e); 
        }
        add("Center", display = new Panel());
        display.setLayout(new VFlowLayout());
        display.setFont (new Font ("Helvetica", Font.BOLD , 12));

        progress.showProgress("Event Queue");
        queue = EventQueue.getQueue();
        queue.init();
        JDialog eventDialog = new JDialog(new JFrame(), "Event Queue");
        eventDialog.getContentPane().add(queue.getDisplay());
        eventDialog.pack();
        windowmanager.addWindow(eventDialog);

        progress.showProgress("Environment Display");
        JDialog environmentDialog = new JDialog(new JFrame(), "Environment Display");
        environmentDialog.getContentPane().add(Display.getDisplay());
        environmentDialog.pack();
        windowmanager.addWindow(environmentDialog);

        windowmanager.pack();
        windowmanager.setVisible(true);

        // Generic scripter
        scripts = new simulator.Scripter();

        // Start the server
        server = new RootServer(PORT);

        progress.showProgress("System starting, please wait ");
        logger.log("System initialized.");
        logger.log(Simulator.VERSION);
        initGraph();
    }
    /*----------------------------*
     *     cconfig functions      *
     *----------------------------*/
    
    /*----------------------------*
     *     control functions      *
     *----------------------------*/

    /**
     * sets autostart on or off.
     * @param b whether to enable autostart.
     * @param numagents the number of agents to start after.
     */
    public void setAutoStart(boolean b, int numagents) 
    { 
        autostart = b;
        if (autostart)
        {
            autostart_agents = numagents;
            logger.log("enabling autostart after " + numagents + 
                    " agent(s) connect.");
            if (!running && agentcount >= numagents)
                setRunning(true);
        }
    }

    private DumbTextInterface tinter = null;
    
    public void setTextInter(boolean b)
    {
        text_inter = b;
        if (tinter == null)
            tinter = new DumbTextInterface();
        if (text_inter)
        {
            tinter.start();
        } else {
            tinter.stop();
        }
    }
    
    /**
     * Merges the new task structure with the existing one if
     *  an old one exists.  Then create and send the subjective views to
     *  the agents.
     * @param replace: boolean.  true means replace any previously existing
     *         portion of the task structure with the same name with the new
     *         structure.  false means add the new structure without replacing
     * @return The global grammar generator object
     */
    public void addTaems(Taems newview) 
    {
        // Kill resource levels in incoming task
        Enumeration e = newview.findNodes(new Resource());
        while (e.hasMoreElements()) 
        {
            Resource r = (Resource)e.nextElement();
            r.setState(Double.NEGATIVE_INFINITY);
        }

        synchronized (mergelock) 
        {
            logger.log("Creating new objective view", 2);

            // Merge em
            logger.log("Merging...", 3);
            global_view.addTaems(newview, false);
            logger.log("Done merging.", 3);

            // Update taems resource links
            Locale l = getLocale("Global");
            e = l.getResources();
            while (e.hasMoreElements()) 
            {
                Object o = e.nextElement();
                if (o instanceof Resources) 
                {
                    Resources r = (Resources)o;
                    r.setCurrentValue(r.getCurrentValue());
                    r.setMaximum(r.getMaximum());
                    r.setMinimum(r.getMinimum());
                }
            }

            //    matchingResources(newview.resources);
            displayTaems(global_view);

            logger.log("Created new objective view", 3);
        }
    }


    /**
     * Resets the simulation, be careful, this function is not really tested.
     *
     */
    public synchronized void reset() 
    {
        Enumeration	agentsList;
        Agent agent;

        Clock.getClock().reset();
        steps = 0;
        if (agentTable != null) {	// Kill all the agents
            agentsList = agentTable.keys();
            logger.log("Resetting agents.", 2);
            while(agentsList.hasMoreElements()) {
                agent = (Agent)agentTable.get(agentsList.nextElement());
                logger.log("Resetting agent " + agent.getID(), 3);
                agent.reset();
            }
        }
        scripts.reset();
        logger.log("Simulator reset.");
    }

    /**
     * Quits the simulator and kills all agents connected to before exiting.
     */
    public synchronized void quit() 
    {
        Enumeration	agentsList;
        Agent agent;

        if (agentTable != null) {	// Kill all the agents
            agentsList = agentTable.keys();
            logger.log("Disconnecting agents.", 2);
            while(agentsList.hasMoreElements()) {
                agent = (Agent)agentTable.get(agentsList.nextElement());
                logger.log("Disconnecting agent " + agent.getID(), 3);
                agent.disconnect();
            }
        }

        logger.log("System terminated.");
        Runtime.getRuntime().exit(0);
    }

    /**
     * Waits for all running agents to complete, then increments the
     * clock and sends a pulse to all available agents.  This should
     * only be called by the central simulation process.  The agents
     * should execute before the EventQueue gets processed, so this
     * function should get called at the beginning of the process
     * loop.
     */
    public void pulse() 
    {
        Hashtable agentTableCopy;
        Enumeration	agentsList;
        Agent agent;

        synchronized (agentTable) {
            agentTableCopy = (Hashtable)agentTable.clone();
        }

        // Tick tok
        Clock.tick();
        logger.log("Sensing pre-processing", 3);
        globalSensing.pulsing();


        // Pulse the agents
        if (agentTableCopy != null) {
            agentsList = agentTableCopy.keys();
            logger.log("Sending pulses...", 2);
            while(agentsList.hasMoreElements()) {
                agent = (Agent)agentTableCopy.get(agentsList.nextElement());
                logger.log("Pulsing agent " + agent.getID(), 2);
                actionLabel.setText("Pulsing agent " + agent.getID());
                agent.sendPulse();
            }
            logger.log("Pulse sending completed", 3);
        }

        // Wait for the agents to complete
        boolean done = false;
        int timeout = 0;
        int DELAY = 100;
        logger.log("Waiting for agents to finish", 3);
        actionLabel.setText("Waiting for agents to finish");
        while((!done) && (timeout < TIMEOUT)) {
            done = true;
            if (agentTableCopy != null) {  // check
                agentsList = agentTableCopy.keys();
                while(done && agentsList.hasMoreElements()) {
                    agent = (Agent)agentTableCopy.get(agentsList.nextElement());
                    if (agent.isRunning()) done = false;
                }
                try {                    // pause
                    Thread.sleep(DELAY); 
                    timeout += DELAY;
                } catch (InterruptedException e) { logger.log("Error waiting for agents. " + e, 0); }
            }
        }

        // Kill the dead ones
        if (agentTableCopy != null) {
            actionLabel.setText("Looking for dead agents");
            agentsList = agentTableCopy.keys();
            while(agentsList.hasMoreElements()) {
                agent = (Agent)agentTableCopy.get(agentsList.nextElement());
                if (agent.isRunning()) {
                    logger.log("Timeout on waiting for agent " + agent.getName() + "[" + agent.getID() + "] (booted)");
                    agent.disconnect(); // no answer, disconnecting
                    agent.destroy();    // and kill it from the simulation
                }
            }
        }
        graph.repaint();
        global_locale.propagateEnvironmentalModel(global_locale);
        global_locale.clearPropMarks();

        // Done
        logger.log("Waiting completed.", 4);
        actionLabel.setText("");
    }


    public void step(int s) 
    {
        if (s < 0) s = 0;

        steps += s;

        if (steps > 0)
            pauseButton.setText("Pause");
    }

    /**
     * Runs the simulation. Start the main loop consisting to pulsing agents,
     * and running the events after all agents have acking the pulse.
     */
    public synchronized void run() 
    {
        Vector	eventVector;
        Event	event;
        Container eventDisplay = null;

        // Scripts stuff
        scripts.init();
        this.progress.showProgress("Scripts");
        BorderLayout blo = new BorderLayout();
        Dialog scriptDialog = new Dialog(new JFrame(), "Scripts");
        scriptDialog.setLayout(blo);
        scriptDialog.add(scripts.getDisplay(), BorderLayout.CENTER);
        scriptDialog.pack();
        //scriptDialog.setVisible(true);
        windowmanager.addWindow(scriptDialog);

        logger.log("Simulation started.");

        while (true) {
            // Wait
            while (!running) {
                if (steps > 0) { steps--; break; }
                if (! pauseButton.getText().equals("Start"))
                    pauseButton.setText("Start");
                actionLabel.setText("Paused...");
                try { 
                    wait(1); 
                    Thread.sleep(500); 
                } catch (InterruptedException e) { 
                    logger.log("Error during pause: " + e); 
                }
                scripts.pulse(Scripter.PERIODIC);
            }

            // Tick tock, but only if the current time slot is empty
            if (!queue.hasTimeSlot(Clock.getTime())) {
                actionLabel.setText("Ticking...");
                logger.log("Tick started.", 2);
                pulse();
                logger.log("Tick completed.", 3);
            }

            // Run scripts
            scripts.pulse(Scripter.PREEVENTS);

            // Process events
            // if (eventDisplay != null) display.remove(eventDisplay);
            queue.setDisplay(Clock.getTime());
            //Vector oldEventVector = new Vector(25);
            synchronized (mergelock) {
                while (queue.hasTimeSlot(Clock.getTime())) {
                    eventVector = queue.fetchTimeSlot(Clock.getTime());
                    eventLabel.setText("Events: " + eventVector.size());
                    logger.log("Processing events", 2);
                    while(!eventVector.isEmpty()) {
                        logger.log("Getting first element ...", 3);
                        event = (Event)eventVector.firstElement();
                        logger.log("Removing Event " + event.getID(), 3);
                        eventVector.removeElement(event);
                        actionLabel.setText("Realizing event.");
                        statusLabel.setText(event.getStatus());
                        logger.log("Realizing event " + event.getID(), 3);
                        event.realize();
                        logger.log(event.getStatus(), 4);
                        eventLabel.setText("Events: " + eventVector.size());

                        //oldEventVector.addElement(event);
                    }
                }
            }
            statusLabel.setText("No event currently.");
            logger.log("Event processing completed.", 3);

            // Run scripts, restoring event queue first
            //queue.putTimeSlot(oldEventVector, Clock.getTime());
            scripts.pulse(Scripter.POSTEVENTS);
            //queue.removeTimeSlot(Clock.getTime());

            // Pause a moment to allow other synch's to execute
            try { 
                wait(1); 
            } catch (InterruptedException e) { 
                logger.log("Error during pause: " + e); 
            }
        }
    }

    /*----------------------------*
     *       misc accessors       *
     *----------------------------*/

    /**
     * setTimeout just the maximum time that the simulator wait for
     * an agent to ack its pulse. This value could be set in the configuration
     * file.
     * @return the new timeout
     */
    public int setTimeout(int value) 
    { 
        return TIMEOUT=value; 
    }


    /**
     * Sets the server port, restarts the server if available
     */
    public void setPort(int value) 
    {
        if ((PORT != value) && (server != null)) {
            PORT = value;
            server.stop();
            server = new RootServer(PORT);	
        }
    }

    /**
     * Set the flag for the simulation, about applying the NLE's using
     * the quality of the facilitator
     */
    public void setComplexNLE(boolean value) 
    { 
        complexNLEFlag = value; 
    }
   
    /**
     * returns whether the simulator is running.
     */ 
    public boolean isRunning() 
    {
        return (running || (steps > 0));
    }

    /**
     * pauses or starts the simulator.
     */
    public void setRunning(boolean r) 
    {
        running = r;
        if (running) {
            if (! pauseButton.getText().equals("Pause"))
                pauseButton.setText("Pause");
        } else {
            steps = 0;
            if (! pauseButton.getText().equals("Start"))
                pauseButton.setText("Start");
        }
    }
    
    /**
     * Return a list of all Agent connected (string form)
     * @return Agents List
     */
    public String getAllAgents() {
        String answer = "";

        logger.log("Getting all connected agents.", 3);
        for (Enumeration e = agentTable.keys(); e.hasMoreElements(); ){
            Agent agent = (Agent)agentTable.get(e.nextElement());
            answer = agent.getName() + " " + answer;
        }
        logger.log("Found agents: " + answer, 4);
        return(answer);
    }

    /**
     * Tells if the agent is currently being simulated
     * @return True if the agent is still connected
     */
    public boolean hasAgent(Agent a) {
        return agentTable.containsKey(new Integer(a.getID())); 
    }

    /**
     * Adds an agent
     * @param a The agent
     */
    public void addAgent(Agent a) {
        logger.log("Adding agent " + a.getName(), 1);
        agentTable.put(new Integer(a.getID()), a); 
        display.add(a.getDisplay());
        display.validate();
        logger.log("Agent " + a.getName() + " added.", 2);
        agentcountlabel.setText("Agents: " + ++agentcount);
        if (autostart && autostart_agents <= agentcount)
        {
            logger.log("Autostarting!", 2);
            setRunning(true);
        }
    }


    /**
     * Removes an agent
     * @param a The agent
     */
    public synchronized void removeAgent(Agent a) {
        logger.log("Removing agent " + a.getName(), 1);
        if (agentTable.containsKey(new Integer(a.getID())))
        {
            agentTable.remove(new Integer(a.getID())); 
            display.remove(a.getDisplay());
            display.validate();
            logger.log("Agent " + a.getName() + " removed.", 2);
            agentcountlabel.setText("Agents: " + --agentcount);
        }
    }

    /**
     * Find an agent by its id (useful for resending messages).
     * @param id The agent's id number
     * @return Agent's intance or null 
     **/
    public Agent findAgent (int id) {
        logger.log("Looking for agent " + id, 3);

        if (agentTable.containsKey(new Integer(id))) {
            Agent agent = (Agent)agentTable.get(new Integer(id));
            logger.log("Found " + agent.getName(), 2);
            return agent;
        } 

        logger.log("Didn't find " + id, 2);
        return(null);
    }

    /**
     * Find an agent by its name (useful for resending messages).
     * @param String agent's name
     * @return Agent's intance or null 
     */  
    public Agent findAgent (String name) {
        logger.log("Looking for agent " + name, 3);

        for (Enumeration e = agentTable.keys(); e.hasMoreElements(); ){
            Agent agent = (Agent)agentTable.get(e.nextElement());
            logger.log("Looking at agent " + agent.getName() + " for agent " + name, 3);
            if (name.equals (agent.getName())) {
                logger.log("Found " + agent.getName(), 2);
                return(agent);
            }
        }

        logger.log("Didn't find " + name, 2);
        return(null);
    }

    /**
     * Broadcast to all agents this messages except for the sender.
     * @param : Agent sender
     * @param : KQMLMessage msg
     * 
     */
    public void broadcastMsg(Agent sender, KQMLMessage msg) {
        for (Enumeration e = agentTable.keys(); e.hasMoreElements(); ){
            Agent agent = (Agent)agentTable.get(e.nextElement());
            if (!sender.getName().equals(agent.getName())) {
                logger.log("Broadcast for: " + agent.getName(), 2);
                MessageEvent me = 
                    new MessageEvent(0, 0, sender, agent, msg);
                queue.addEvent(me);
            }
        }
    }
    
    /**
     * Returns the generic scripter
     */
    public Scripter getScripter() 
    {
        return scripts;
    }

    /**
     * Read sensing configuration file
     * @param String sensor configuration filename
     */
    public void configureSensingModule(String filename) 
    {
        getGlobalSensing().readConfigurationFile(filename);
    }

    /**
     * Returns the current sensing module
     * @return Sensing object
     */ 
    public Sensing getGlobalSensing() 
    {
        return(globalSensing);
    }
   
    public Graph getGraph()
    {
        return graph;
    }
    
    public void setGraph(Graph g)
    {
        if (g != null)
        {
            graph = g;
        }
    }

    /*----------------------------*
     *    resource functions      *
     *----------------------------*/
    
    /**
     * Returns the display, for use by the agents so they can
     * add or remove themselves.
     * @return A reference to the display panel
     */
    public Panel getDisplay() 
    {
        return display;
    }

    /**
     * Returns the resources vector 
     * @return The resource vector
     */
    public Object getResource(Locale lo, String rs) throws Locale.ResourceDoesNotExist 
    {
        if (lo == null)
            return getResource(rs);
        else
            return lo.getResource(rs);
    }

    public Object getResource(String rsrc) throws Locale.ResourceDoesNotExist 
    {
        return global_locale.getResource(rsrc);
    }

    /**
     * Add a new resource to the simulator, only if this resource doesn't
     * already exits.
     */
    public void addResource(String lo, String nm, Object re) 
    {
        if(lo.equals("Global"))
        {
            global_locale.addResource(nm, re);
            //		((Resources)re).setLocale(global_locale);
        }
        else
            try {
                Locale sl=global_locale.getSubLocale(lo);
                sl.addResource(nm, re);
                ((Resources)re).setLocale(sl);
            } catch (Locale.LocaleDoesNotExist foo) {
                logger.log("Error addResource: Locale "+lo+" isn't defined.", 1);
            }
    }

    public Locale addLocale(String lo, String par, String type)
    {
        return addLocale(lo, par, type, null);
    }

    public Locale addLocale(String lo, String par, String type, Shape shape) 
    {
        Locale newlocale = null;
        if(par.equals("Global"))
            newlocale = global_locale.newSubLocale(lo, type);
        else
            try {
                newlocale = (global_locale.getSubLocale(par)).newSubLocale(lo, type);
            } catch (Locale.LocaleDoesNotExist foo) {
                logger.log("Error addLocale: Parent locale "+lo+" isn't defined.", 1);
            }
        getGraph().addNode(newlocale);
        return newlocale;
    }

    public Linkage addLinkage(String nm, String src, String dst, String type) 
    {
        Locale s=getLocale(src);
        Locale d=getLocale(dst);
        if(s == null) { 
            logger.log("Error addLinkage: Source linkage "+src+" isn't defined.", 1);
            return null;
        }
        if(d == null) {
            logger.log("Error addLinkage: Destination linkage "+dst+" isn't defined.", 1);
            return null;
        }
        Linkage link = s.linkageTo(nm, d, type);
        getGraph().addNode(link.getEdge());
        return link;
    }

    public Locale getLocale(String lo) 
    {
        if(lo.equals("Global"))
            return global_locale;
        else
            try {
                return global_locale.getSubLocale(lo);
            } catch(Locale.LocaleDoesNotExist foo) { 
                logger.log("Error getLocale: Locale "+lo+" isn't defined.", 1);
                return null;
            }
    }

    /** 
     * Loads a class to use to map strings from the agents to the proper
     * locale instances.
     */
    public void setLocationMapping(String lm) 
    {
        locmap=(LocationMapping)LoadClass.load("simulator.locale."+lm, new Object[0]);
    }

    public LocationMapping getLocationMapping() 
    {
        return locmap;
    }

    /**
     * looks for a model already defined for getting informations,
     * simply matching the beginning of the name.
     * Noise_BedRoom matches with Noise resource.
     * like maximum, minimum, units 
     */
    protected String findApproxmativeResource(String lookfor) 
    {
        for (Enumeration e = resources.keys(); e.hasMoreElements(); ){
            Resources res = (Resources)resources.get(e.nextElement());
            if (lookfor.startsWith(res.getName()))
                return(res.getName());
        }
        return null;
    }
    
    /**
     * getResourceStatus of a designated method
     * @param resource name 
     * @return a String containing the status
     * Status format is : R1 30 0 100 
     *                  : Name value minimum maximum
     */
    public String getResourceStatus(Locale lo, String r) 
    {
        try {
            Resources res = (Resources)getResource(lo, r);
            return( r + " " + res.getCurrentValue() + " " 
                    + res.getMinimum() + " " 
                    + res.getMaximum());
        } catch (Locale.ResourceDoesNotExist foo) {
            return "Error Resource not found";
        }
    }

    
    /*----------------------------*
     *        GUI functions       *
     *----------------------------*/
    
    protected void initButtonListeners()
    {
        pauseButton.addActionListener(new ActionListener() 
                {
                    public void actionPerformed(ActionEvent e) 
                    {
                        setRunning(!isRunning());
                    }
                });
        stepButton.addActionListener(new ActionListener() 
                {
                    public void actionPerformed(ActionEvent e) 
                    {
                        if (!running) 
                        {
                            step(1);
                        } else {
                            pauseButton.doClick();
                            steps = 1;
                        }
                    }
                });
        steptoButton.addActionListener(new ActionListener() 
                {
                    public void actionPerformed(ActionEvent e) 
                    {
                        int time = Clock.getTime();
                        try {
                            int dest = Integer.parseInt(steptoField.getText());
                            if ((dest - time) <= 0) {
                                steptoField.setText("0");
                            } else {
                                step(dest - time);
                            }
                            if (running) {
                                pauseButton.doClick();
                            }
                        } catch (NumberFormatException ee) {
                            steptoField.setText("0");
                        }
                    }
                });
        steptoField.addKeyListener(new KeyListener() 
                {
                    public void keyPressed(KeyEvent e) 
                    {
                        if (e.getKeyCode() == KeyEvent.VK_ENTER)
                        {
                            steptoButton.doClick();
                        }
                    }
                    public void keyReleased(KeyEvent e) { }
                    public void keyTyped(KeyEvent e) { }
                });    
        quitButton.addActionListener(new ActionListener() 
                {
                    public void actionPerformed(ActionEvent e) 
                    {
                        quit();
                    }
                });
        resetButton.addActionListener(new ActionListener() 
                {
                    public void actionPerformed(ActionEvent e) 
                    {
                        reset();
                    }
                });
        aboutButton.addActionListener(new ActionListener() 
                {
                    public void actionPerformed(ActionEvent e) 
                    {
                        displayAbout();
                    }
                });
    }

    protected void initWindowManager()
    {
        eventLabel = new Label("Events:");
        actionLabel = new Label("Loading");
        statusLabel = new Label("No event currently.");
        agentcountlabel = new Label("Agents: 0");
        randomLabel = new Label("Random seed: " + random.getSeed());
        windowmanager.addInfo(Clock.getClock().getLabel());
        windowmanager.addInfo(agentcountlabel);
        windowmanager.addInfo(eventLabel);
        windowmanager.addInfo(actionLabel);
        windowmanager.addInfo(statusLabel);
        windowmanager.addInfo(randomLabel);

        pauseButton = new JButton("Start");

        stepButton = new JButton("Step");    
        steptoButton = new JButton("Step To");    
        steptoField = new TextField("0", 3);
        JPanel temppanel = new JPanel();
        temppanel.setLayout(new GridLayout(2,1));
        temppanel.add(steptoButton);
        temppanel.add(steptoField);

        quitButton = new JButton("Quit");
        resetButton = new JButton("Reset");
        aboutButton = new JButton ("About");

        initButtonListeners();

        windowmanager.addControl(pauseButton);
        windowmanager.addControl(stepButton);
        windowmanager.addControl(temppanel);
        windowmanager.addControl(quitButton);
        windowmanager.addControl(resetButton);
        windowmanager.addControl(aboutButton);
        windowmanager.pack();

    }
    
    protected void initGraph()
    {
        setGraph(new LocaleGraph("Locale Display"));
    }


    /**
     * Displays the evironment and resources. This function creates the 
     * Global Environmental Display. This is the window with the map and the 
     * resources on the side.
     */
    public void displayEnvironment() {
        if (resources != null) {
            graph.addTree(global_locale);
            showResourceDisplay(graph);
        }
    }
    
    /**
     * Create and display an About Panel. 
     */ 
    public void displayAbout() 
    {
        JPanel display;
        displayAbout = new JDialog(new JFrame(), "About...");
        // And the logo
        try {
            JPanel logo = new JPanel();
            String http="http://mas.cs.umass.edu/research/mass/graphics/simulate/simulator.GIF";
            if (System.getProperty("MASS") != null) 
                http=System.getProperty("MASS");
            logo.add(new ImageComponent(new URL(http)));
            displayAbout.getContentPane().add("North", logo);
        } catch (MalformedURLException e) { 
            logger.log("Error finding image: " + e); 
        }

        // ...and the main display
        display = new JPanel();
        TextArea text = new TextArea("                  About this Multi Agent Survivability Simulator.....\n\n                             http://dis.cs.umass.edu/research/mass/\n\n\n   Written by: Bryan Horling and Regis Vincent\n   Helped by: Peter Amstutz, Brett Benyo, Kyle Rawlins\n\n\n\n           Department of Computer and Information Science\n           University of Massachusetts\n           Amherst, Massachusetts 01003.\n\n    This code was written at the Multi-Agent Systems Lab.\n    Department of Computer Science, University of Massachusetts,\n    Amherst, MA 01003.\n    Copyright (c) 1997,1998,1999,2000 UMASS CS Dept. All rights are reserved.\n\n\n    Development of this code was partially supported by:\n       The Defense Advanced Research Projects Agency (DARPA) \n       and Rome Laboratory, Air Force Materiel Command, USAF, \n       under agreement number F30602-97-1-0249. \n\n\n   ", 30,100,TextArea.SCROLLBARS_VERTICAL_ONLY); 
        text.setEditable(false);
        display.add(text);
        displayAbout.getContentPane().add("Center", display);
        Panel south = new Panel();
        south.setLayout(new GridBagLayout());
        south.setFont (new Font ("Helvetica", Font.BOLD, 12));

        //Panel container = new Panel();
        //container.setFont (new Font ("Helvetica", Font.BOLD , 12));
        Button OKButton = new Button ("OK");
        OKButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                displayAbout.dispose();
                }
                });
        //    container.add(OKButton);
        addComponent(south, OKButton, 0, 0, 1, 1, GridBagConstraints.HORIZONTAL, GridBagConstraints.CENTER);
        displayAbout.getContentPane().add("South",south);
        displayAbout.setSize(new Dimension(200,200));
        displayAbout.setBackground(Color.white);
        displayAbout.pack();
        displayAbout.setVisible(true);
    }
    
    /**
     * Displays a taems data structure in a window.  Repeated calls
     * re-use this same window
     * @param ttaems The Taems structure to display
     */
    public void displayTaems(Taems t) {
        int INITIAL_WIDTH = 500;
        int INITIAL_HEIGHT = 400;
        if (global_view_frame == null) {
            global_view_frame = new JFrame("Global Task Structure"); 

            global_view_frame.setSize(INITIAL_WIDTH, INITIAL_HEIGHT);
            Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            global_view_frame.setLocation(screenSize.width/2 - INITIAL_WIDTH/2,
                    screenSize.height/2 - INITIAL_HEIGHT/2);
            global_view_frame.getContentPane().add(t, BorderLayout.CENTER);
            JLabel jt = new JLabel("See the locale display for state details on non-global resources.", SwingConstants.CENTER);
            jt.setBackground(t.getBackground());
            global_view_frame.getContentPane().add(jt, BorderLayout.SOUTH);
            windowmanager.addWindow(global_view_frame);
        }
    }
    
    public void showResourceDisplay(Graph g)
    {
        JFrame frame;
        frame = new JFrame("Locale Display");
        frame.setSize(RESOURCE_WINDOW_INITIAL_WIDTH, RESOURCE_WINDOW_INITIAL_HEIGHT);
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        frame.setLocation(screenSize.width/2 - RESOURCE_WINDOW_INITIAL_WIDTH/2,
                screenSize.height/2 - RESOURCE_WINDOW_INITIAL_HEIGHT/2);
        frame.getContentPane().add(g, BorderLayout.CENTER);
        //frame.setVisible(true);
        windowmanager.addWindow(frame);
    }     

    public class DumbTextInterface implements Runnable
    {
        boolean stopped = true;
        
        public DumbTextInterface()
        {

        }

        public void stop()
        {
            stopped = true;
        }

        public void start()
        {
            if (!stopped)
                return;
            Thread t = new Thread(this);
            stopped = false;
            t.start();
        }

        public void run()
        {
            while (!stopped)
            {
                synchronized (System.in)
                {
                    try {
                        int available = System.in.available();
                        while (available-- > 0)
                        {
                            char c = (char) System.in.read();
                            if (c == '\n' || c == '\r')
                            {
                                if (running)
                                    setRunning(false);
                                System.err.println("textinter: stepping...");
                                step(1);
                            }
                        }
                    } catch (IOException e) {
                        System.err.println("TextInterface: IOException");
                        e.printStackTrace();
                    }
                }
                Thread.yield();
            }
        }
    }
}
