/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
* $Revision: 1.2 $
* Author : Regis Vincent [vincent@cs.umass.edu]
* Creation date: 24 Dec 98 19:53
* Last file update: $Date: 2005/01/19 19:00:05 $
*****************************************************************/

package simulator;

/* Local imports */
import utilities.Log;
import utilities.LoadClass;

/* Global Includes */
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.io.*;
import java.lang.reflect.*;
import simulator.sensor.*;
import simulator.locale.*;

/**
 * Sensing module read the authorization for allowing agent to
 * get informations of the world. 
 * The file defines the keyword used in the protocol, the imprecision
 * associated with each sensor, the exact functions to call, the optional
 * parameters.
 */
public class Sensing {
  private Hashtable sensorTable = new Hashtable();
  private Log log;
  
  /**
   * Constructor: don't know what to write down here.
   */

  public Sensing() {
    log = Log.getDefault();
    if (sensorTable == null) 
      sensorTable = new Hashtable();
  }
  
  public Sensing(String filename) {
    this();
    readConfigurationFile(filename);
  }

  public  void pulsing() {
     // need to pulse all sensing module.
    for (Enumeration e = sensorTable.elements() ; e.hasMoreElements(); ) {
      Sensor sd = (Sensor)e.nextElement();
      sd.pulse();
    }
  }
    

  /**
   * Read the file passed as argument, and add it in the keywords hashtable.
   *
   */
  public void readConfigurationFile(String filename) {
    BufferedReader in;

    try {
      log.log("Trying to process Sensing script " + filename, 2);
      FileReader reader = new FileReader(filename);
      in = new BufferedReader(reader);
    } catch (FileNotFoundException e) {
      log.log("Error: Sensing script \"" + filename + "\" not found", 0);
      return;
    }
    try {
      int line=0;
      while (in.ready()) {
	String s = in.readLine();
	line++;
	s = s.trim();
	if (s.startsWith(";") || (s.length() == 0)) continue;
	parseLine(s,line);
      }
    } catch (EOFException e) {
    } catch (IOException e) {
      log.log("Error reading from Sensor file " + filename, 0);
    }
    catch (SensorParseException e) {
    }

  }

  /**
   * Parse this sensor lines and convert it into a Sensor object.
   * @param String s, the line to parse
   * @param int line, the line number
   * @return Sensor object.
   * <P>
   * <B>Be careful, this function add the Sensor in the sensor
   * hashtable.</B>
   */
  public synchronized Sensor parseLine(String s, int l) throws SensorParseException {
    // Tokenize it
    StringTokenizer tokens = new StringTokenizer(s, ",");
    String tok = tokens.nextToken();
    
    if (tok.startsWith("Sensor") || tok.startsWith("sensor")) {
      // Sensor, SensorClassName, SensorSpecificName, Data...
      String cls;
      Object args[] = new Object[2];
      Sensor sensor;
      
      // Read the class name
      cls = tokens.nextToken().trim();
      
      // Read the specific instance name
      args[0] = tokens.nextToken().trim();
      log.log("Sensor simulator.sensor." + cls + " " + args[0], 2);
      
      // Read the arguments
      tok = "";
      while (tokens.hasMoreTokens()) {
	if (tok.length() == 0)
	  tok = tokens.nextToken();
	else
	  tok = tok + "," + tokens.nextToken();
      }
      args[1] = tok;
      // Create the instance
      try {
	sensor = (Sensor)LoadClass.load("simulator.sensor."+cls, args);
      } catch (Exception e) {
	log.log("Error creating Sensor (line " + l + ")" + e, 0);
	throw new SensorParseException("Error creating Sensor (line "+ l + ")" );

      }
      if (sensor == null) {
	log.log("Error creating Sensor (line " + l + ")" , 0);
	throw new SensorParseException("Error creating Sensor (line "+ l + ")" );
      }
      else {
	  sensor.init();
	  addSensor(sensor);
	  return sensor;
      }
      
    }

    return null;
  }
    
    public synchronized Sensor parseLine(String s) throws SensorParseException{
      return parseLine(s, 1);
    }


  /**
   * addSensor add a sensor object to the main HashTable.
   */
  public Object addSensor(Sensor s) { 
      if(! sensorTable.containsKey(s.getName())) {
          log.log("Adding sensor " + s.getName(), Log.LOG_INFO);
	  sensorTable.put(s.getName(), s);
	  return(s);
      }
      else return null;
  }
    
  /**
   * getSensor returns a sensor object (which contains Reflection, Parameters)
   */
  public Sensor getSensor(String keyword) { 
    return((Sensor)sensorTable.get(keyword));
  }

  /**
   * Returns a list of available sensors
   * @return An enumeration of all the sensors
   */
  public Enumeration getSensors() { 
    return sensorTable.elements();
  }

  /**
   * Delete Sensor. In case of error you could delete a sensor.
   */
  public Object deleteSensor(Sensor s) {
	return sensorTable.remove(s.getName());
  }

    /**
     * Fucntion returns a Sensor object if found, or
     * null otherwise.
     */
    public Object getObjectSensed(Locale loc, String call) {  
	Sensor s ;
	StringTokenizer tokens = new StringTokenizer(call, ",");
	String tok = tokens.nextToken();
	String keyword = tok.trim();

	// Read the class name
	log.log("Try to find " + keyword + " in " + loc.getName(),5);
	s = getSensor(keyword);
    
	if (s == null) {
	    log.log("Not found here in " + loc.getName(),5);
	    return null;
	}
	else { 
	    return s.getObjectSensed(loc);
	}
    }

  /**
   * Main sensing function that takes the agent's message and try to read
   * the sensor
   */
    public String runSensing(Locale loc, String call) throws SensingErrorException {
        Sensor s ;
        StringTokenizer tokens = new StringTokenizer(call, ",");
        String tok = tokens.nextToken();
        String keyword = tok.trim();
        // Read the class name

        s = getSensor(keyword);
    
        if (s == null) {
            log.log("Sensing.runSensing: Error with this argument " + call + " in Locale "+ 	loc.getName(),0);
            throw new SensingErrorException("Sensing.runSensing with this argument " + call);
        }
    
        // Read the arguments
        tok = "";
        while (tokens.hasMoreTokens()) {
            if (tok.length() == 0)
                tok = tokens.nextToken();
            else
                tok = tok + "," + tokens.nextToken();
        }
    
        return(s.sense(loc, tok).toString());
    }
 
//    /**
//    * Loads a class, given its name
//    * @param n The class name
//    * @param args The constructor arguments
//    * @return The class instance, or null if an error occured
//    */
//   public Object loadClass(String n, Object []args) {
//     Object o = null;

//     log.log("Creating " + n, 1);

//     // Find the class
//     Class c = null;
//     n = n.trim();
//     try {
//       c = Class.forName("simulator.sensor." + n);
//     } catch (ClassNotFoundException z) {
//       try {
// 	c = Class.forName(n);
//       } catch (ClassNotFoundException e) {
// 	log.log("Error: Cannot find class " + n + "\n" + e, 0);
//       }
//     }

//     // Make an instance
//     if (c != null) {
//       try {
// 	Class cargs[] = new Class[args.length];
// 	for (int i = 0; i < args.length; i++)
// 	  cargs[i] = args[i].getClass();
// 	Constructor cons = c.getConstructor(cargs);
// 	o = cons.newInstance(args);
//       } catch (Exception e) {
// 	log.log("Error creating object " + c + ": " + e, 0);
//       }
//     }

//     return o;
//   }
    
   public String toString() {
     String answer = "[Sensor defined]\n";
       // need to pulse all sensing module.
       for (Enumeration e = sensorTable.elements() ; e.hasMoreElements(); ) {
 	Sensor sd = (Sensor)e.nextElement();
 	answer = answer + sd.toString() + "\n";
       }
     return(answer);
   }

}


