/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Scripter.java
 ***************************************************************************************/

package simulator;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.awt.*;

/* Local Includes */
import simulator.*;
import simulator.script.*;
import utilities.*;


/**
 * The script running engine.  Responsible for reading in script
 * specification data and instantiating all the necessary parts of
 * each script.  During runtime, the Scripter will periodically 
 * check each of the installed scripts to determine if they should
 * be activated.  The Scripter can also be used to determine which
 * scripts are installed.
 */
public class Scripter {
    /**
     * Pulse type constants
     */
    public static int NONE       = 0x00000000;
    public static int PREEVENTS  = 0x0000000F;
    public static int PERIODIC   = 0x000000F0;
    public static int POSTEVENTS = 0x00000F00;
    public static int OTHER      = 0xF0000000;

    private Log logger;
    private Panel display;
    private static Vector sources = null;
    private Vector backupSources;
    private int type = NONE;

    private Vector scripts = new Vector(20);
	
    /**
     * Constructor
     */
    public Scripter() {

	// Nothing happens here
    }

    /**
     * Reset function called by the simulator (when someone hit
     * the reset button. 
     */
    public void reset() { 
	scripts = new Vector(20);
	if (backupSources != null) {
	    Enumeration e = backupSources.elements();
	    while (e.hasMoreElements()) {
		readScriptFile((String)e.nextElement());
	    }
	}
	makeDisplay();
    }

    /**
     * Init, read the different scripting files.
     */  
    public void init() {
	logger = Log.getDefault();
    
	// Read the file
	if (sources != null) {
	    Enumeration e = sources.elements();
	    while (e.hasMoreElements()) {
		readScriptFile((String)e.nextElement());
	    }
	    backupSources = (Vector)sources.clone();
	    sources.removeAllElements();
	    
	}

	// Setup the display
	display = new Panel();
	display.setLayout(new VFlowLayout(VFlowLayout.TOP));
	display.setFont (new Font ("Helvetica", Font.BOLD , 14));
	makeDisplay();
    }		

    /**
     * Reads a config file to instantiate the current set of scripts
     * @param f The name of the script file to read
     */
    public void readScriptFile(String f) {
	BufferedReader in;

	if (f == null) {
	    logger.log("Script file not specified", 1);
	    return;
	}

	try {
	    logger.log("Trying to process script file " + f, 2);
	    FileReader reader = new FileReader(f);
	    in = new BufferedReader(reader);
	} catch (FileNotFoundException e) {
	    logger.log("Error: File \"" + f + "\" not found", 0);
	    return;
	}

	try {
	    // Read in the file
	    Script script = null;
	    int line = 0;
	    while (in.ready()) {
		String s = in.readLine();
		line++;
		script = augmentScript(script, s, line);
	    }

	    // Add the last one
	    if (script != null) {
		addScript(script);
		script = null;
	    }

	} catch (EOFException e) {
	    logger.log("Finished reading script file", 2);
	} catch (IOException e) {
	    logger.log("Error reading from resource file " + f, 0);
	}
    }

    /**
     * Reads a config vector, which has essentially the same format
     * as the config file, with one line per item in the vector (element
     * 0 represents the "top" of the config file).
     * @param v The vector to read from
     */
    public void readScriptVector(Vector v) {
	BufferedReader in;

	if (v == null) {
	    if (logger != null)
		logger.log("Script vector is null", 1);
	    return;
	}

	if (logger != null)
	    logger.log("Trying to process script vector", 2);
    
	// Read from the vector
	Script script = null;
	int line = 0;
	while (!v.isEmpty()) {
	    String s = (String)v.firstElement();
	    v.removeElement(s);
	    line++;
	    script = augmentScript(script, s, line);
	}
    
	// Add the last one
	if (script != null) {
	    addScript(script);
	    script = null;
	}
    }

    /**
     * Augments a Script object, given another line of config input.
     * If a new Script object is encountered, the working object
     * is added via addScript.
     * @param The Script to add to, or null if not started
     * @param The next line of config to parse
     * @param The line number currently being read
     * @return The augmented Script
     */
    public Script augmentScript(Script script, String s, int line) {

	// Wheedle it
	s = s.trim();
	if (s.startsWith(";") || (s.length() == 0)) return script;
    
	// Tokenize it
	StringTokenizer tokens = new StringTokenizer(s, ",");
	String tok = tokens.nextToken();

	if (tok.startsWith("S") || tok.startsWith("s")) {
	    // Script, ScriptClassName, ScriptSpecificName, Data...
	    String cls;
	    Object args[] = new Object[2];
      
	    // Add it
	    if (script != null) {
		addScript(script);
		script = null;
	    }
      
	    // Read the class name
	    cls = tokens.nextToken().trim();
      
	    // Read the specific instance name
	    args[0] = tokens.nextToken().trim();
      
	    // Read the arguments
	    tok = "";
	    while (tokens.hasMoreTokens()) {
		if (tok.length() == 0)
		    tok = tokens.nextToken();
		else
		    tok = tok + "," + tokens.nextToken();
	    }
	    args[1] = tok;
      
	    // Create the instance
	    try {
		script = (Script)LoadClass.load("simulator.script."+cls, args);
	    } catch (Exception e) {
		logger.log("Error creating Script (line " + line +  "): " + e, 0);
	    }
	    if (script == null)
		logger.log("Error creating Script (line " + line +  ")", 0);
      
	} else if (tok.startsWith("A") || tok.startsWith("a")) {
	    // Assertion, AssertionClassName, Data...
	    String cls;
	    Object args[] = new Object[1];
      
	    // Check
	    if (script == null) {
		logger.log("No script defined, ignoring assertion (line " + line + ")", 0);
		return script;
	    }
      
	    // Read the class name
	    cls = tokens.nextToken().trim();
      
	    // Read the arguments
	    tok = "";
	    while (tokens.hasMoreTokens()) {
		if (tok.length() == 0)
		    tok = tokens.nextToken();
		else
		    tok = tok + "," + tokens.nextToken();
	    }
	    args[0] = tok;
      
	    // Create the instance
	    Assertion a = null;
	    try {
		a = (Assertion)LoadClass.load("simulator.script."+cls, args);
	    } catch (Exception e) {
		logger.log("Error creating Assertion (line " + line +  "): " + e, 0);
	    }
	    if (a == null)
		logger.log("Error creating Assertion (line " + line +  ")", 0);
	    else
		script.addAssertion(a);
      
	} else if (tok.startsWith("R") || tok.startsWith("r")) {
	    // Reaction, ReactionClassName, Data...
	    String cls;
	    Object args[] = new Object[1];
      
	    // Check
	    if (script == null) {
		logger.log("No script defined, ignoring reaction (line " + line + ")", 0);
		return script;
	    }
      
	    // Read the class name
	    cls = tokens.nextToken().trim();
      
	    // Read the arguments
	    tok = "";
	    while (tokens.hasMoreTokens()) {
		if (tok.length() == 0)
		    tok = tokens.nextToken();
		else
		    tok = tok + "," + tokens.nextToken();
	    }
	    args[0] = tok;
      
	    // Create the instance
	    Reaction r = null;
	    try {
		r = (Reaction)LoadClass.load("simulator.script."+cls, args);
	    } catch (Exception e) {
		logger.log("Error creating Reaction (line " + line +  "): " + e, 0);
	    }
	    if (r == null)
		logger.log("Error creating Reaction (line " + line +  ")", 0);
	    else
		script.addReaction(r);
	}
  
	return script;
    }

    //   /**
    //    * Loads a class, given its name
    //    * @param n The class name
    //    * @param args The constructor arguments
    //    * @return The class instance, or null if an error occured
    //    */
    //   public Object loadClass(String n, Object []args) {
    //     Object o = null;

    //     logger.log("Creating " + n, 1);

    //     // Find the class
    //     Class c = null;
    //     n = n.trim();
    //     try {
    //       c = Class.forName("simulator.script." + n);
    //     } catch (ClassNotFoundException z) {
    //       try {
    // 	c = Class.forName(n);
    //       } catch (ClassNotFoundException e) {
    // 	logger.log("Error: Cannot find class " + n + "\n" + e, 0);
    //       }
    //     }

    //     // Make an instance
    //     if (c != null) {
    //       try {
    // 	Class cargs[] = new Class[args.length];
    // 	for (int i = 0; i < args.length; i++)
    // 	  cargs[i] = args[i].getClass();
    // 	Constructor cons = c.getConstructor(cargs);
    // 	o = cons.newInstance(args);
    //       } catch (Exception e) {
    // 	logger.log("Error creating object: " + e, 0);
    //       }
    //     }

    //     return o;
    //   }

    /**
     * Sends a pulse to each of the scripts.
     * <P>
     * Note that the effects of scripts responding to PERIODIC 
     * pulses are not guarunteed to be deterministic (since they
     * are not tied to the time-pulsing system).
     * @param type The type of pulse which is being sent (see the 
     * constants noted above)
     */
    public void pulse(int type) {
	if (scripts.size() == 0) return;

	Enumeration e = getScripts();
	if ((type & PERIODIC) > 0)
	    logger.log("(SCR) Pulsing scripts (type " + type + ")", 5);
	else
	    logger.log("(SCR) Pulsing scripts (type " + type + ")", 3);

	setPulseType(type);

	while (e.hasMoreElements()) {
	    Script s = (Script)e.nextElement();
	    s.pulse();
	}

	setPulseType(NONE);
    }

    /**
     * Adds a script to the script list, calling init just after it is
     * added.
     * @param e The script to add
     */
    public synchronized void addScript(Script s) {

	logger.log("(SCR) Adding script " + s, 2);

	scripts.addElement(s);
	s.init();

	if (getDisplay() != null)
	    makeDisplay();
    }
  
    /**
     * Removes a script from the list
     * @param e The script to remove
     * @return true if the script was sucessfully removed
     */
    public synchronized boolean removeScript(Script s) {
	boolean suc = false;

	if (s != null)
	    suc = scripts.removeElement(s);

	if (getDisplay() != null)
	    makeDisplay();
    
	return suc;
    }
  
    /**
     * Returns an enumeration of the scripts
     * @return An enumeration
     */
    public Enumeration getScripts() {

	return scripts.elements();
    }

    /**
     * Finds a script in the list by its name.
     * @param name The name of the script to search for
     * @return The script, or null if none found
     */
    public Script findScript(String name) {
	Enumeration	e = getScripts();
    
	logger.log("(EQ) Looking for script " + name, 2);
	while(e.hasMoreElements()) {
	    Script s = (Script)e.nextElement();

	    if (s.getName().equalsIgnoreCase(name)) {
		logger.log("(EQ) Found script " + s, 2);
		return s;
	    }
	}
	logger.log("(EQ) Didn't find script.");
	return null;
    }

    /**
     * Adds a source config file.
     * @param s The source pathname
     */
    public static void addSource(String s) {

	if (sources == null)
	    sources = new Vector();

	sources.addElement(s);
    }

    /**
     * Returns the display
     * @return A panel
     */
    public Container getDisplay() {
	return display;
    }

    /**
     * Returns the type of pulse currently being sent out.
     * @return The pulse type
     */
    public int getPulseType() { return type; }

    /**
     * Sets the type of pulse currently being sent out.
     * @param The new type
     */
    public void setPulseType(int t) { type = t; }

    /**
     * Creates the display
     */
    public synchronized void makeDisplay() {
	Enumeration e = getScripts();

	// Drop old stuff
	display.removeAll();

	// Add each event
	if (e.hasMoreElements() == false)
	    display.add(new Label("      No Scripts      ", Label.CENTER));
	while (e.hasMoreElements())
	    display.add(((Script)e.nextElement()).getDisplay());

	// Update
	display.validate();
    }
}
