/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ResourceEvent.java
 ***************************************************************************************/
package simulator;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.security.*;

/* Local Inclues */
import simulator.Event;
import utilities.Clock;

/**
 * An resource event object
 */
public class ResourceEvent extends Event {
  public static final float NO_CHANGE = Float.MIN_VALUE;
  Resources resource;
  float cur, min, max = 0;
  boolean set = false;
  
  /**
   * Constructor.  The resource state will be set or adjusted
   * by the values entered here.  If you don't want to change
   * a particular value, just use NO_CHANGE for the value
   * of that field.
   * @param s The start time (offset from current time)
   * @param e The execution time
   * @param r The resource to change
   * @param c The change to the current value
   * @param mn The change to the minimum value
   * @param mx The change to the maximum value
   * @param st Set if true, adjust otherwise
   */
  public ResourceEvent(int s, int e, Resources r, float c, float mn, float mx, boolean st) {
    super(s, e);
    resource = r;
    cur = c;
    min = mn;
    max = mx;
    set = st;
  }
  
  /**
   * Constructor.  The resource state will be adjusted
   * by the values entered here.  If you don't want to change
   * a particular value, just use NO_CHANGE for the value
   * of that field.
   * @param s The start time (offset from current time)
   * @param e The execution time
   * @param r The resource to change
   * @param c The new current value
   * @param mn The new minimum value
   * @param mx The new maximum value
   */
  public ResourceEvent(int s, int e, Resources r, float c, float mn, float mx) {
    this(s, e, r, c, mn, mx, false);
  }
  
  /**
   * Returns a string of the event's status, as determined
   * by the simulation clock.  This assumes the event queue
   * is functioning correctly, it does not actually check
   * how the event is being executed, so it should only be
   * called on events returned from a queue slot.
   * @return A String representing the event status
   */
  public String getStatus() {
    int time = Clock.getTime();
    String status = "";
    
    if (time == startTime)
      status = resource + " resource being adjusted";
    else if (time < startTime)
      status = "Resource event not processed";
    else
      status = resource + " resource adjusted.";
    
    status = "ID:" + id + " Status: " + status;
    return (status);
  }
  
  /**
   * Realizes the particular event.  Min and max get set
   * before the current value.
   * @return True if the realization was successfull
   */
	public boolean realize() {
		if (resource == null) return false;
    
		if (set) {
			if (min != NO_CHANGE)
				resource.setMinimum(min);
			if (max != NO_CHANGE)
				resource.setMaximum(max);
			if (cur != NO_CHANGE) {
				Float diff = new Float(resource.getCurrentValue() - cur);
				resource.setCurrentValue(cur);
				resource.getLocale().propagateResourceChange(resource.getLocale(), 
															 resource, diff, null);
				resource.getLocale().clearPropMarks();
			}
		} else {
			if (min != NO_CHANGE)
				resource.setMinimum(resource.getMinimum() + min);
			if (max != NO_CHANGE)
				resource.setMaximum(resource.getMaximum() + max);
			if (cur != NO_CHANGE) {
				resource.setCurrentValue(resource.getCurrentValue() + cur);
				resource.getLocale().propagateResourceChange(resource.getLocale(), 
															 resource, new Float(cur), null);
				resource.getLocale().clearPropMarks();
			}
		}
    
		return true;
	}

  /**
   * Returns the state code, which should hopefully be relatively
   * unique (based on the event's contents) and deterministic
   * @return the state code
   */
  public long stateCode() {
    long code = super.stateCode();

    try {
      ByteArrayOutputStream devnull = new ByteArrayOutputStream(512);
      MessageDigest md = MessageDigest.getInstance("SHA");
      DigestOutputStream mdo = new DigestOutputStream(devnull, md);
      DataOutputStream data = new DataOutputStream(mdo);

      // Enter the data
      data.writeLong(code);
      data.writeFloat(cur);
      data.writeFloat(min);
      data.writeFloat(max);
      data.writeUTF(resource.toString());

      // Compute the hash value
      byte hasharray[] = md.digest();
      for (int i = 0; i < Math.min(8, hasharray.length); i++)
	code += (long)(hasharray[i] & 255) << (i * 8);
    } catch (IOException ignore) {
      System.err.println("Error: " + ignore);
    } catch (NoSuchAlgorithmException complain) {
      throw new SecurityException(complain.getMessage());
    }

    return code;
  }
}
