/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	MessageEvent.java
 ***************************************************************************************/
package simulator;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.security.*;

/* Local Inclues */
import simulator.Agent;
import simulator.Event;
import utilities.*;


/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @version $Revision: 1.2 $
 * @author Regis Vincent (vincent@cs.umass.edu)<P>
 * <B> Description:</B>
 * An message event object
 */
public class MessageEvent extends Event implements Serializable {
  KQMLMessage	msg;
  private static  Log logger;
  protected	Agent	source, dest;
  
  /**
   * Constructor
   * @param src The event's source agent
   * @param dst The event's destination agent
   * @param str The message to send
   * @see Event#Event(int, int)
   */
  public MessageEvent(int s, int e, Agent src, Agent dst, KQMLMessage str) {
    super(s, e);
    source = src;
    dest = dst;
    msg = str;
    logger = Log.getDefault();
    logger.log("(Message event from " + src.getName() + " to " + dst.getName() + " content : " + str + " In " + (e-s+1) + " Clicks", 2);
  }
  
  /**
   * Returns a string of the event's status, as determined
   * by the simulation clock.  This assumes the event queue
   * is functioning correctly, it does not actually check
   * how the event is being executed, so it should only be
   * called on events returned from a queue slot.
   * @return A String representing the event status
   */
  public String getStatus() {
    int time = Clock.getTime();
    String status = "";
    
    if (time == startTime)
      status = "Message sent by sender.";
    else if (time == (startTime + execTime))
      status = "Message has arrived at receiver.";
    else if (time < startTime)
      status = "Message waiting to be sent.";
    else if (time > (startTime + execTime))
      status = "Message delivered.";
    else 
      status = "Message en route to receiver.";
    
    status = "ID:" + id + " Status: " + status;
    return (status);
  }
  
  /**
   * Realizes the particular event
   * @return True if the realization was successfull
   */
  public boolean realize() {
    int time = Clock.getTime();
    if (time == (startTime + execTime))
      dest.sendMsg(msg);
    return true;
  } 

  /**
   * Updates the status display<BR>
   * This function is used in the EventQueue display window. 
   * The display looks like that: <BR>
   * <I>Message from </I>  <B>agent1</B> to  <B>agent2</B>  <B>T:</B> -4
   */
  public void updateDisplay() {
    //String monitore  = new String("Message from " + source.getName() + 
    //				  " to " + dest.getName() + " T:" + 
    //				  (Clock.getTime() -  (startTime + execTime)));
    //display.setText(monitore);
  }

  
  /**
   * Returns the state code, which should hopefully be relatively
   * unique (based on the event's contents) and deterministic
   * @return the state code
   */
  public long stateCode() {
    long code = super.stateCode();

    try {
      ByteArrayOutputStream devnull = new ByteArrayOutputStream(512);
      MessageDigest md = MessageDigest.getInstance("SHA");
      DigestOutputStream mdo = new DigestOutputStream(devnull, md);
      DataOutputStream data = new DataOutputStream(mdo);

      // Enter the data
      data.writeLong(code);
      data.writeUTF(msg.toString());
      if (source != null)
        data.writeUTF(source.getName());
      data.writeUTF(dest.getName());

      // Compute the hash value
      byte hasharray[] = md.digest();
      for (int i = 0; i < Math.min(8, hasharray.length); i++)
	code += (long)(hasharray[i] & 255) << (i * 8);
    } catch (IOException ignore) {
      System.err.println("Error: " + ignore);
    } catch (NoSuchAlgorithmException complain) {
      throw new SecurityException(complain.getMessage());
    }

    return code;
  }
}
