/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
* $Source: /nfs/shelob/cvs/masl/Mass/modules/simulator/simulator/Listener.java,v $
* $Revision: 1.2 $
******************************************************************
*
*                   Listener.java
*
*     Author : Regis Vincent [vincent@cs.umass.edu]
*              
*
*     Creation date: 22 Aug 97 12:37
*  Last file update: $Date: 2005/01/19 19:00:05 $
*****************************************************************/
package simulator;

/* Global import */
import java.net.*;
import java.io.*;
import java.util.*;

/* Local Import */
import simulator.Agent;
import utilities.*;
import taems.*;
import taems.parser.*;

/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @version $Revision: 1.2 $
 * @author Regis Vincent (vincent@cs.umass.edu), Bryan Horling (bhorling@cs.umass.edu)<P>
 * <B> Description:</B>
 * The Listener is a independent thread that spent all its time to listen or 
 * written on a particular socket. Attached each agent connected, the Listener
 * defines the behaviour of agent message. When a incoming message arrive, the
 * Listener call the processMsg function, which do something related to the 
 * message type. For example, if a ack pulse message comes, the processMsg
 * changes the agent status from running to waiting.
 */
public class Listener implements  Runnable {
  private EventQueue	queue;
  private BufferedReader input;
  private BufferedWriter output;
  private Agent agent;
  private Log logger;
  private ReadTTaems reader;

  /**
   * Constructor that initializes the private variables and begins a new 
   * Thread.
   * @param a is the Agent attached to this socket.
   * @param i is the BufferedReader (created by the Server).
   * @param o is the BufferedWriter (also created by the Server).
   */
  public Listener(Agent a, BufferedReader i, BufferedWriter o) {
    input = i;
    output = o;
    agent = a;
    logger = Log.getDefault();
    reader = new ReadTTaems(logger);
    agent.setListener(this) ;
    queue = EventQueue.getQueue();
    logger.log("(Listener: " + agent.getName() + ") Listener registered with agent " + agent.getName(), 1);
    new Thread(this).start();
  }

 /**
   * sendMsg is the generic function for sending a message. The message
   * is actually a KQML Message. 
   * @param ms is KQML message
   * @see agent.simpliest.KQMLMessage
   */
  public void sendMsg(KQMLMessage m)
  {
    if ((m.getSourceAddr() == null) || (m.getSourceAddr().equals("")))
      m.setSourceAddr("simulator");
    logger.log("(Listener: " + agent.getName() + ") Sending message \"" + 
	       m + "\" to " + agent.getName(), 3);
    m.send(output);
  }
  

  /**
   * processMsg is called, each time that an incoming message comes. The aim 
   * to this function is to define the behaviour depending of the message type.
   * @param KQML Message
   */
    public synchronized  void processMsg(KQMLMessage msg) {
        logger.log("(Listener: " + agent.getName() + ") Received message \"" + msg 
                   + "\" from " + agent.getName(), 2);

        String receiver = msg.getDestAddr();
        String contentword = msg.contentWord();
        String contentdata = msg.contentData();

        if (contentword == null) { contentword = "" ; }
        if (contentdata == null) { contentdata = "" ; }

        // It was not for the simulator 
        if (!receiver.equalsIgnoreCase("simulator")) {
            if (receiver.equalsIgnoreCase("*")) 
                agent.broadcastMsg(msg);
            else {
                Agent to = agent.findAgent(receiver);
                if (to != null) {
                    MessageEvent me = 
                        new MessageEvent(0, 0, this.agent, to, msg);
                    queue.addEvent(me);
                } else
                    logger.log("(Listener: " + agent.getName() + ") Message not correct: \"" + 
                               msg + "\" from agent " + agent.getName());
            } 
        }
        else  { // For the simulator
            String perf = msg.getPerformative();
            // Pulse
            if (perf.equalsIgnoreCase("reply")) {
                String reply = msg.getField("in-reply-to");
                if ((reply.equalsIgnoreCase("pulse")) || 
                    (contentdata.equalsIgnoreCase("pulse"))) {
                    logger.log("(Listener: " + agent.getName() + ") Message is an pulse reply.", 3);
                    agent.waiting();
                }
                // Disconnect
                else if (reply.equalsIgnoreCase("disconnect")) {
                    logger.log("(Listener: " + agent.getName() + ") Message is an reply of disconnect.", 3);
                    logger.log("(Listener: " + agent.getName() + ") Disconnecting agent " + agent.getName());
                    agent.disconnect();
                    agent.destroy();
                    //Thread.currentThread().stop();
                }
            }
            else if (perf.equalsIgnoreCase("ask")) {

                if (contentword.equalsIgnoreCase("AllAgents")) {
                    // GetAllAgents
                    logger.log("(Listener: " + agent.getName() + ") Message is a GetAllAgents.", 3);
                    KQMLMessage reply = new KQMLMessage("reply", "( ConnectedAgent " +
                                                        this.agent.getAllAgents() + " )"
                                                        , msg.getSourceAddr() );
                    reply.addField("in-reply-to", msg.getField("reply-with"));
                    this.agent.sendMsg(reply);
                }
                else if (contentword.equalsIgnoreCase("Sensor")) {
                    // Resource R1
                    if (contentdata != null) {
                        agent.runSensing(contentdata, msg.getField("reply-with") );
                    }
                }
                else if (contentword.equalsIgnoreCase("Status")) { 
                    // Status M1
                    if (contentdata != null) {
                        StringTokenizer s = new StringTokenizer(contentdata);
                        String methodName = s.nextToken(); //get the MethodName
                        String ID = msg.getField("reply-with");
                        if (ID == null) 
                            ID= methodName;
                        KQMLMessage reply = new KQMLMessage("reply", "( Status " + 
                                                            agent.getMethodStatus(ID) +
                                                            " )", msg.getSourceAddr() );
                        reply.addField("in-reply-to", msg.getField("reply-with"));
                        reply.addField("type", "control");
                        this.agent.sendMsg(reply);
                    }

                } else if (contentword.equalsIgnoreCase("ExecuteStart")) {
                    // " ExecuteStart " + a.label + " " + a.duration + " " + a.cost + " 
                    // + a.quality 
                    logger.log("(Listener: " + agent.getName() + ") Message is a ExecuteMethod", 3);
                    StringTokenizer s = new StringTokenizer(contentdata);
                    String methodName = s.nextToken();
                    String ID = msg.getField("reply-with");
                    taems.Method method;
                    if (ID == null) 
                        ID= methodName;
                    /* Get the method */
                    if (Simulator.global_view != null) {
                        //method = (taems.Method)agent.getTaemsHandler().findNode(new Method(methodName, new taems.Agent(agent.getName())));
                        method = (taems.Method)Simulator.global_view.findNode(new Method(methodName, new taems.Agent(agent.getName())));

                        if ((method != null) &&
                            ((!agent.getName().equals(method.getAgent().getLabel())) ||
                             method.getNonLocal()))
                            /* not autorized to execute a method that doesn't belongs to
                               the agents */
                            method = null;

                    }
                    else 
                        method = null;

                    if (method != null)  {
                        methodName = method.getLabel();
                        if (!this.agent.isAlreadyInExecution(ID)) {
                            logger.log("(Listener: " + agent.getName() + ")Method : " +
                                       method + "ID : " + ID ,3);
                            /* Extract Distributions */
                            Outcome outc = method.chooseOutcome(Simulator.getComplexNLE(),
                                                                TaemsRandom.getDefault());
                            Distribution costDst = outc.getCost();
                            Distribution durationDst = outc.getDuration();
                            Distribution qualityDst = outc.getQuality();
                            logger.log("(Listener: " + agent.getName() + ") Dc = " + costDst + " Dd = " + durationDst + " Dq = " + qualityDst, 3);
                            TaemsRandom.getDefault().setDistribution(durationDst);
                            float duration = TaemsRandom.getDefault().nextValue();
                            if (duration < 1) {
                                logger.log("(Listener: " + agent.getName() + 
                                           ")  ERROR Duration less than 1 (" + duration
                                           + ") set to 1",2);
                                duration = 1;
                            }
                            TaemsRandom.getDefault().unsetDistribution();
                            logger.log("(Listener: " + agent.getName() + 
                                       ")  Duration choosen : " + duration,2);
                            ExecuteEvent me = new ExecuteEvent(0,
                                                               (int)duration,
                                                               this.agent,
                                                               method,
                                                               ID,
                                                               durationDst,
                                                               qualityDst, 
                                                               costDst,
                                                               outc);   
	     
                        }
                        else {
                            KQMLMessage reply = new KQMLMessage("error", "( ExecuteError " + 
                                                                methodName  + " ID: " + ID + 
                                                                " methodAlreadyInExecution )", 
                                                                msg.getSourceAddr() );
                            reply.addField("in-reply-to", msg.getField("reply-with"));
                            logger.log("(Listener: " + agent.getName() +
                                       ") Error ExecuteError " + methodName  
                                       + " methodAlreadyInExecution ",1);
                            this.agent.sendMsg(reply);
                        }
                    }
                    else {
                        KQMLMessage reply = new KQMLMessage("error", "( ExecuteError " + 
                                                            methodName  + " methodNotFound )", 
                                                            msg.getSourceAddr() );
                        reply.addField("in-reply-to", msg.getField("reply-with"));
                        logger.log("(Listener: " + agent.getName() + 
                                   ") Error methodNotFound " + methodName,1);
                        this.agent.sendMsg(reply);
                    }
                } 
                else {
                    logger.log("(Listener: " + agent.getName() + ") Unknown message: \"" + msg +
                               "\" from agent " + agent.getName(),1);
                }
            } else if (perf.equalsIgnoreCase("tell")) {
                if (contentword.equalsIgnoreCase("Location"))
                    // Update display location
                    agent.setLocation(contentdata);
                else if (contentword.equalsIgnoreCase("Display"))
                    // Update display picture
                    agent.setPicture(contentdata);
                else if (contentword.equalsIgnoreCase("DisplayImageURL")) {
		    // Update random picture
		    StringTokenizer tok = new StringTokenizer(contentdata);
		    String id = tok.nextToken();
		    String url = tok.nextToken();
		    Display.getDisplay().updateDisplay(id, url);
		}
                else if (contentword.equalsIgnoreCase("DisplayImageLocation")) {
		    // Update random picture
		    StringTokenizer tok = new StringTokenizer(contentdata);
		    String id = tok.nextToken();
		    String loc = tok.nextToken();
		    Display.getDisplay().updateLocation(id, loc);
		}
                else if (contentword.equalsIgnoreCase("AddTaems") ||
                         contentword.equalsIgnoreCase("TextualTaems")) {
                    // Update objective structure
                    logger.log("Reading new objective structure from " + agent.getName(), 2);
                    StringReader r = new StringReader(msg.contentData());
                    Taems objective = reader.readTTaems(r);
                    if (objective == null) {
                        logger.log("TTaems object was unable to be parsed, bailing.", 0);
                        return;
                    } else
                        logger.log("TTaems object parsed", 3);

                    String key = objective.getLabel();
                    if (key == null)
                        key = "Unknown";

                    objective.setLabel(key);
                    agent.updateTaems(key, objective);

                } else if (contentword.equalsIgnoreCase("RemoveTaems")) {
                    // Update objective structure
                    logger.log("Removing objective structure from " + agent.getName(), 2);
                    String key = msg.contentData();
                    if ((key == null) || key.equals(""))
                        agent.removeAllTaems();
                    else
                        agent.removeTaems(key);

                } else if (contentword.equalsIgnoreCase("ExecuteSet")) {
                    // " ExecuteSet " + a.label + " " + a.start + " " + a.finish + " 
                    // + a.cost + " " + a.quality
                    logger.log("(Listener: " + agent.getName() + ") Message is a ExecuteSet", 3);
                    StringTokenizer s = new StringTokenizer(contentdata);
                    String methodName = s.nextToken();
                    int start = Integer.parseInt(s.nextToken());
                    int finish = Integer.parseInt(s.nextToken());
                    float cost = Float.parseFloat(s.nextToken());
                    float quality = Float.parseFloat(s.nextToken());

                    taems.Method method;
                    /* Get the method */
                    if (Simulator.global_view != null) {
                        method = (taems.Method)Simulator.global_view.findNode(new Method(methodName, new taems.Agent(agent.getName())));
                    }
                    else 
                        method = null;

                    if (method != null)  {
                        method.setStartTime(start);
                        method.setFinishTime(finish);
                        method.setCurrentQuality(quality);
                        method.setCurrentCost(cost);
                        method.setCurrentDuration(finish - start);
                        method.updateNLEs(finish);
                        method.setStatus(Node.COMPLETED);
                    }

                } else if (contentword.equalsIgnoreCase("NewSensor")) {
                    // Resource R1
                    if (contentdata != null) {
                        logger.log("Got remote sensor description: " + contentdata, 3);
                        agent.addSensor(contentdata);
                    }

                } else if (contentword.equalsIgnoreCase("NewScript")) {
                    // Script junk
                    if (contentdata != null) {
                        logger.log("Got a new script. (length " + contentdata.length() + ")", 3);
                        StringTokenizer tok = new StringTokenizer(contentdata, "\n");
                        Vector v = new Vector();
                        while (tok.hasMoreTokens()) {
                            v.addElement(tok.nextToken());
                        }
                        Simulator.getSimulator().getScripter().readScriptVector(v);
                    }
                } else if (contentword.equalsIgnoreCase("RemoveScript")) {
                    if (contentdata != null) {
                        logger.log("Removing a script (" + contentdata + ")", 3);
                        Simulator.getSimulator().getScripter().removeScript(Simulator.getSimulator().getScripter().findScript(contentdata));
                    }
                }
            }
            else if (perf.equalsIgnoreCase("error")) {
                if (contentword.equalsIgnoreCase("ExecuteExtend")) {
                    // "Extend " + time + method.label
                    logger.log("(Listener: " + agent.getName() + ") Message is an ExecuteExtend Message", 3);
                    if (contentdata != null) {
                        StringTokenizer s = new StringTokenizer(contentdata);
                        int time = Integer.parseInt(s.nextToken());
                        String methodName = "";
                        try {
                            methodName = s.nextToken();
                        } catch (NoSuchElementException e) { }
                        /* Get the method */
                        if (methodName.equals("")) 
                            agent.extendAllExecutions(time);
                        else {
                            String ID = msg.getField("reply-with");
                            if (ID == null) 
                                ID= methodName;
                            agent.extend(ID, time);
                        }
                    }
                }
                if (contentword.equalsIgnoreCase("Abort")) {
                    // " ExecuteAbort " + method.label 
                    logger.log("(Listener: " + agent.getName() + ") Message is a Abort Message", 3);
                    if (contentdata != null) {
                        StringTokenizer s = new StringTokenizer(contentdata);
                        String methodName = s.nextToken();
                        /* Get the method */
                        if (methodName.equals("")) 
                            agent.killAllExecutions();
                        else {
                            String ID = msg.getField("reply-with");
                            if (ID == null) 
                                ID= methodName;
                            agent.abort(ID);
                        }
                    }
                    else 
                        agent.killAllExecutions();
                }
            }
        }
    }
  
  /**
   * Infinite loop, that call processMsg() when a message arrives.
   */
  public void run() {
    while(true) {
	//String msg = input.readLine();
	String msg = Message.receive(input);
	logger.log("(Listener: " + agent.getName() + ") message received",5);
	if (msg == null) {
		logger.log("(Listener: " + agent.getName() + ") Error: Unable to read from agent");		
		logger.log("(Listener: " + agent.getName() + ") " + agent.toString());
		agent.disconnect();
		agent.destroy();
		//Thread.currentThread().stop();
	}
	KQMLMessage kqml = new KQMLMessage(msg);
	this.processMsg(kqml);
    }
  }
}
