/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	ExecuteEvent.java
 ***************************************************************************************/
package simulator;

/* Global Includes */
import java.io.*;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.security.*;

/* Local Inclues */
import simulator.Agent;
import simulator.Event;
import taems.*;
import utilities.*;
import simulator.locale.*;

/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @version $Revision: 1.2 $
 * @author Bryan Horling (bhorling@cs.umass.edu), Regis Vincent (vincent@cs.umass.edu), Brett Benyo (bbenyo@cs.umass.edu)<P>
 * <B> Description:</B>
 * An message event object
 */
public class ExecuteEvent extends Event implements Serializable {
  protected	  Agent	source ;
  protected       String ID;
  protected       float[] durationDistribution;
  protected       float[] costDistribution;
  protected       float[] qualityDistribution;
  protected       Outcome chosenOutcome;
  protected	  static TaemsRandom r;
  protected       static Log logger;
  protected       Vector isOverloaded=new Vector();
  protected       boolean waitForOverloading=false;
  protected       Vector limitsNLE= new Vector();
  protected       Hashtable firstClickDimension;
  protected       Hashtable lastClickDimension;
  protected       Hashtable perTimeClickDimension;
  protected       boolean addLastClickDimension=false;
  protected       float expectedFinalQuality;
  protected       float expectedFinalCost;
  public          Method  method;
  
    /**
   * Constructor
   * @param src The event's source agent
   * @param method The method's name to execute
   * @param ID uniq's ID for this execution
   * @param dD the duration Distribution (float [])
   * @param qD the quality distribution (float [])
   * @param cD the cost Duration (float [])
   * @see Event#Event(int, int)
   */
    public ExecuteEvent(int s, int e, Agent src, Method method, String ID,
			float[] dD, float[] qD, float[] cD, Outcome outc) {
	this(s,  e, src,  method, ID);
	Agent source = src;
	float quality, cost;
	chosenOutcome = outc;
	durationDistribution = dD;
	costDistribution = cD;
	qualityDistribution = qD;
	r = TaemsRandom.getDefault();
	logger = Log.getDefault();
	r.setDistribution(qualityDistribution);
	quality = r.nextValue();
	expectedFinalQuality = quality;
	r.unsetDistribution();
	Resource Re = (Resource)perTimeClickDimension.get("quality");
	Re.increment =  quality / (e -s);
	r.setDistribution(costDistribution);
	cost = r.nextValue();
	expectedFinalCost = cost;
	r.unsetDistribution();
	Re = (Resource)perTimeClickDimension.get("cost");
	Re.increment =  cost / (e -s);
	logger.log("(Execute event [" + ID + "] " + method.getLabel() + 
		   ") Final Quality : " + quality + "(" + 
		   ((Resource)perTimeClickDimension.get("quality")).increment + ")" + 
		   " Final Cost : " + cost + "(" + 
		   ((Resource)perTimeClickDimension.get("cost")).increment + ")" + 
		   " In " + (e-s) + " Clicks (Outcome: " + 
		   chosenOutcome.getLabel() + ")",2);
	method.setStartTime(Clock.getTime() + s);
	// If the method is going to fail (zero quality then no consumation, no production)
	EventQueue.getQueue().addEvent(this);
	if (quality != 0) {
            Enumeration enumr = method.getAffectingInterrelationships();
            while (enumr.hasMoreElements())       {
                Interrelationship ir = (Interrelationship)enumr.nextElement();
                if (ir instanceof LimitsInterrelationship)
                    {
                        // I should also check for DURATION INDEPENDENT vs. PERTIME CLICK STUFF
                        logger.log("Adding limits NLE to list",2);
                        limitsNLE.addElement(ir);
                    }
            }
	    // Grab the resource NLEs for this task
	    enumr = method.getAffectedInterrelationships();
	    dealWithResources(enumr);
	}
    }

    protected void renormalizeResources() {
        Enumeration e = perTimeClickDimension.elements();
        while (e.hasMoreElements()) {
            Resource r = (Resource)e.nextElement();
            r.setIncrement(r.getTotal() / execTime);
        }
    }


    /**
     * This ugly function, deals with the resources NLEs for the current method.
     * Most if this work is to sort thru all the NLEs and the resource model they 
     * want to implement and stick that in the right Hashtable.
     * Any comments or adds should be reported to me, please.
     */
    private void dealWithResources(Enumeration enumr) {
	float res_value;
	while (enumr.hasMoreElements())       {
	    Interrelationship ir = (Interrelationship)enumr.nextElement();
	    if (ir instanceof ProducesInterrelationship) {
		logger.log("Adding " + ir.getClass().getName() + " NLE to list",2);
		Resources res;
		try {
		    res = (Resources)Simulator.getSimulator().getResource(source.getLocale(),
									  ir.getTo().getLabel());
		} catch (simulator.locale.Locale.ResourceDoesNotExist x) {
		    res=null;
		}
		if (res != null) {
		    r.setDistribution(ir.getCost());
		    res_value = (float)r.nextValue();
		    r.unsetDistribution();
		    res_value = 0 - res_value;
		    // check model
		    if (ir.getModel().equals(Interrelationship.DURATION_INDEPENDENT)) {
			logger.log("This NLE (" + ir.getClass().getName() +
				   ") is DURATION INDEPENDENT and will be realized at the LAST click",2);
			if (lastClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)lastClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value; 
			    lastClickDimension.remove(res.getName());
			}
			lastClickDimension.put(res.getName(), new Resource(res.getName(),
									   res_value));
		    }
		    else {
			logger.log("This NLE (" + ir.getClass().getName() + 
				   ") is PER TIME UNIT and will be realized at EVERY click",2);
			if (perTimeClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)perTimeClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value;                    
			    perTimeClickDimension.remove(res.getName());
			}
			else
			    addProducedResource(res);
			perTimeClickDimension.put(res.getName(), new Resource(res.getName(),
 									      res_value));
 
			if (firstClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)firstClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value; 
			}
			res.useResources(res_value, this);
			res.getLocale().propagateResourceChange(res.getLocale(), res, new Float(res_value), this);
			res.getLocale().clearPropMarks();
		    }
		}
		else {
		    logger.log("********* Error: What is this resource " +
			       ir.getTo().getLabel() + " *************",0);          
		}
	    }
	    else if (ir instanceof ConsumesInterrelationship) {
		logger.log("Adding " + ir.getClass().getName() + " NLE to list",2);
		Resources res;
		try {
		    res = (Resources)Simulator.getSimulator().getResource(source.getLocale(),
									  ir.getTo().getLabel());
		} catch (simulator.locale.Locale.ResourceDoesNotExist x) {
		    res = null;
		}
		if (res != null) {
		    r.setDistribution(ir.getCost());
		    res_value = (float)r.nextValue();
		    r.unsetDistribution();
		    // check model
		    if (ir.getModel().equals(Interrelationship.DURATION_INDEPENDENT)) {
			logger.log("This NLE (" + ir.getClass().getName() +
				   ") is DURATION INDEPENDENT and will be realized at the FIRST click",2);
			if (firstClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)firstClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value;
			    firstClickDimension.remove(res.getName());
			}
			firstClickDimension.put(res.getName(), new Resource(res.getName(),res_value));

			if (perTimeClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)perTimeClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value;                             }
			else
			    addUsedResource(res);     
			res.useResources(res_value, this);
			res.getLocale().propagateResourceChange(res.getLocale(), res, new Float(res_value), this);
			res.getLocale().clearPropMarks();
		    }
		    else {
			logger.log("This NLE (" + ir.getClass().getName() +
				   ") is PER TIME UNIT and will be realized at EVERY click",2);
			if (perTimeClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)perTimeClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value;    
			    perTimeClickDimension.remove(res.getName());
			}

			perTimeClickDimension.put(res.getName(), new Resource(res.getName(), res_value));

			if (firstClickDimension.containsKey(res.getName())) {
			    Resource  resource = (Resource)firstClickDimension.get(res.getName());
			    res_value = resource.getIncrement() + res_value;  
			}
			else
			    addUsedResource(res);
			
			res.useResources(res_value, this);
			res.getLocale().propagateResourceChange(res.getLocale(), res, new Float(res_value), this);
			res.getLocale().clearPropMarks();
		    }
		}
		else {
		    logger.log("********* Error: What is this resource " +
			       ir.getTo().getLabel() + " *************"      ,0);
		}
	    }
	}
    }

  
  
    /**
   * Constructor
   * @param src The event's source agent
   * @param method The method's name to execute
   * @param ID uniq's ID for this execution
   * @param dD the duration Distribution
   * @param qD the quality distribution
   * @param cD the cost Duration
   * @see Event#Event(int, int)
   */
  public ExecuteEvent(int s, int e, Agent src, Method method, String ID,
		      Distribution dD, Distribution qD, Distribution cD, Outcome outc)
  {
    this(s,e,src,method, ID, dD.getArray(),qD.getArray(),cD.getArray(), outc);
  }

  /**
   * Constructor
   * @param src The event's source agent
   * @param method The method's name to execute
   * @param ID uniq's ID for this execution
   * @see Event#Event(int, int)
   */
  public ExecuteEvent(int s, int e, Agent src, Method method, String ID)
  {
    super(s, e-1);
    source = src;
    this.ID = ID;
    this.method = method;
    src.addExecution(this);
    perTimeClickDimension = new Hashtable();
    perTimeClickDimension.put("quality", new Resource("quality", 0));
    perTimeClickDimension.put("cost", new Resource("cost", 0));
    firstClickDimension = new Hashtable();
    lastClickDimension = new Hashtable();
    chosenOutcome = null;
  }


  /** 
   * compute the NLE that should be updated and create
   * new Events.
   */

  protected void updateNLE(Enumeration NLEs) {
    Enumeration e = NLEs;
    EventQueue queue = EventQueue.getQueue();
    
    logger.log("(Execute Method " + method.getLabel() + ") " + " NLEs to update");
    while (e.hasMoreElements())
      {
	Interrelationship rel = (Interrelationship)e.nextElement();
	// Check for the chosen Outcome
	if (rel.needToBeUpdated()) {
	  // Create the Event !!
	  if (rel instanceof EnablesInterrelationship) {
	    r.setDistribution(rel.getDelay());
	    EnablesEvent event = new EnablesEvent((int)r.nextValue(),
						  (EnablesInterrelationship)rel,source);
	    queue.addEvent(event);
	    r.unsetDistribution();
	  }
	  if (rel instanceof DisablesInterrelationship) {
	    r.setDistribution(rel.getDelay());
	    DisablesEvent event = new DisablesEvent((int)r.nextValue(),
						    (DisablesInterrelationship)rel,source);
	    queue.addEvent(event);
	    r.unsetDistribution();
	  }
	  if (rel instanceof FacilitatesInterrelationship) {
	    r.setDistribution(rel.getDelay());
	    FacilitatesEvent event = new FacilitatesEvent((int)r.nextValue(),
							  (FacilitatesInterrelationship)rel,source);
	    queue.addEvent(event);
	    r.unsetDistribution();
	  }
	  if (rel instanceof HindersInterrelationship) {
	    r.setDistribution(rel.getDelay());
	    HindersEvent event = new HindersEvent((int)r.nextValue(),
						  (HindersInterrelationship)rel,source);
	    queue.addEvent(event);
	    r.unsetDistribution();
	  }
	}
      }
  }
  
  
  /**
   * Returns a string of the event's status, as determined
   * by the simulation clock.  This assumes the event queue
   * is functioning correctly, it does not actually check
   * how the event is being executed, so it should only be
   * called on events returned from a queue slot.
   * @return A String representing the event status
   */
  public String getStatus() {
    int time = Clock.getTime();
    String status = "";
    
    if (time == startTime)
      status = "Event starting up.";
    else if (time == (startTime + execTime))
      status = "Event completing.";
    else if (time < startTime)
      status = "Event waiting for execution.";
    else if (time > (startTime + execTime))
      status = "Event execution completed.";
    else 
      status = "Event " + method.getLabel() + " executing.";
    
    status = "ID:" + id + " Status: " + status;
    return (status);
  }
  
  /**
   * Aborts the event before execution completed. This generate an 
   * ExecuteAbort message. The message has the following form:<BR>
   * <B>ExecuteAbort</B> <I>MethodName</I> Duration Cost Quality 
   * [<B>Used</B> ResourceName Value]+
   * @param a : ID or Method Name
   */
  public void abort(String a) {
    if (a == null || (source.getName().equals(a)) || (ID.equals(a)))
      {
	logger.log("Aborting execution of "+this.method.getLabel());
	
	// Aborted methods may have zero quality
	String type = (String)method.getAttribute("ExecutionType");
	if (type != null && type.equalsIgnoreCase("anytime")) {
	   // Then it's anytime and we leave it alone
	} else {
	   // Otherwise we nix it
           Resource Res = (Resource)perTimeClickDimension.get("quality");
           Res.setTotal(0);
	   Res.setIncrement(0);
	}

	EventQueue q = EventQueue.getQueue();
	q.removeEvent(this);
	
	// Sending back informations
	sendEndExecution("ExecuteAbort");
	execTime = 0;
	
	// Free Resources !!!
	super.abort(a);
      }
    source.removeExecution(this);
  }

  
  /** 
   * Finished an execution and clean all the mess up.
   */
  private void sendEndExecution(String msgType) {
    int time = Clock.getTime();
    //    Simulator.global_view.setNodeStatus(method.getLabel(), Node.COMPLETED);
    method.setFinishTime(time);
    method.setCurrentQuality(((Resource)perTimeClickDimension.get("quality")).total);
    method.setCurrentCost(((Resource)perTimeClickDimension.get("cost")).total);
    method.setCurrentDuration(this.execTime+1);
    method.setStatus(Node.COMPLETED);

    String content = msgType + " "  + this.method.getLabel() + " " +
      (this.execTime+1) + " " + ((Resource)perTimeClickDimension.get("cost")).total +
      " " + ((Resource)perTimeClickDimension.get("quality")).total ; 
    if (perTimeClickDimension.size() > 2) {
      content = content + " Used" ;
      for (Enumeration e = perTimeClickDimension.keys(); e.hasMoreElements(); ) { 
	Resource Re = (Resource)perTimeClickDimension.get(e.nextElement()); 
	if (!(Re.name.equals("cost")) && !(Re.name.equals("quality")))
	  content = content + " " + Re.name + " " + Re.total;
      }
    } 
    KQMLMessage reply = new KQMLMessage("reply", "(" + content +")", 
					this.source.getName() );
    reply.setSourceAddr("simulator");
    reply.addField("in-reply-to", ID);
    this.source.sendMsg(reply);

    // Update resources
    freeUsedResources();
    freeProducedResources();

    if (((Resource)perTimeClickDimension.get("quality")).total > 0)
      updateNLE(method.getAffectedInterrelationships());

  }

  
  /**
   * Realizes the particular event. In the ExecuteEvent case, this function 
   * tests if the method is done. If it's true, the simulator returns this 
   * message:<BR>
   * <B>ExecuteEnd</B> <I>MethodName</I> Duration Cost Quality [<B>Used</B> ResourceName Value]+
   * @return True if the realization was successfull
   * 
   */
  public boolean realize() {
    Enumeration e;
    int time = Clock.getTime();
    
    if ((time >= startTime) && (time <= (startTime + execTime)))
      {
        method.setStatus(Node.EXECUTING);
	for (e = perTimeClickDimension.keys(); e.hasMoreElements(); ) { 
	  Resource Re = (Resource)perTimeClickDimension.get(e.nextElement()); 
	  logger.log(Re.toString(),5);
	  Re.updateTotal();
	  if (waitForOverloading) {
	      Re.removeTotal();
	      Re.setOverload(0);
	  }
	}
	logger.log("(Execute Method " + this.method.getLabel() + " [" + ID +
		   "]) Quality = " + 
		   ((Resource)perTimeClickDimension.get("quality")).total + 
		   " Cost = " + ((Resource)perTimeClickDimension.get("cost")).total ,1);
	// That means that the Produces stuff was limited at the end of the
	// last click, I should remove the entry.
	if (addLastClickDimension) {
	    float res_value;
	    Resources res;
	    Resource Re;
	    addLastClickDimension=false;
	    for (e = lastClickDimension.keys(); e.hasMoreElements(); ) { 
		Re = (Resource)lastClickDimension.get(e.nextElement()); 
		try {
		    res = (Resources)Simulator.getSimulator().getResource(source.getLocale(),Re.name);
		} catch (simulator.locale.Locale.ResourceDoesNotExist x) {
		    continue;
		}
		Re.updateTotal();
		
		if (perTimeClickDimension.containsKey(res.getName())) {
		    Resource  resource = (Resource)perTimeClickDimension.get(res.getName());
		    res_value = resource.getIncrement();
		    res.useResources(resource.getIncrement(), this);
		    res.getLocale().propagateResourceChange(res.getLocale(), res, new Float(res_value), this);
		    res.getLocale().clearPropMarks();
		}
		else {
		    freeUsedResource(res);
		}
	    }
	}
	// Before the last click, Add the lastClickDimension
	if (time ==  (startTime + execTime)) {
	    float res_value;
	    Resources res;
	    Resource Re;
	    addLastClickDimension = true;
	    for (e = lastClickDimension.keys(); e.hasMoreElements(); ) { 
		Re = (Resource)lastClickDimension.get(e.nextElement()); 
		res_value = Re.getIncrement();
		Re.updateTotal();
		try {
		    res = (Resources)Simulator.getSimulator().getResource(source.getLocale(),Re.name);
		} catch (simulator.locale.Locale.ResourceDoesNotExist x) {
		    continue;
		}
		if (perTimeClickDimension.containsKey(res.getName())) {
		    Resource  resource = (Resource)perTimeClickDimension.get(res.getName());
		    res_value = resource.getIncrement() + res_value;
		} 
		else 
		    addProducedResource(res);
		res.useResources(res_value, this);
		res.getLocale().propagateResourceChange(res.getLocale(), res, new Float(res_value), this);
		res.getLocale().clearPropMarks();
	    }
	}
	updateUsedResources();
	if (time == startTime) {
	    float res_value;
	    Resources res;
	    Resource Re;
	    for (e = firstClickDimension.keys(); e.hasMoreElements(); ) { 
		Re = (Resource)firstClickDimension.get(e.nextElement()); 
		try {
		    res = (Resources)Simulator.getSimulator().getResource(source.getLocale(),Re.name);
		} catch (simulator.locale.Locale.ResourceDoesNotExist x) {
		    continue;
		}
		Re.updateTotal();
		if (perTimeClickDimension.containsKey(res.getName())) {
		    Resource  resource = (Resource)perTimeClickDimension.get(res.getName());
		    res_value = resource.getIncrement();
		    res.useResources(resource.getIncrement(), this);
		    res.getLocale().propagateResourceChange(res.getLocale(), res, new Float(res_value), this);
		    res.getLocale().clearPropMarks();
		}
		else {
		    freeUsedResource(res);
		}
	    }
	}
	if (!waitForOverloading)
	  updateProducedResources();
	waitForOverloading = false;
	isOverloaded.removeAllElements();
      }
    
    if (time == (startTime + execTime)) {
      sendEndExecution("ExecuteEnd");
      source.removeExecution(this);
      logger.log(monitoreEvent(),2);
    }
    
    updateDisplay();
    return true;
  }
  

  /**
   * In case of overloading this will uses the limits NLE's.
   * @param resource is the resource name
   * @param value is how much this resource is overloaded.
   */
  public synchronized void overloadResources(String resource, float value) { 
    // Just nullify this pulse for the quality
    logger.log("(ExecuteEvent " + method.getLabel() + "_" + source.getName() +
	       ") Resources " + resource + " Overloaded " + value,1);
    if (!isOverloaded.contains(resource)) {
      Enumeration enumr = limitsNLE.elements();
      float powerValue, overloadValue, incrementValue;
      logger.log("Overloading : ready to look at " + limitsNLE.size() + 
		 " limits",5);
      int time = Clock.getTime();
      while (enumr.hasMoreElements())       {
	Resource Res;
	boolean durationIndepedant=false;
	Interrelationship ir = (Interrelationship)enumr.nextElement();
	if (ir.getModel().equals(Interrelationship.DURATION_INDEPENDENT))
	    durationIndepedant = true;
	if (ir.getFrom().getLabel().equals(resource))
	  {
	    logger.log("Overloading :" + ir.getClass().getName() + " in progress...",2);
	    if (ir.getCost() != null) {
	      r.setDistribution(ir.getCost());
	      powerValue = (float)r.nextValue();
	      r.unsetDistribution();
	      /* implement this f*cking stupid model */
	      powerValue = 1 +  powerValue;
	      Res = (Resource)perTimeClickDimension.get("cost");
	      incrementValue = Res.getIncrement();
	      overloadValue = incrementValue - (powerValue * incrementValue);
	      if (durationIndepedant) {
		  expectedFinalCost = powerValue * expectedFinalCost;
		  if (expectedFinalCost < 0) 
		      expectedFinalCost = 0;
		  incrementValue = expectedFinalCost / (getFinish() - getStart() +1);
		  if (incrementValue == -0) 
		      incrementValue = 0;
		  float newTotal = incrementValue * (time - getStart()+1);
		  logger.log("Overloading : " + ir.getClass().getName() + 
			     " New Total Cost " + expectedFinalCost + 
			     "$ New Increment: " + incrementValue + 
			     "$ (current cost = " + newTotal + "$)" ,5);
		  Res.setTotal(newTotal);
		  Res.setIncrement(incrementValue);
	      } else {
		  expectedFinalCost = expectedFinalCost - overloadValue;
		  logger.log("Overloading : " + ir.getClass().getName() + 
			     " New Total Cost " + expectedFinalCost + 
			     "$ New Increment: " + incrementValue + 
			     "$ (overload value = " + overloadValue + "$)" ,5);
		  if (!waitForOverloading) {
		      Res.setOverload(overloadValue);
		      waitForOverloading = Res.removeTotal();
		      logger.log("Wait = " + waitForOverloading, 5);
		      if (!waitForOverloading)
			  Res.setOverload(0);
		  }
		  else {
		      logger.log("Wait = " + waitForOverloading + " Overload value = " + 
				 overloadValue + Res.getOverload(), 5);
		      Res.setOverload(overloadValue + Res.getOverload());
		  }
	      }
	    }
 	    if (ir.getQuality() != null) {
 	      r.setDistribution(ir.getQuality());
 	      powerValue = (float)r.nextValue();
	      r.unsetDistribution();
	      /* implement this f*cking stupid model */
	      powerValue = 1 -  powerValue;
	      if (powerValue < 0) 
		  powerValue = 0;
	      Res = (Resource)perTimeClickDimension.get("quality");
 	      incrementValue = Res.getIncrement();
 	      overloadValue = incrementValue - (powerValue * incrementValue);
	      Res.setOverload(overloadValue);
	      if (durationIndepedant) {
		  expectedFinalQuality = powerValue * expectedFinalQuality;
		  if (expectedFinalQuality < 0) 
		      expectedFinalQuality = 0;
		  incrementValue = expectedFinalQuality / (getFinish() - getStart() +1);
		  if (incrementValue == -0) 
		      incrementValue = 0;
		  float newTotal = incrementValue * (time - getStart()+1);
		  logger.log("Overloading : " + ir.getClass().getName() + 
			     " New Total Quality " + expectedFinalQuality + 
			     " New Increment: " + incrementValue + 
			     " (current quality = " + newTotal + ")" ,5);
		  Res.setTotal(newTotal);
		  Res.setIncrement(incrementValue);
	      } else {
		  expectedFinalQuality = expectedFinalQuality - overloadValue;
		  logger.log("Overloading : " + ir.getClass().getName() + 
			     " New Total Quality " + expectedFinalQuality + 
			     " New Increment: " + incrementValue +
			       " (overload value = " + overloadValue + ")" ,5);
		  if (!waitForOverloading) {
		      Res.setOverload(overloadValue);
		      waitForOverloading = Res.removeTotal();
		      logger.log("Wait = " + waitForOverloading, 5);
		      if (!waitForOverloading)
			  Res.setOverload(0);
		  }
		  else {
		      logger.log("Wait = " + waitForOverloading + " Overload value = " + 
				 overloadValue + Res.getOverload(), 5);
		      Res.setOverload(overloadValue + Res.getOverload());
		  }
	      }
	    }
 	    if (ir.getDuration() != null) {
 	      r.setDistribution(ir.getDuration());
 	      powerValue = (float)r.nextValue();
	      r.unsetDistribution();
	      /* implement this f*cking stupid model */
	      powerValue =  powerValue;
	      logger.log("Overloading :" + ir.getClass().getName() + 
			 " Power time = " +  powerValue,5);
 	      int extendedTime = Math.round(powerValue);
	      if (durationIndepedant) {
		  logger.log("Overloading :" + ir.getClass().getName() + 
			      " duration = " +  (getFinish()-getStart()+1) +
			     " start = " + getStart() + " end = " + getFinish() ,5);
		
		  extendedTime = Math.round((getFinish()-getStart()+1) * powerValue );
	      }
 	      logger.log("Overloading :" + ir.getClass().getName() + 
			 " Time extended  " +  extendedTime,5);
	      if (extendedTime != 0) {
		  EventQueue.getQueue().extendEvent(this,extendedTime);
		  if (durationIndepedant) {
		      incrementValue = expectedFinalQuality / (getFinish() - getStart() +1);
		      if (incrementValue == -0) 
			  incrementValue = 0;
		      float newTotal = incrementValue * (time - getStart()+1);
		      logger.log("Overloading : " + ir.getClass().getName() + 
				 " New Total Quality " + expectedFinalQuality + 
				 " New Increment: " + incrementValue + 
				 " (current quality = " + newTotal + ")" ,5);
		      Res = (Resource)perTimeClickDimension.get("quality");
		      Res.setTotal(newTotal);
		      Res.setIncrement(incrementValue);
		      incrementValue = expectedFinalCost / (getFinish() - getStart() +1);
		      if (incrementValue == -0) 
			  incrementValue = 0;
		      newTotal = incrementValue * (time - getStart()+1);
		      logger.log("Overloading : " + ir.getClass().getName() + 
				 " New Total Cost " + expectedFinalCost + 
				 "$ New Increment: " + incrementValue + 
				 "$ (current cost = " + newTotal + "$)" ,5);
		      Res = (Resource)perTimeClickDimension.get("cost");
		      Res.setTotal(newTotal);
		      Res.setIncrement(incrementValue);
		  }
	      }
	    }
	  }
      }
      // Wait until execution
      waitForOverloading = true;
      // Avoid to reduce several times the quality for one method
      isOverloaded.addElement(resource);
    }		
    if (perTimeClickDimension.containsKey(resource)) {
        ((Resource)perTimeClickDimension.get(resource)).total = 
	    ((Resource)perTimeClickDimension.get(resource)).total + value;
    }
    else if (firstClickDimension.containsKey(resource)) {
	((Resource)firstClickDimension.get(resource)).total = 
	    ((Resource)firstClickDimension.get(resource)).total + value;
    }  
    else if (lastClickDimension.containsKey(resource)) {
	((Resource)lastClickDimension.get(resource)).total = 
	    ((Resource)lastClickDimension.get(resource)).total + value;
    }
  }
  

  /**
   * Updates the status display<BR>
   * This function is used in the EventQueue display window. 
   * The display looks like that: <BR>
   * <I>MethodName</I>  <B>C:</B> 2.4 <B>Q:</B> 34.43 <B>T:</B> -4
   */
  public void updateDisplay() {
    String monitore  = new String(source.getName() +
				  " " + method.getLabel() + " C:" + 
				  ((Resource)perTimeClickDimension.get("cost")).total + " Q:" + 
				  ((Resource)perTimeClickDimension.get("quality")).total + " T:" + 
				  (Clock.getTime() -  (startTime + execTime)));
    display.setText(monitore);
  }

  /** 
   * returns a String containing the informations about the ExecuteEvent
   * It looks like : <BR>
   * <I>MethodName</I> <B>Running</B> <B>C:</B> 2.4 
   * <B>Q:</B> 34.43 <B>T:</B> -4 [<B>Used</B> ResourceName Value]+
   */
  public String monitoreEvent() {
    String monitore  = new String(method.getLabel() + " Running " + 
		((Resource)perTimeClickDimension.get("cost")).total + " " + 
		((Resource)perTimeClickDimension.get("quality")).total + " " + 
		(Clock.getTime() -  (startTime + execTime)));
    if (perTimeClickDimension.size() > 2) {
      monitore = monitore + " Used" ;
      for (Enumeration e = perTimeClickDimension.keys(); e.hasMoreElements(); ) { 
	  Resource Re = (Resource)perTimeClickDimension.get(e.nextElement()); 
	  if (!(Re.name.equals("cost")) && !(Re.name.equals("quality"))) {
	      float upper0Total =  Re.total;
	      if (upper0Total < 0) 
		  upper0Total = (float)0.0;
	      monitore = monitore + " " + Re.name + " " + upper0Total;
	  }
      }
    }
    if (firstClickDimension.size() > 0) {
	for (Enumeration e = firstClickDimension.keys(); e.hasMoreElements(); ) { 
	    Resource Re = (Resource)firstClickDimension.get(e.nextElement()); 
	    float upper0Total =  Re.total;
	    if (upper0Total < 0) 
		upper0Total = (float)0.0;
	    monitore = monitore + " " + Re.name + " " + upper0Total;
	}
    }
    if (lastClickDimension.size() > 0) {
	for (Enumeration e = lastClickDimension.keys(); e.hasMoreElements(); ) { 
	    Resource Re = (Resource)lastClickDimension.get(e.nextElement()); 
	    float upper0Total =  Re.total;
	    if (upper0Total < 0) 
		upper0Total = (float)0.0;
	    monitore = monitore + " " + Re.name + " " + upper0Total;
	}
    }
    return(monitore);
  }

  /**
   * getMethodName returns the method Name of this event
   */
  public String getMethodName() {
    return method.getLabel();
  }

  /**
   * getMethod returns the Method object
   */
  public Method getMethod() {
    return method;
  }

  /**
   * Acessor for the perTimeClickDimension table
   * @return A reference to the perTimeClickDimension table
   */
  public Hashtable getDimensions() { return perTimeClickDimension; }

  /**
   * Acessor for the total for a specific entry in the perTimeClickDimension table
   * @return The current total, if it exists, or Float.NaN if it don't
   */
  public float getDimension(String dim) {

     if (perTimeClickDimension.containsKey(dim)) {
        Resource Re = (Resource)perTimeClickDimension.get(dim);
        return Re.total;
     }

     return Float.NaN;
  }

  /**
   * Acessor for the total for a specific entry in the dimensions table
   * @param dim The dimension to adjust
   * @param total The new total
   */
  public void setDimension(String dim, float v) {

     if (perTimeClickDimension.containsKey(dim)) {
        Resource Re = (Resource)perTimeClickDimension.get(dim);
        Re.total = v;
     }
  }

  /**
   * GetExeID() return the Execution ID given by the agent.
   * @return ID 
   */
  public String getExeID() { return(ID); }

  /**
   * Returns the outcome chosen for this execution
   * @return chosenOutcome 
   */
  public Outcome getOutcome() { return chosenOutcome; }

  /**
   * Returns the state code, which should hopefully be relatively
   * unique (based on the event's contents) and deterministic
   * @return the state code
   */
  public long stateCode() {
    long code = super.stateCode();

    try {
      ByteArrayOutputStream devnull = new ByteArrayOutputStream(512);
      MessageDigest md = MessageDigest.getInstance("SHA");
      DigestOutputStream mdo = new DigestOutputStream(devnull, md);
      DataOutputStream data = new DataOutputStream(mdo);

      // Enter the data
      data.writeLong(code);
      data.writeUTF(source.getName());
      data.writeUTF(method.getLabel());
      data.writeUTF(ID);
      data.writeUTF(monitoreEvent());

      // Compute the hash value
      byte hasharray[] = md.digest();
      for (int i = 0; i < Math.min(8, hasharray.length); i++)
	code += (long)(hasharray[i] & 255) << (i * 8);
    } catch (IOException ignore) {
      System.err.println("Error: " + ignore);
    } catch (NoSuchAlgorithmException complain) {
      throw new SecurityException(complain.getMessage());
    }

    return code;
  }


  /**
   * Internal class used to store how much resources this execution has taken.
   * For each execution, we create a Resource intance per resource used i.e., 
   * one for Hot Water, one for Electricity, one for Noise. We use theses
   * instances to keep track of how many units this execution consumes.
   */

  class Resource extends Object {
    public String name;
    public float  increment;
    public float  total;
    public float  overload;
    /** 
     * Generic Constructor
     */
    public Resource() {}
    
    /**
     * Constructor for the resource n
     */
    public Resource(String n) { name = n; }

    /**
     * Constructor for the resource n and the execution will consume i 
     * units per click
     */
    public Resource(String n, float i) {
      name = n;
      increment = i;
    }

    /**
     * Increment the total used with the increment !
     */
    public void updateTotal() { total = total + increment ; }
   
    /**
     * Decrement the total used with the increment, this function is used when
     * the resource has been overloaded.
     * @return a boolean, true means, it couldn't deleted the total (non
     * zero quality non accepted
     * false, means everything is fine.
     */
    public boolean removeTotal() { 
      total = total - overload ; 
      // limit the quality to zero, not negative value.
      if (total < 0) {
	total = 0; 
	return true;
      }
      return false;
    }
    public void setTotal(float t) { total = t; }
    public float getTotal() { return total; }
    public float getIncrement() { return increment; }
    public void setIncrement(float inc) {  increment = inc;}
    public void setOverload(float inc) { overload = inc;}
    public float getOverload() { return overload; }

    public String toString() {
      return("[" + name  +", " + total + " ("+ increment + ")]" );
    }
  }
}



