/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	EventQueue.java
 ***************************************************************************************/
package simulator;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.awt.*;

/* Local Includes */
import simulator.*;
import utilities.*;

/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @version $Revision: 1.2 $
 * @author Bryan Horling (bhorling@cs.umass.edu)<p>
 * <B> Description:</B>
 * The event "queue" is actually a hashtable of vectors, where each vector
 * in the table is a collection of the events which will terminate, or take
 * effect, during that descrete time interval.  A hashtable was used to store
 * these objects to facilitate a (possibly) sparse and dynamic temporally
 * ordered range of objects.
 */
public class EventQueue extends Hashtable {
  private static final int HASH_CAPACITY = 53;
  private static final int SLOT_CAPACITY = 10;
  private static EventQueue queue = new EventQueue();
  private Random random;
  private Log logger;
  private Panel display;
	
  /**
   * Private constructor
   */
  private EventQueue() {
    super(HASH_CAPACITY);
  }
  
  /**
   * Returns the global event queue
   * @return The event queue
   */
  public static EventQueue getQueue() {
    return queue;
  }
  
  /**
   * Inits the queue
   * @return The event queue
	 */  
  public void init() {
    logger = Log.getDefault();
    random = new Random(TaemsRandom.getDefault().getSeed());
    
    // Setup the display
    display = new Panel();
    display.setLayout(new VFlowLayout(VFlowLayout.TOP));
    display.setFont (new Font ("Helvetica", Font.BOLD , 14));
    setDisplay(0);
  }		
  
  /**
   * Adds an event to the correct slot
   * @param e The element to add
   */
  public synchronized void addEvent(Event e) {
    try {
      logger.log("(EQ) Adding event " + e.getID(), 3);
      for (int time = e.getStart(); time <= e.getFinish(); time++) {
	logger.log("(EQ) Adding event to time slot " + time, 3);
	//getTimeSlot(time).insertElementAt(e, 0);
	getTimeSlot(time).addElement(e);
	logger.log("(EQ) Event " + e.getID() + " added to slot " + time, 3);
	//try { wait(1); } catch (InterruptedException ex) { logger.log("(EQ) Error during addEvent pause: " + ex); }
      }
      logger.log("(EQ) Event " + e.getID() + " added.", 3);
      logger.log(e.getStatus());
      
    } catch (Exception ex) {
      logger.log("(EQ) Error while adding event " + e, 0);
      logger.log("(EQ) Exception: " + ex, 0);
    }
  }
  
  /**
   * Removes an element from the table
   * @param e The element to remove
   * @return true if the event was removed, false otherwise
   */
  public synchronized boolean removeEvent(Event e) {
    boolean success = false;
    
    try {
      logger.log("(EQ) Removing event " + e.getID(), 3);
      for (int time = e.getStart(); time <= e.getFinish(); time++) {
	logger.log("(EQ) Removing event from time slot " + time, 3);
	success = (getTimeSlot(time).removeElement(e) || success);
	if (success)
	  logger.log("(EQ) Event " + e.getID() + " removed from slot " + time, 3);
	else
	  logger.log("(EQ) Event " + e.getID() + " not removed from slot " + time, 3);
      }
      logger.log("(EQ) Event " + e.getID() + " removed.", 3);
      
      
    } catch (Exception ex) {
      logger.log("(EQ) Error while removing event " + e);
      logger.log("(EQ) Exception: " + ex);
    }
    return success;
  }
  
  /**
   * Finds an event in the queue by its id number.  This
   * may be obtained from any time during its execution.
   * @param id The id number to search for
   * @return The event, or null if none found
   */
  public synchronized Event findEvent(long id) {
    Enumeration	slots = keys();
    Vector        slot;
    Enumeration	events;
    Event		e;
    
    logger.log("(EQ) Looking for event " + id, 3);
    while(slots.hasMoreElements()) {
      slot = (Vector)this.get(slots.nextElement());
      events = slot.elements();
      while(events.hasMoreElements()) {
	e = (Event)events.nextElement();
	if (e.getID() == id) {
	  logger.log("(EQ) Found event " + e.getID(), 4);
	  return e;
	}
      }
    }
    logger.log("(EQ) Didn't find event.", 3);
    return null;
  }
  
  /**
   * Extends the execution time of an event.  Note this does no
   * time referene checking (so it will let you "extend" an event
   * which has already completed, if it has not already been removed).
   * @param e The element to remove
   * @param d The time to extend it by
   * @return true if the event was extended, false otherwise
   */
  public boolean extendEvent(Event e, int d) {
    logger.log("(EQ) Extending event " + e.getID(), 3);

    int ofin = e.getFinish();
    e.extend(d);
    for(int time = ofin + 1; time <= e.getFinish(); time++) {
      getTimeSlot(time).addElement(e);
      logger.log("(EQ) Event " + e.getID() + " added to slot " + time, 4);
    }

    logger.log("(EQ) Event " + e.getID() + " extended.", 3);

    return true;
  }
  
  /**
   * Delays the start execution time of an event.  Note this does no
   * time referene checking (so it will let you "delay" an event
   * which has already started, which you probably don't want).
   * @param e The element to remove
   * @param d The time to extend it by
   * @return true if the event was delayed, false otherwise
   */
  public boolean delayEvent(Event e, int d) {
    
    logger.log("(EQ) Delaying event " + e.getID(), 3);
    if (removeEvent(e)) {
      e.delay(d);
      addEvent(e);
      logger.log("(EQ) Event " + e.getID() + " delayed.", 4);
      return true;
    }
    logger.log("(EQ) Event " + e.getID() + " not delayed.", 3);
    return false;
  }
  
  /**
   * Finds the oldest time slot in the queue
   * @return The slot number of this slot
   */
  public int findOldestTimeSlot() {
    Enumeration	slots = keys();
    Integer		slot;
    int		oldest = Integer.MAX_VALUE;
    
    logger.log("(EQ) Finding oldest time slot.", 3);
    while(slots.hasMoreElements()) {
      slot = (Integer)slots.nextElement();
      if (slot.intValue() < oldest)
	oldest = slot.intValue();
    }
    logger.log("(EQ) Found oldest time slot - " + oldest, 4);
    return oldest;
  }
  
  /**
   * Used to tell if a time slot exists or not.  A non existant
   * time slot has either already occurred or never had events
   * added to it
   * @return true if the time slot exists
   */
  public synchronized boolean hasTimeSlot(int slotNum) {
    return this.containsKey(new Integer(slotNum));
  }
  
  /**
   * Gets the time slot event vector
   * @param slotNum The slot to get
   * @return The slot
   */
  public synchronized Vector getTimeSlot(int slotNum) {
    Vector slot;
    
    logger.log("(EQ) Getting time slot " + slotNum, 3);
    if (hasTimeSlot(slotNum)) {
      slot = (Vector)get(new Integer(slotNum));
    } else {
      slot =  initTimeSlot(slotNum);
    }
    logger.log("(EQ) Time slot " + slotNum + " gotten.", 4);
    return slot;
    //return shuffleTimeSlot(slot);
  }
  
  /**
   * Fetches the time slot event vector, deleting from the table afterwards
   * @param slotNum The slot to fetch
   * @return The slot
   */
  public synchronized Vector fetchTimeSlot(int slotNum) {
    
    logger.log("(EQ) Fetching time slot " + slotNum, 3);
    Vector	slot = this.getTimeSlot(slotNum);
    removeTimeSlot(slotNum);
    logger.log("(EQ) Time slot " + slotNum + " fetched.", 4);
    //return slot;
    return shuffleTimeSlot(slot);
  }
  
  /**
   * Removes a time slot
   * @param slotNum The slot to remove
   */
  public synchronized void removeTimeSlot(int slotNum) {
    
    logger.log("(EQ) Removing time slot " + slotNum, 3);
    if (hasTimeSlot(slotNum)) {
      remove(new Integer(slotNum));
    } else {
      // Do nothing
    }
    logger.log("(EQ) Time slot " + slotNum + " removed.", 4);
  }
  
  /**
   * Inits a slot number
   * @param slotNum The slot to create
   * @return The created slot
   */
  private Vector initTimeSlot(int slotNum) {
    
    logger.log("(EQ) Making time slot " + slotNum, 3);
    Vector v = new Vector(SLOT_CAPACITY);
    putTimeSlot(v, slotNum);
    logger.log("(EQ) Time slot " + slotNum + " created.", 4);
    return v;
  }
  
  /**
   * Puts an entire time slot into a the table
   * @param v The slot to add
   * @param slotNum The slot to add to
   */
  protected void putTimeSlot(Vector v, int slotNum) {
    
    logger.log("(EQ) Adding vector to time slot " + slotNum, 3);
    put(new Integer(slotNum), v);
  }  

  /**
   * Sorts, then shuffles a time slot (vector)
   * @param vec The vector to shuffle
   * @return The shuffled vector
   */
  private Vector shuffleTimeSlot(Vector vec) {

    // Sort it first
    logger.log("(EQ) Sorting time slot.", 4);
    for (int i = 0; i < vec.size()-1; i++) {
      int enumr = i;
      Event e = (Event)vec.elementAt(enumr);
      for (int j = i; j < vec.size(); j++) {
	if (((Event)vec.elementAt(j)).stateCode() < e.stateCode()) {
	  enumr = j;
	  e = (Event)vec.elementAt(enumr);
	}
      }
      if (i != enumr) { // swap
	vec.setElementAt(vec.elementAt(i), enumr);
	vec.setElementAt(e, i);
      }
    }
    for (int i = 0; i < vec.size(); i++)
      logger.log("(EQ) Element " + i + " code: " + ((Event)vec.elementAt(i)).stateCode(), 5);

    // Then shuffle
    logger.log("(EQ) Shuffling time slot.", 4);
    for (int i = 0; i < vec.size(); i++) {
      int enumr = Math.abs(random.nextInt()) % vec.size();
      if (i != enumr) { // swap
	Object e = vec.elementAt(enumr);
	vec.setElementAt(vec.elementAt(i), enumr);
	vec.setElementAt(e, i);
      }
    }
    for (int i = 0; i < vec.size(); i++)
      logger.log("(EQ) Element " + i + " code: " + ((Event)vec.elementAt(i)).stateCode(), 5);

    return vec;
  }

  /**
   * Returns the display
   * @return A panel
   */
  public Container getDisplay() {
    return display;
  }

  /**
   * Sets the display for a particular time slot
   * @return A panel displaying the time slot
   */
  public synchronized void setDisplay(int slotNum) {
    Vector eventVec = getTimeSlot(slotNum);
    Enumeration e = eventVec.elements();

    // Drop old stuff
    display.removeAll();

    // Add each event
    if (e.hasMoreElements() == false)
      display.add(new Label("      Empty      ", Label.CENTER));
    while (e.hasMoreElements())
      display.add(((Event)e.nextElement()).getDisplay());

    // Update
    display.validate();
  }
}
