/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Event.java
 ***************************************************************************************/
package simulator;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.awt.*;
import java.security.*;

/* Local Inclues */
import simulator.Agent;
import utilities.Clock;
import utilities.Log;

/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @version $version$
 * @author Bryan Horling, (bhorling@cs.umass.edu)<P>
 * <B> Description:</B>
 * An object containing the information which describes a particular event.
 */
public abstract class Event extends Object implements Serializable {
  private static long EVENT_NUM = 0;
  protected long id;
  protected int startTime, execTime;
  protected Label display = new Label("");
  protected Vector resourcesInUse;
  protected Vector resourcesInProduction;
  
  /**
   * Constructor, this sets up the event and automatically gives
   * it a unique numerical id.  
   * @param s The start time (offset from current time)
   * @param e The execution time
   */
  public Event(int s, int e) {
    id = EVENT_NUM++;
    resourcesInUse = new Vector();
    resourcesInProduction = new Vector();
    startTime = Clock.getTime() + s;
    execTime = e;
  }
  
  /**
   * Delays the start time
   * @param d The delay time
   */
  public void delay(int d) {
    startTime += d;
  }
  
  /**
   * Extends the execution time
   * @param e The extend time
   */
  public void extend(int e) {
    execTime += e;
  }
  
  /**
   * Gets the event's ID number
   * @return The ID number
   */
  public long getID() {
    return id;
  }
  
  /**
   * Returns the expected start time
   * @return The start time
   */
  public int getStart() {
    return startTime;
  }
  
  /**
   * Returns the expected finish time
   * @return The finish time
   */
  public int getFinish() {
    return startTime + execTime;
  }
  
  /**
   * Abort an event before the specified time
   */
  public void abort(String a)
  {
    // Free Resources !!!
    freeUsedResources();
    freeProducedResources();
  }
    
  /** 
   *  Add Resources Uses by this Event()
   */
  public synchronized void addUsedResource(Resources r) {
    resourcesInUse.addElement(r);
  }

  /** 
   *  Add Resources Produces by this Event()
   */
  public synchronized void addProducedResource(Resources r) {
    resourcesInProduction.addElement(r);
  }

  /** 
   *Freeying resources uses by an Event()
   */
  public synchronized void freeUsedResources() {
    Enumeration e;
    e = resourcesInUse.elements();
    while (e.hasMoreElements())
      {
	Resources res = (Resources)e.nextElement();
	res.releaseResources(this);
      }
    resourcesInUse.removeAllElements();
  }
    
        /** 
	 *Freeying resources uses by an Event()
	 */
    public synchronized void freeUsedResource(Resources res) {
	res.releaseResources(this);
	resourcesInUse.removeElement(res);
    }
    
    /** 
     *Freeying resources uses by an Event()
     */
  public synchronized void freeProducedResources() {
    Enumeration e;
    e = resourcesInProduction.elements();
    while (e.hasMoreElements())
      {
	Resources res = (Resources)e.nextElement();
	res.releaseResources(this);
      }
    resourcesInProduction.removeAllElements();
  }

  
     /** 
      *Freeying resources uses by an Event()
      */
  public synchronized void freeProducedResource(Resources res) {
      res.releaseResources(this);
      resourcesInProduction.removeElement(res);
  }

  /**
   * In case of overloading this will uses the limits NLE's
   */
  public synchronized void overloadResources(String resource, float value) { }

  /** 
   * Updating at each pulse the resources used by
   * this Event()
   */
  public synchronized void updateUsedResources() {
    Enumeration e;
    e = resourcesInUse.elements();
    while (e.hasMoreElements())
      {
	Resources res = (Resources)e.nextElement();
	res.updateResources(this);
      }
  }

   /** 
   * Updating at each pulse the resources produced by
   * this Event()
   */
  public synchronized void updateProducedResources() {
    Enumeration e;
    e = resourcesInProduction.elements();
    while (e.hasMoreElements())
      {
	Resources res = (Resources)e.nextElement();
	res.updateResources(this);
      }
  }

  /**
   * Returns the current status of the event
   * @return A String representing the event status
   */
  public abstract String getStatus();

  /**
   * Updates the status display
   */
  public void updateDisplay() { display.setText("ID:" + id + " ??"); }

  /**
   * Returns the current display of the event
   * @return A component which is the display
   */
  public Label getDisplay() { return display; }

  /**
   * Realizes the particular event
   * @return True if the realization was successfull
   */
  public abstract boolean realize();
  
  /**
   * Returns the state code, which should hopefully be relatively
   * unique (based on the event's contents) and deterministic
   * @return the state code
   */
  public long stateCode() {
    long code = -1;

    try {
      ByteArrayOutputStream devnull = new ByteArrayOutputStream(512);
      MessageDigest md = MessageDigest.getInstance("SHA");
      DigestOutputStream mdo = new DigestOutputStream(devnull, md);
      DataOutputStream data = new DataOutputStream(mdo);

      // Enter the data
      data.writeLong(ObjectStreamClass.lookup(getClass()).getSerialVersionUID());
      data.writeInt(startTime);
      data.writeInt(execTime);

      // Compute the hash value
      byte hasharray[] = md.digest();
      for (int i = 0; i < Math.min(8, hasharray.length); i++)
	code += (long)(hasharray[i] & 255) << (i * 8);
    } catch (IOException ignore) {
      System.err.println("Error: " + ignore);
    } catch (NoSuchAlgorithmException complain) {
      throw new SecurityException(complain.getMessage());
    }

    return code;
  }

  /**
   * Returns the event as a string
   * @return The stringified version of the event
   */
  public String asString() {
    return ("[Event-" + id + "] (Start: " + startTime + " Exec: " + execTime + ")");
  }
}
