/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/***************************************************************************************
 *	Display.java
 ***************************************************************************************/
package simulator;

/* Global Includes */
import java.util.*;
import java.net.*;
import java.awt.*;
import java.awt.image.*;

/* Local Includes */
import utilities.Log;

/**
 * Displays a global view of the current environment, based on messages
 * which are (hopefully) received from each connected agent.
 */
public class Display extends Canvas {
  private static int DEF_TABLESIZE = 10;
  private static Display display = new Display(); 
  private Image background;
  private Log logger;
  private Hashtable locations, displays, images;
  private transient Image offscreen;
  private transient Graphics offgraphics;
  private MediaTracker tracker = new MediaTracker(this);
  private int count = 0;
	
  /**
   * Private constructor
   */
  private Display() {
    locations = new Hashtable(DEF_TABLESIZE);
    displays = new Hashtable(DEF_TABLESIZE);
    images = new Hashtable(DEF_TABLESIZE);
  }
  
  /**
   * Returns the global display object
   * @return The display object
   */
  public static Display getDisplay() {
    return display;
  }
  
  /**
   * Inits the display
   */  
  public void init() {
    logger = Log.getDefault();
  }

  /**
   * Sets the background image
   */
  public void setBackground(String url) {

    background = getImage(url);
    try {
      tracker.waitForAll();
    } catch (InterruptedException e) { 
      logger.log("Error " + e,0);
    }
    repaint();
  }

  /**
   * Fetches an image
   * @param str The URL of the image
   * @return The image, or null if not found
   */
  public Image getImage(String str) {
    Image i = null;

    try {
      URL url = new URL(str);
      i = Toolkit.getDefaultToolkit().getImage(url);
      tracker.addImage(i, count++);
    } catch (MalformedURLException e) {
      logger.log("Error: " + e,0);
    }

    return i;
  }

  /**
   * Updates an agent's location
   * @param n The agent's name
   * @param l The new location
   */
  public void updateLocation(String n, String l) {

    if (l != null && !l.equalsIgnoreCase("null")) {
        Point p = new Point();
        StringTokenizer t = new StringTokenizer(l, ",");
        p.x = Integer.parseInt(t.nextToken());
        p.y = Integer.parseInt(t.nextToken());
        locations.put(n, p);
    } else {
        locations.remove(n);
    }

    repaint();
  }

  /**
   * Updates an agent's display
   * @param n The agent's name
   * @param d The URL of the new display image
   */
  public void updateDisplay(String n, String d) {
    Image i;

    displays.put(n, d);
    if (!images.contains(d)) {
        i = getImage(d);
        if (i != null)
            images.put(d, i);
    }

    repaint();
  }

  /**
   * Removes an agent from the display
   * @param n The agent's name
   */
  public void removeAgent(String n) {

    displays.remove(n);
    locations.remove(n);

    repaint();
  }

  /**
   * Updates the image
   * @param g The graphics object to use
   */
  public synchronized void update(Graphics g) {

    if (offscreen == null) {
      Dimension d = getPreferredSize();
      offscreen = createImage(d.width, d.height);
      offgraphics = offscreen.getGraphics();
      offgraphics.setFont(getFont());
    }

    // Wait for graphics to load
    try {
      tracker.waitForAll();
    } catch (InterruptedException e) {
      logger.log("Error: " + e,0);
    }

    // Draw the background
    offgraphics.drawImage(background, 0, 0, null);

    // Now draw each agent thing
    Enumeration e = new utilities.SortEnumeration(displays.keys());
    String key;
    while (e.hasMoreElements()) {
      key = (String)e.nextElement();
      Point point = (Point)locations.get(key);
      if (point == null) continue;

      Image display = (Image)images.get(displays.get(key));
      if (display != null) {
	offgraphics.drawImage(display, point.x, point.y, null);
      }
    }

    paint(g);
  }

  /**
   * Paints the display
   * @param g The graphics object to use
   */
  public synchronized void paint(Graphics g) {
    if (offscreen == null) update(g);
    g.drawImage(offscreen, 0, 0, null);
  }

  /**
   * Returns the size of the image, basically the size of the background
   */
  public Dimension getPreferredSize() {
    Dimension d = new Dimension(100,100);

    if (background != null) {
        d.height = background.getHeight(this);
        d.width = background.getWidth(this);
    }

    return d;
  }
  public Dimension getMaximumSize() { return getPreferredSize(); }
  public Dimension getMinimumSize() { return getPreferredSize(); }
}




