/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

package simulator;

/* Global Includes */
import java.io.*;
import java.util.*;
import java.security.*;

/* Local Inclues */
import simulator.Agent;
import simulator.Event;
import taems.*;
import utilities.*;

/**
 * <B>Module</B>: simulator<p>
 * <B>Copyright</B>: UMASS - MASL 1998<P>
 * @version $Revision: 1.2 $
 * @author Regis Vincent (vincent@cs.umass.edu)<P>
 * <B> Description:</B>
 * DisablesEvent extends the Event class.
 * This class describes how to deal with an Non Local 
 * Effect called Disables.
 */
public class DisablesEvent extends Event implements Serializable {
  private static  Log logger;
  protected DisablesInterrelationship disable;
  protected	  Agent	agent ;
/**
 * Constructor:
 * @param s is the time that this Disables method takes to propagate.
 * @param relation type is the Enable Interrelationship instance
 * @see Event
 */
  public DisablesEvent(int s, DisablesInterrelationship relation, Agent a) {
    super(0, s);
    agent = a;
    this.disable = relation;
    logger = Log.getDefault();
    logger.log("(Disables " + disable.getLabel() + ") from " + disable.getFrom().getLabel() + " to " + disable.getTo().getLabel() + " with a " + s + " clicks ");
  }

  /**
   * Returns a string of the event's status, as determined
   * by the simulation clock.  This assumes the event queue
   * is functioning correctly, it does not actually check
   * how the event is being executed, so it should only be
   * called on events returned from a queue slot.
   * @return A String representing the event status
   */
  public String getStatus() {
    int time = Clock.getTime();
    String status = "";
    
    if (time == startTime)
      status = "Disables added.";
    else if (time == (startTime + execTime))
      status = "Disables received by the receiver.";
    else if (time < startTime)
      status = "Disables waiting to be sent.";
    else if (time > (startTime + execTime))
      status = "Disables done.";
    else 
      status = "Disables en route to receiver.";
    
    status = "ID:" + id + " Status: " + status;
    return (status);
  }

  /**
   * Realizes the particular event
   * This function is called at each click during the time
   * of the propagation.
   * @return True if the realization was successfull
   */
  public boolean realize() {
    int time = Clock.getTime();
    if (time == (startTime + execTime)) {
      disable.update(time);
      if (disable.getFrom().getAgent().matches(disable.getAgent()))
        sendEndRealize("ActivateInterrelationship");
    }
    updateDisplay();
    return true;
  }
  
    /**
     * End of the Enables, send a message back to the agent to
     * warn it about the activation of an interrelationship.
     */
    private void sendEndRealize(String msgType) {
	String content = msgType + " " + disable.getLabel() + " " + execTime;
	KQMLMessage reply = new KQMLMessage("tell", "(" + content +")", 
					    agent.getName() );
	reply.setSourceAddr("simulator");
	agent.sendMsg(reply);
    }

  /**
   * Updates the status display. 
   */
  public void updateDisplay() {
    display.setText(monitoreEvent());
  }

  /** 
   * returns a String containing the informations about the ExecuteEvent
   * It looks like : <BR>
   * Get-Address C: 2.4 Q: 34.43 T: -4
   */
  public String monitoreEvent() {
    return(new String(disable.getLabel() +  " T:" + 
		      (Clock.getTime() - (startTime + execTime))));
  }

  /**
   * Returns the state code, which should hopefully be relatively
   * unique (based on the event's contents) and deterministic
   * @return the state code
   */
  public long stateCode() {
    long code = super.stateCode();

    try {
      ByteArrayOutputStream devnull = new ByteArrayOutputStream(512);
      MessageDigest md = MessageDigest.getInstance("SHA");
      DigestOutputStream mdo = new DigestOutputStream(devnull, md);
      DataOutputStream data = new DataOutputStream(mdo);

      // Enter the data
      data.writeLong(code);
      data.writeUTF(disable.toString());

      // Compute the hash value
      byte hasharray[] = md.digest();
      for (int i = 0; i < Math.min(8, hasharray.length); i++)
	code += (long)(hasharray[i] & 255) << (i * 8);
    } catch (IOException ignore) {
      System.err.println("Error: " + ignore);
    } catch (NoSuchAlgorithmException complain) {
      throw new SecurityException(complain.getMessage());
    }

    return code;
  }
}

