/* Copyright (C) 2005, University of Massachusetts, Multi-Agent Systems Lab
 * See LICENSE for license information
 */

/*****************************************************************
******************************************************************
*
*               Agent.java @version 1.0
*
*     Author : Regis Vincent [vincent@cs.umass.edu]
*     
*        
*
*     Creation date: 22 Aug 97 10:41
*****************************************************************/

package simulator;

/* Global imports */
import java.io.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import javax.swing.*;

/* Local imports */
import utilities.*;
import taems.*;
import simulator.*;
import simulator.sensor.*;
import simulator.locale.LocationMapping;
import simulator.locale.Locale;

/**
 * Agent class, this is the representation of an agent. Each agent connected
 * to the server creates its own instance. All agents are store in a static
 * hashtable.
 */
public class Agent extends GraphNode {
  private static int NUM_AGENTS = 0;
  private String name;
  private int id;
  private Listener listener;
  private boolean status;
  private Panel display;
  private Vector executions;
  private ImageComponent picture;
  private ImageComponent picture2;
  private Label statusLabel;
  //private Taems taemsHandler;
  private Hashtable taems = new Hashtable();
  private Sensing sensing;
  public Socket socket;
  Locale agentlocale = null;
  Label localelabel;
    
  /**
   * @param s The sockets structure 
   * @param n The agent's name, if nil defaults to Agent_X, where X is its id
   */
  public Agent(Socket s, String n) {
    String suffix = "";
    int num = 1;

    socket = s;
    id = ++NUM_AGENTS;

    // Find a name
    if (n == null) n = "Agent";
    name = n;
    while (findAgent(name + suffix) != null)
	suffix = "_" + num++;
    name += suffix;

    status = false;

    executions = new Vector();
    sensing = new Sensing();

    setupDisplay();
  }

  /**
   * Sets up the agent's display panel
   */
  private void setupDisplay() {
      display = new Panel();
      display.setLayout(new FlowLayout(FlowLayout.LEFT));

      try {
	  display.add(picture = new ImageComponent(new URL("http://dis.cs.umass.edu/research/mass/graphics/simulate/agent_icon.GIF")));
      } catch (MalformedURLException e) { System.err.println("(Agent) Error finding image: " + e); }
      display.list();
      display.add(new Label("Name: " + name));
      display.add(new Label("ID: " + id));
      localelabel = new Label();
      if(agentlocale!=null && 
	  agentlocale.getName()!=null)
	  localelabel.setText("Locale: " + agentlocale.getName());
      else localelabel.setText("Locale:                     ");
      display.add(localelabel);
      display.add(statusLabel = new Label("Running: " + status));
      
      
      Button killButton = new Button("Kill");
      killButton.addActionListener(new ActionListener() {
	      public void actionPerformed(ActionEvent e) {
		  disconnect();
	      }
	  });
      display.add(killButton);
      
      Button pulseButton = new Button("Pulse");
      pulseButton.addActionListener(new ActionListener() {
	      public void actionPerformed(ActionEvent e) {
		  sendPulse();
	      }
	  });
      display.add(pulseButton);
  }

  /**
   * Updates the agent's picture display
   * @param str URL of the image to use
   */
  public void setPicture(String str) {

    try {
      URL url = new URL(str);
      picture.setImage(url);
    } catch (MalformedURLException e) {
      System.err.println("Error: " + e);
    }

    Display.getDisplay().updateDisplay(getName(), str);
  }

  /**
   * Updates the agent's location
   * @param str The physical location of the agent
   */
  public void setLocation(String str) {
      setLocale(Simulator.getSimulator().getLocationMapping().mapLocationToLocale(str));

      Log.getDefault().log("Setting agent "+name+" to location "+getLocale().getName());
      // XXX should be reenabled later XXX
      Display.getDisplay().updateLocation(getName(), str);
  }
    
  /**
   * Returns the agent's display
   * @return A referene to the display panel
   */
  public Panel getDisplay() {
    return display;
  }

  /**
   * Returns the agent's ID number
   * @return The agent's (ID) number
   */
  public synchronized int getID() {
    return id;
  }

  /**
   * Returns the agent's name
   * @return The agent's name
   */
  public synchronized String getName() {
    return name;
  }

  /**
   * Disconnects the agent
   */
  public void disconnect() {
    killAllExecutions();
    removeAllTaems();
    KQMLMessage dis = new KQMLMessage("ask", "(disconnect)", getName());
    dis.setSourceAddr("simulator");
    dis.addField("type", "control");
    dis.addField("reply-with", "(disconnect)");
    sendMsg(dis);
    if (agentlocale != null)
    {
        ContainerNodePanel p = (ContainerNodePanel)
            Simulator.getSimulator().getGraph().getLayers().getTopPanel(agentlocale);
        if (p != null)
        {
            agentlocale.removeContained(p, this);
            p.revalidate();
        }
    }
    Display.getDisplay().removeAgent(getName());

  }

  /**
   * Resets the agent
   */
  public void reset() {
    killAllExecutions();

    KQMLMessage dis = new KQMLMessage("ask", "(reset)", getName());
    dis.setSourceAddr("simulator");
    dis.addField("type", "control");
    dis.addField("reply-with", "(reset)");
    sendMsg(dis);

    sensing = new Sensing();
    
    KQMLMessage msg = new KQMLMessage("(Name " + getName()+ " )", getName());
    msg.setSourceAddr("simulator");
    msg.addField("type", "control");
    sendMsg(msg);
    msg = new KQMLMessage("(RandomSeed " + Simulator.getSimulator().random.getSeed() + " )", getName());
    msg.setSourceAddr("simulator");
    msg.addField("type", "control");
    sendMsg(msg);
    msg = new KQMLMessage("(Time " + Clock.getTime() + " )", getName());
    msg.setSourceAddr("simulator");
    msg.addField("type", "control");
    sendMsg(msg); 
   }

  /**
   * Add sensing capabilities to the agent. Only for this agent.
   * @param String s - the sensor line definition
   */
  public void addSensor(String s) {
    try {
      sensing.parseLine(s);
    } catch (simulator.sensor.SensorParseException e) { 
      System.err.println(e.getMessage());
    }
  }

  /**
   * Gets a handle to the agent's sensing stuff
   */
  public Sensing getSensing() {
    return sensing;
  }

  public Enumeration getSensors() {
    return sensing.getSensors();
  }
 
  /**
   * runSensing answer to a question from the agent
   * The order is sensor defines at the agent level first then
   * if not defines sensors defines at the simulator level.
   * @param String call - call to run 
   * @param String ID - ID of the message
   * Side effect Sends back the answer to the agent
   */
    public void runSensing(String call, String ID) {
        String answer="";
        boolean globaloveride = false;
        boolean error=false;
        Simulator sim = Simulator.getSimulator();
        Log log = Log.getDefault();

        if (sensing != null) {
            try {
                if (sim.getGlobalSensing().getObjectSensed(sim.getLocale("Global"), call)
                    == sensing.getObjectSensed(getLocale(), call)) {
                    answer = sim.getGlobalSensing().runSensing(sim.getLocale("Global"), call);
                    globaloveride = true;
                }
            } catch (simulator.sensor.SensingErrorException x) {
                error = true;
                log.log("Sensing error "+x, 1);
                answer = x.getMessage();
            }
	    
            //		}
            if(! globaloveride) {
                try {
                    answer = sensing.runSensing(getLocale(), call);
                }
                catch (SensingErrorException e) {
                    error = true;
                    log.log("Sensing error "+e, 1);
                    answer = e.getMessage();
                }
            }
        }
	
        String mesg;
        if (error) 
            mesg = "( SensorData Error ";
        else
            mesg = "( SensorData ";
    
        KQMLMessage reply = new KQMLMessage("reply", mesg + answer + " )",name);
        reply.addField("type", "control");
        reply.addField("in-reply-to", ID);
        sendMsg(reply);
    }  

  /**
   * Finds an agent using its name 
   * @param String agent's name
   * @return Agent's intance or null 
   */
  public  Agent findAgent(String name) {
    return(Simulator.getSimulator().findAgent(name));
  }

  /**
   * Broadcast to all agents this messages except for the sender
   * @param: KQMLMessage msg
   * 
   */
  public void broadcastMsg(KQMLMessage msg) {
    Simulator.getSimulator().broadcastMsg(this,msg);
  }
  

  /**
   * Return a list of all Agent connected (string form)
   * @return Agents List
   */
  public String getAllAgents() {
    return(Simulator.getSimulator().getAllAgents());
  }

  /**
   * Adds the agent to the simulation
   */
  public  void add() {
    Simulator.getSimulator().addAgent(this);
  }

  /**
   * Removes the agent from the simulation
   */
  public void remove() {
    status = false;
    Simulator.getSimulator().removeAgent(this);
  }

  /**
   * Kills the agent, make sure you disconnect() first
   */
   public void destroy() {
     try {
       socket.close();
     }
     catch (IOException io) {
       System.err.println("(Agent) Error closing destroying agent: " + io);
     }
     remove();
   }

  /**
   * Extends all current Executions
   * @param i the time to extend them by
   */
  public void extendAllExecutions(int i) {
    Enumeration e;
    e = executions.elements();
    while (e.hasMoreElements())
      {
	ExecuteEvent exe = (ExecuteEvent)e.nextElement();
        EventQueue.getQueue().extendEvent(exe, i);
        exe.renormalizeResources();
      }	
  }

  /**
   * Extends one Execution
   * @param method ID
   * @param i the time to extend it by
   */
  public boolean extend(String ID, int i) {
    Enumeration e;
    e = executions.elements();
    while (e.hasMoreElements()) {
      ExecuteEvent exe = (ExecuteEvent)e.nextElement();
      if (ID.equals(exe.getExeID())) {
        EventQueue.getQueue().extendEvent(exe, i);
        exe.renormalizeResources();
	return true;
      }
    }
    System.err.println("(Agent) Error extend couldn't find the method " + ID + " from agent " + name);
    return false;
  }

  /**
   * Kill all current Executions
   */
  public void killAllExecutions() {
    Enumeration e;
    e = executions.elements();
    while (e.hasMoreElements())
      {
	ExecuteEvent exe = (ExecuteEvent)e.nextElement();
	exe.abort(null);
      }	
  }

  /**
   * abort a designated method
   * @param method ID
   */
  public boolean abort(String ID) {
    Enumeration e;
    e = executions.elements();
    while (e.hasMoreElements()) {
      ExecuteEvent exe = (ExecuteEvent)e.nextElement();
      if (ID.equals(exe.getExeID())) {
	exe.abort(null);
	return true;
      }
    }
    System.err.println("(Agent) Error abort couldn't find the method " + ID + " from agent " + name);
    return false;
  }


  /**
   * getMethodStatus of a designated method
   * @param method ID
   * @return a String containing the status
   */

  public String getMethodStatus(String ID) {
    Enumeration e;
    e = executions.elements();
    while (e.hasMoreElements()) {
      ExecuteEvent exe = (ExecuteEvent)e.nextElement();
      if (ID.equals(exe.getExeID())) 
	return(exe.monitoreEvent());
    }
    return("Error Method not found");
  }

 /**
   * getResourceStatus of a designated method
   * @param resource name 
   * @return a String containing the status
   * Status format is : R1 30 0 100 
   *                  : Name value minimum maximum
   */

  public String getResourceStatus(String r) {
    return(Simulator.getSimulator().getResourceStatus(agentlocale, r));
  }

    public void setLocale(Locale l) {
	    //System.err.println("l is " + l + ", agentlocale is " + agentlocale);
	    if (agentlocale != null)
	    {
	        ContainerNodePanel p = (ContainerNodePanel)
                Simulator.getSimulator().getGraph().getLayers().getTopPanel(agentlocale);
            if (p != null)
            {
                agentlocale.removeContained(p, this);
                p.revalidate();
            }
	    }
	    if (l != null)
	    {
	        ContainerNodePanel p = (ContainerNodePanel)
                Simulator.getSimulator().getGraph().getLayers().getTopPanel(l);
            if (p != null)
            {
                l.addContained(p, this);
                p.revalidate();
            }
            if (localelabel != null)
                localelabel.setText("Locale: " + l.getName());	        
	    }
	    agentlocale = l;

    }
    
    public Locale getLocale() {
        if (agentlocale == null)
            return Simulator.getSimulator().getLocale("Global");
        else
            return agentlocale;
    }    

  /** 
   * Add a new Event in the Executions list
   */
  public void addExecution(ExecuteEvent exe) {
    executions.addElement(exe);
  }

  /**
   * Test if this method is not already in execution
   * To avoid multiples request on the same agent.
   */
  public boolean isAlreadyInExecution(String ID) {
    Enumeration e;
    e = executions.elements();
    while (e.hasMoreElements()) {
      ExecuteEvent exe = (ExecuteEvent)e.nextElement();
      if (ID.equals(exe.getExeID())) 
	return(true);
    }
    return(false);
  }

  /**
   * Remove a finished Event in the Executions list
   */

  public void removeExecution(ExecuteEvent exe) {
    executions.removeElement(exe);
  }

  
  /**
   * Set the private field listener. The Listener is the thread waiting for
   * message from a particular agent.
   */
  public void setListener(Listener l)
  {
    listener = l;
  }

  /** 
   * Pulsing changes the status of the agent from waiting to running and
   * notify all threads. This function is used by the clock.
   */
  public synchronized void pulsing()
  {
    if (status == false) {
      status = true;
      notifyAll();
    }
    statusLabel.setText("Status: " + status);
    sensing.pulsing();
  }
  
  /**
   * waiting funtion is the opposite of pulsing, it changes the status from
   * running to waiting. This function is used by the Listener, when it 
   * receives the ack pulse message from the agent.
   */
  public synchronized void waiting()
  {
    if (status){
      status = false;
      notifyAll();
    }
    statusLabel.setText("Status: " + status);
  }	

  /**
   * isRunning tests if the agent is running or waiting depending of the status
   * parameter.
   */
  public boolean isRunning()
  {
    return(status);
  }

  /**
   * Returns a stringified version of the agent
   * @return The description of the agent
   */
  public String toString()
  {
    String str = "";

    str += "[Agent] " + name + System.getProperty("line.separator");
    str += "    ID: " + id + System.getProperty("line.separator");
    str += "    Status: ";
    if(isRunning()) 
      str += "running" + System.getProperty("line.separator");
      else str += "waiting" + System.getProperty("line.separator");
    //str += "    Name: " +  name + System.getProperty("line.separator");
    str += "    Socket: " + socket.toString() + System.getProperty("line.separator");
    str += "    Listener: " + listener.toString() + System.getProperty("line.separator");
    return str;
  }

  /** 
   * This is the generic function for sendind a message to an agent.
   * @param s is the message.
   */
  public void sendMsg(KQMLMessage s) {
    listener.sendMsg(s);
  }

  /**
   * Returns the Taems object associated with a key, or null if
   * none found
   */ 
  public Taems getTaems(Object key) {
      return (Taems)taems.get(key);
  }

  /** 
   * Adds a new Taems task structure for this agent
   */
  public void addTaems(Object key, Taems t) {
      if (key == null) {
          Log.getDefault().log("Warning, tried to add null-key Taems structure for " + getName(), 1);
          return;
      }
      if (t == null) {
          Log.getDefault().log("Warning, tried to add null Taems structure for " + getName(), 1);
          return;
      }

      // Remove old if it exists
      removeTaems(key);

      // Add new
      Log.getDefault().log("Adding Taems structure [" + key + "] for agent " + getName(), 2);
      taems.put(key, t.clone());
      Simulator.getSimulator().addTaems(t);
      Log.getDefault().log("Done adding Taems", 3); 
  }

  /** 
   * Removes a new Taems task structure from this agent
   */
  public Taems removeTaems(Object key) {
      if (key == null) {
          Log.getDefault().log("Warning, tried to remove null-key Taems structure for " + getName(), 1);
          return null;
      }

      // Remove it
      Taems t = getTaems(key);
      if (t != null) {
          Log.getDefault().log("Removing Taems structure [" + key + "] from agent " + getName(), 2);
          taems.remove(key);
          Log.getDefault().log("Done removing Taems", 3); 
          return Simulator.getSimulator().global_view.removeTaems(t);
      }
      
      return null;
  }

    /**
     * Convenience function that removes all the agent's Taems structures.
     */
    public void removeAllTaems() {
        Enumeration e = taems.keys();
        while(e.hasMoreElements()) {
            removeTaems(e.nextElement());
        }
    }

  /**
   * Updates an existing Taems task structure for the agent
   */
  public void updateTaems(Object key, Taems t) {
      if (key == null) {
          Log.getDefault().log("Warning, tried to update null-key Taems structure for " + getName(), 1);
          return;
      }
      if (t == null) {
          Log.getDefault().log("Warning, tried to update null Taems structure for " + getName(), 1);
          return;
      }

      Log.getDefault().log("Updating Taems structure [" + key + "] for agent " + getName(), 2);
      if (getTaems(key) == null) {
          // Don't have one to update
          addTaems(key, t);
          
      } else {
          // Remove, import and remerge
          Taems old = removeTaems(key);
          old.importTaems(t);
          taems.put(key, old.clone());
          Simulator.getSimulator().addTaems(old);
      }
      Log.getDefault().log("Done updating Taems", 3); 
  }

  /**
   * setTaemsHandler store the top level node of the TTAEMS 
   * structure, to be able to remove the task when the
   * agents get disconnected.
   */
    /*
  public void setTaemsHandler(Object key, Taems t) {
    //removeTaems();

    Log.getDefault().log("Got a new objective structure for " + getName() + ", merging", 2);

    if (taemsHandler == null) {
        taemsHandler = (Taems)root.clone();
        Simulator.getSimulator().addTaems(root);

    } else {
        Taems old = Simulator.getSimulator().global_view.removeTaems(taemsHandler);
        old.importTaems(root);
        taemsHandler = (Taems)old.clone();
        Simulator.getSimulator().addTaems(old);        
    }

    //Simulator.getSimulator().addTaems(objective);
  }
    */

  /** 
   * getTaemsHandler returns the Root Node previously stored
   */
    /*
  public Taems getTaemsHandler() {
    return(this.taemsHandler);
    }*/

  /**
   * Remove the Task Structure of this agent from the
   * simulation stuff
   */
    /*
  protected void removeTaems() {
    if (getTaemsHandler() != null) {
      Log.getDefault().log("Unmerging old structure", 2);
      Simulator.global_view.removeTaems(getTaemsHandler());
      Log.getDefault().log("Done unmerging old structure", 3);
    }
    }*/

  /**
   * This function is used by the clock to send the pulse message to an agent.
   */
  public void sendPulse() {
    pulsing();
    KQMLMessage dis = new KQMLMessage("ask", "(pulse)", getName());
    dis.setSourceAddr("simulator");
    dis.addField("type", "control");
    dis.addField("reply-with", "(pulse)");
    sendMsg(dis);
  }
   
  ImageIcon picture3;
  JLabel blah;
  public void initGraphics(JPanel p)
  {
      JPanel panel;
      if (p == null)
          panel = new JPanel();
      else {
          panel = p;
          return;
      }
      panel.setLayout(new GridLayout(1, 1));

      try {
          //panel.add(picture2 = new ImageComponent(new URL("http://dis.cs.umass.edu/research/mass/graphics/simulate/agent_icon.GIF")));
          picture3 = new ImageIcon(new URL("http://dis.cs.umass.edu/research/mass/graphics/simulate/agent_icon.GIF"));
          panel.add(blah = new JLabel(picture3));
      } catch (MalformedURLException e) { System.err.println("(Agent) Error finding image: " + e); }
      blah.addNotify();
      //display.add(new Label("Name: " + name));
      //display.add(new Label("ID: " + id));
      //localelabel = new Label();
      //if(agentlocale!=null && 
      //agentlocale.getName()!=null)
      //localelabel.setText("Locale: " + agentlocale.getName());
      //else localelabel.setText("Locale:                     ");
      //display.add(localelabel);
      //display.add(statusLabel = new Label("Running: " + status));

      /*
         Button killButton = new Button("Kill");
         killButton.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
         disconnect();
         }
         });
         display.add(killButton);

         Button pulseButton = new Button("Pulse");
         pulseButton.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
         sendPulse();
         }
         });
         display.add(pulseButton);
       */
  }

  }
