/**************************************************************************
 * ExampleProblemSolver.java
 *
 * Update the manifest lines below to match your component's name and
 * package.  These are used to generate the manifest listing when 
 * generating jar files of your classes.
 *
 * manifest: Name: agent/example/ExampleProblemSolver.class
 * manifest: Java-Bean: True
 **************************************************************************/

package agent.example;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.BufferedReader;
import java.io.IOException;

import agent.base.AgentComponent;
import agent.base.AgentEvent;
import agent.base.ListenerVector;
import agent.simplest.*;
import agent.simplest.ActionEvent;
import agent.mass.TaemsAction;
import agent.mass.SimpleTaemsReader;

import taems.*;

import utilities.Connection;
import utilities.Message;
import utilities.KQMLMessage;

/**
 * This is a shell JAF component which demonstrates how the various
 * functions and stages can be used.  This class extends ProblemSolver,
 * but you can also just extend agent.base.AgentComponent to make
 * a completely new component.
 *
 * @see agent.base.AgentComponent
 */
public class ExampleProblemSolver extends AgentComponent implements PropertyEventListener, MessageEventListener, ActionEventListener {

  // The components this component will make use of
  protected Log log;
  protected State state;
  protected Execute execute;
  protected Communicate communicate;
  protected SimpleTaemsReader taemsreader;

  // This component produces events, so it must have a
  // place to store registered event listeners
  protected ListenerVector listeners = new ListenerVector();

  // A place to store the component's log facility id
  protected int FAC_EPS;

  // Constant declarations for parameter names
  public static final String PERFORMACTIONS = "PerformActions";


  /**
   * Default constructor.  Component constructors should only modify
   * things that are "constant" as far as the thread of execution is
   * concerned and are independent of other installed components. 
   * This will typically include setting up GUIs and calculating
   * environmental constants that aren't user- configurable.  You'll
   * also want to note your dependencies and configuration parameters
   * here.
   */
  public ExampleProblemSolver() {
    super();

    // Declare your dependencies.  You should declare a dependency
    // to any component that your component cannot run without.
    addDependency("Log");
    addDependency("State");
    addDependency("Communicate");
    addDependency("SimpleTaemsReader");

    // The State component can be used here to declare any properties
    // your component will make use of.  This declaration tells State
    // that it should look for it in the configuration file or
    // environment, and read it in when found.
    State.addParameterInfo(PERFORMACTIONS, "Boolean", "Automatically complete actions as they are started.", Boolean.FALSE);
  }

  /**
   * The initialization procedure.  Called when the system state is
   * stable (constructors have completed).  Typically, this should be
   * used to setup component links and initialize any transient
   * structures the component makes use of (open files, sockets, etc.)
   * Relatively permanent inter-component relationships can also be
   * set up here (i.e., it's ok to have a static addListener statement
   * hooking the State component to Communications, rather than hand
   * specifying it in the builder tool.) This will be called just
   * once, upon system start up.
   */
  public void init() {
    super.init();

    // Get handles to any other components you will need to make
    // use of.  This usually corresponds to those components that
    // you listed as dependencies in the constructor.
    log = (Log)agent.simplest.State.findComponent("Log");
    state = (State)agent.simplest.State.findComponent("State");
    execute = (Execute)agent.simplest.State.findComponent("Execute");
    communicate = (Communicate)agent.simplest.State.findComponent("Communicate");
    taemsreader = (SimpleTaemsReader)agent.simplest.State.findComponent("SimpleTaemsReader");

    // This function uses Log to create a "facility code" for your
    // component.  This allows you to flag your log messages with
    // a unique identifier.  This is used to allow you to have 
    // different logging levels for different components, which
    // can make log files much easier to read by reducing 
    // unnecessary logging.
    FAC_EPS = log.getFacilityID(this);

    // I'll make myself a PropertyEventListener here.  This means
    // that I will receive the property events generated by the
    // State component (see the property* methods below).  You'll
    // make a call similar to this for each event stream you want
    // to listen to.  (or none at all if you don't care about
    // events)
    state.addPropertyEventListener(this);

    // I'll become a message and action event listener in a similar way.
    communicate.addMessageEventListener(this);
    execute.addActionEventListener(this);

    // We'll just try out the new facility code here.  Note that
    // we are using the handle to Log provided by the findComponent
    // function above.
    log.log("I'm done with the init function", Log.LOG_INFO, FAC_EPS);
  }

  /**
   * Called when the component is ready to execute.  Threads, if any,
   * should be started here, along with any other of the "real"
   * services a given component is responsible for.
   */
  public void begin() {
    super.begin();

    // Use the state component to get access to properties that
    // have been specified in the configuration file, by 
    // environment variables, or by other components.
    String name = (String)state.getProperty(State.NAME);
    log.log("Hi, my name is " + name, Log.LOG_INFO, FAC_EPS);

    // State can also be used to get stream and reader objects
    // to files in the external configuration data file.  Here
    // we get a handle on the agent's Taems file and do some
    // minimal processing on it.
    String filename = (String)state.getProperty("ConTaems");
    BufferedReader r = state.getExtendedEntryReader(filename);
    int num = 0;
    try {
       for (; r.ready(); num++) r.readLine();
    } catch (IOException e) {
       log.log("Error reading from " + filename + ": " + e, Log.LOG_ERR, FAC_EPS);
    }
    log.log(filename + " contains " + num + " lines of data", Log.LOG_INFO, FAC_EPS);
  }

  /**
   * Called periodically by the Control component, either in response
   * to actual passage of time or in response to some other stimuli it
   * perceives as a time indicator (such as a pulse from a time
   * controller).  Use this method as you would the contents of an
   * unbounded while loop - they will be called periodically forever,
   * and thus can be used to effect the normal moment-to-moment
   * behavior of a given component.
   */
  public void pulse() {
    super.pulse();

    // It's useful to know what time it is
    int time = ((Integer)state.getProperty(State.TIME)).intValue();

    // This component will look at the currently running TaemsActions
    // and "perform" them by simulating them from their Taems QCD
    // expectations.  All other actions will be simply finished.
    boolean perform = ((Boolean)state.getProperty(PERFORMACTIONS)).booleanValue();
    if (perform && execute.isExecuting()) {
       Enumeration e = execute.actions();
       while (e.hasMoreElements()) {
         Action a = (Action)e.nextElement();

         if (a instanceof TaemsAction) {
           TaemsAction ta = (TaemsAction)a;
           ScheduleElement scheduled = ta.getScheduleElement();
           int finish = (int)scheduled.getFinishTime().calculateMax(); 

           if (time >= finish) {
             log.log("Completing action " + a.getName(), Log.LOG_INFO, FAC_EPS);
             Outcome outcome = ta.getMethod().getGlobalOutcome();
             a.setQuality(outcome.getQuality().calculateMax());
             a.setCost(outcome.getCost().calculateMax());
             a.complete();

           } else {
             log.log("Performing action " + a.getName(), Log.LOG_INFO, FAC_EPS);
             // Actually, I'm just loafing about, but who will know?
           }

         } else {
           log.log("Completing action " + a.getName(), Log.LOG_INFO, FAC_EPS);
           a.setQuality(1);
           a.setCost(1);
           a.complete();
         }
       }
    }

    // If we are done executing things, recreate the task structures
    // and start all over again to keep busy.
    if (execute.isDone()) {
        taemsreader.removeAllStructures();
        taemsreader.makeAllStructures();
    }

    // This demonstrates how to fire events from your component,
    // although you probably would do it in response to an interesting
    // activity taking place, rather than on every pulse as is shown here.
    // Note that startFireEvent, an AgentComponent method, is used to
    // actually initiate the event.  See the fireEvent method for
    // more details.
    startFireEvent(new ExampleEvent(this, "Foobar " + time, ExampleEvent.STATE1));
  }

  /**
   * Called when the component is ended, which basically only happens
   * when the agent is terminated.  This should do any cleanup or
   * post-execution processing you might need.  Note that this is
   * separate from Java's finalize() method, and was added because
   * finalize() is not always called upon program termination - you
   * can still have regular finalize() methods.
   */
  public void end() {
    super.end();
  }

  /**
   * This component produces an ExampleEvent stream.  In other
   * words, it will "fire" ExampleEvent objects to any object which
   * has registered itself as an ExampleEventListener.  This function,
   * and its analogue removeExampleEventListener provide the
   * facilities for registering and unregistering from this stream.
   * @param l The listener which is registering for the stream
   * @see #removeExampleEventListener
   */
  public void addExampleEventListener(ExampleEventListener l) {
    listeners.add(l);
  }

  /**
   * Unregisters a listener.
   * @param l The listener which is registering for the stream
   * @see #addExampleEventListener
   */
  public void removeExampleEventListener(ExampleEventListener l) {
    listeners.remove(l);
  }

  /**
   * This is the function which sends an event to each of the
   * registered listeners.  It enumerates through the listeners
   * and calls the appropriate listener interface method, depending
   * on the ID of the event to be sent.
   * <p>
   * Note that it is possible to create tangled event firing
   * stacks when components fire events in reaction to events
   * they have received.  This can lead to situations where one
   * component might be receiving events that are responses to
   * other events they have yet to receive.  To avoid this problem,
   * AgentComponent provides a helper function, startFireEvent,
   * which should be used to initiate event firing, rather than
   * calling this function directly.
   * @param e The event to send
   * @see agent.base.AgentComponent#startFireEvent
   */
  protected void fireEvent(AgentEvent event) {
    ExampleEvent e = (ExampleEvent)event;
    ExampleEventListener l;
    Enumeration enum = listeners.elements();

    log.log("Firing an example event", Log.LOG_DEBUG2, FAC_EPS);

    while (enum.hasMoreElements()) {
      l = (ExampleEventListener)enum.nextElement();

      if (!e.isActive()) break;

      switch (e.getID()) {
      case ExampleEvent.STATE1:
        l.exampleState1(e);
        break;
      case ExampleEvent.STATE2:
        l.exampleState2(e);
        break;
      default:
        break;
      }
    }
    
    // Log the event itself
    if (log != null)
        log.log(e, Log.LOG_INFO, FAC_EPS);
  }

  /**
   * Watch for interesting property changes.  If you are trying
   * to monitor changes to state properties, or track when a 
   * particular value is set, you can place the code here.
   * @param e The event to look at 
   */
  public void propertyChanged(PropertyEvent e) { 
    String key = e.getKey().toString();

    if (key.equals(PERFORMACTIONS)) {
      // Someone set my property!
      boolean perform = ((Boolean)e.getProperty()).booleanValue();
      log.log("I will " + (perform ? "" : "not ") + "perform actions as they start.", Log.LOG_INFO, FAC_EPS);
     }
  }
  public void propertyAdded(PropertyEvent e) { propertyChanged(e); }
  public void propertyRemoved(PropertyEvent e) { }

  /**
   * Monitor any messages that might get sent out or received
   * @param me The event to look at 
   */
  public void messageReceived(MessageEvent me) {
    String perf;
    Message m = (Message)me.getMessage();

    // If it's a KQML message, get its performative
    if (m instanceof KQMLMessage && me.getConnection().getType() != Connection.KQML) {
      KQMLMessage km = (KQMLMessage)m;
      perf = km.getPerformative();

    // otherwise assume it's a "tell" message
    } else {
      perf = "tell";
    }

    if (perf.equalsIgnoreCase("tell")) {
      // You can retrieve the entire contents of a message
      // with the getContent() method.  However, we have
      // found it to be useful to organize messages with
      // a content "word" and "data", where the "word" is
      // the first whitespace delimited text and the "data"
      // is everything after it.  This way an agent can 
      // easily encode and parse a message of the form
      // "MessageType This is the message data".

      // This gets the first word in the content
      String word = m.contentWord();

      // This gets the remainder of the content
      String data = m.contentData();

      // We'll ignore the pulse messages and log everything else
      if (!word.equals("pulse")) {
         log.log("I got a message: " + data, Log.LOG_INFO, FAC_EPS);
      }
    } 
  }
  public void messageSent(MessageEvent me) {}

  /**
   * These are called when action events take place.
   * You can use these methods to track the results of activities,
   * or to trigger reactions when particular actions are started
   * or completed.
   * @param e The action event which took place
   */
  public void actionStarted(ActionEvent e) {
    boolean perform = ((Boolean)state.getProperty(PERFORMACTIONS)).booleanValue();

    if (perform) {
       Action a = e.getAction();
       log.log("Starting action " + a.getName(), Log.LOG_INFO, FAC_EPS);

       // Set it local so Execute knows we are finishing it
       a.setData("Local", Boolean.TRUE);
    }
  }
  public void actionAborted(ActionEvent e) { }

  public void actionCompleted(ActionEvent e) { 

      // This bit is domain and task structure specific. We'll assume
      // that somewhere,  somehow, this agent knows that when the
      // Task Put-Away-Dishes gets quality, it should report that
      // information to another agent "OtherAgent".  The OtherAgent
      // would then be responsible for updating its local Taems
      // view to reflect this information.

      // This Taems-finding code assumes MultiTaems is true
      Hashtable table = (Hashtable)state.getProperty(SimpleTaemsReader.CONTAEMS);
      Taems taems = (Taems)table.get(table.keys().nextElement());

      if (taems != null) {
          Task search = new Task("Put-Away-Dishes",
                                 new Agent((String)state.getProperty(State.NAME)),
                                 null);
          Task n = (Task)taems.findNode(search);
          if ((n != null) &&
              (n.getCurrentQuality() > 0) &&
              !n.hasAttribute("SentInformation") &&
              (communicate.findDefaultConnection() != null)) {
              // To be clever, we'll actually use the MASS ExecuteEnd message
              // format, so the data will be automatically interpreted
              // by OtherAgent.  If OtherAgent has started a "local" execution
              // of the nonlocal method "Put-Away-Dishes", then this should
              // cause it to automatically update its Taems structure.
              // See the comment after this code for another way of accomplishing this.

              // This tells it to activate the IR.  Normally the recieving agent
              // would do this itself by modifying its task structure, we use this
              // method for convenience.
              String content = "(ActivateInterrelationship Enables_1 0)";
              KQMLMessage message = new KQMLMessage("tell", content, "OtherAgent");
              communicate.sendMessage(message);

              // ...and this tells it to both complete the action which is (presumable)
              // running, and update the relevant task structure the action
              // is based on.
              content = "(" +
                  "ExecuteEnd " +
                  n.getLabel() + " " +
                  n.getCurrentDuration() + " " +
                  n.getCurrentCost() + " " +
                  n.getCurrentQuality() + " " +
                  ")";
              message = new KQMLMessage("tell", content, "OtherAgent");
              message.addField("in-reply-to", n.getLabel());
              communicate.sendMessage(message);

              // A flag so we only sent it once
              n.setAttribute("SentInformation", Boolean.TRUE);

              // Note that there is a less automated way to do this as well.  Instead
              // of relying on Execute to correctly parse and interpret the message,
              // you may also send the data in some other arbitrary format.  On the
              // recieving end, the agent must then parse this data, and use it to
              // update its local task structure.  This can be done like this:
              //
              // String methodname; // Get these values from your message
              // float q, c;
              // int d;
              // 
              // Taems taems = <your taems view>;
              // Method m = taems.findNode(new Method(methodname, null));
              // m.removeAllOutcomes();
              // m.addOutcome(new Outcome("Final_Results",
              //              new Distribution(new Float(q), new Float(1.0)),
              //              new Distribution(new Float(d), new Float(1.0)),
              //              new Distribution(new Float(c), new Float(1.0)),
              //              (float)1.0
              // ));
              // 
              // You may also use m.getOutInterrelationships() to get the IRs
              // originating at the node, and use setActive(true) to activate them.
              //
              // If you have an Action running corresponding to the method in
              // question, you will also need to call Execute.completeAction()
              // on it.  If this is the case, you can also just set the Q,C,D
              // values on the action, and Execute will automatically update
              // the corresponding method if the UpdateDist parameter is true.
          }
      }
  }
}
